#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Xyzzy::Eval;
use Data::Dumper;

BEGIN { use_ok('Xyzzy::Crypto') or BAIL_OUT('need Xyzzy::Crypto to run') }
BEGIN { use_ok('Xyzzy::Crypto::Config') or BAIL_OUT('need Xyzzy::Crypto::Config to run') }

my $cfg = new_ok('Xyzzy::Crypto::Config') or BAIL_OUT('need a Xyzzy::Crypto::Config object to run');

$cfg->set_cryptosecret('hunter2');

my $crypto = new_ok('Xyzzy::Crypto', [cfg => $cfg]) or BAIL_OUT('need a Xyzzy::Crypto object to run');

my $token = $crypto->create_token('test', 'foo', 'bar');

like($token, qr{^[a-zA-Z0-9_.]+$}, "manages to create a token");

foreach my $r ([], [''], ['', ''], ["zxnrbl"], ["zxnrbl", ''], ['', "zxnrbl"], ["\xEB"], ["\x{3042}"]) {
	local $Data::Dumper::Indent = 0;
	local $Data::Dumper::Terse = 1;
	my $repr = Dumper($r);
	my $token = $crypto->create_token('test', @$r);
	my ($time, $salt, @res) = $crypto->check_token('test', $token, 3);
	is_deeply(\@res, $r, "keys are passed in and out unscathed: $repr");
}

suicide(sub { $crypto->create_token('test', "x\0x") }, qr{Bad characters in request}, "rejects malformed data");

my ($time, $salt, @foobar) = $crypto->check_token('test', $token, 3);
my $foo = $crypto->check_token('test', $token, 3);

is($foobar[0], 'foo', "first data item survived");
is($foobar[1], 'bar', "second data item survived");
is($foo, 'foo', "first data item survived in scalar context");

suicide(sub { $crypto->check_token('test', 'zxnrbl', 3) }, qr{Short token}, "rejects short tokens");
suicide(sub { $crypto->check_token('test', 'sdhflshflsaiudhfalskjhfdlajshfdlajshfdlakjshfdlajhfd', 3) }, qr{Bad signature on token}, "rejects malformed tokens");
suicide(sub { $crypto->check_token('bad', $token, 3) }, qr{Bad signature on token}, "rejects tokens without correct secret data");
suicide(sub { $crypto->check_token('test', $token, -3) }, qr{Expired token}, "rejects expired tokens");

my $wallclock = 1234567890;

undef &Xyzzy::Crypto::time;
*Xyzzy::Crypto::time = sub { $wallclock };
undef &Xyzzy::Crypto::random_bytes;
*Xyzzy::Crypto::random_bytes = sub { my ($self, $n) = @_; return 'A' x $n };

($time, $salt, $token) = $crypto->create_token('test', 'foo', 'bar');

is($token, '5bkEPNWKKHdp_gfmh3DzihrpWHQTStMU6rzNzey3lMQABGLVPIjYgEFBQUFBQUFBQUFBQUFBQUFmb28AYmFy', "deterministic token has the expected value");
is($time, $wallclock * 1000000, "deterministic token has the expected timestamp");
like($salt, qr{^A+$}, "deterministic token has the expected salt");

my $othertoken = $crypto->create_token('test2', 'foo', 'bar');

isnt($othertoken, $token, "changing the hidden part changes the token");

$wallclock -= 10;

suicide(sub { $crypto->check_token('test', $token, 3) }, qr{Future token}, "rejects tokens with future timestamps");

$crypto->jitter(60);

survive(sub { $crypto->check_token('test', $token, 3) }, "accepts tokens with future timestamps as long as jitter is allowed");

done_testing();
