use utf8;
use strict;
use warnings FATAL => 'all';

$SIG{HUP} = sub {};

use Time::HiRes qw(time sleep setitimer clock_gettime CLOCK_PROCESS_CPUTIME_ID CLOCK_MONOTONIC);

my $wall_proc_start = clock_gettime(CLOCK_MONOTONIC)
	- clock_gettime(CLOCK_PROCESS_CPUTIME_ID);

use POSIX qw(setsid strftime);

BEGIN {
	use Sys::Syslog qw(openlog closelog syslog);

	my $logname = 'xyzzy';
	my $logcontext = '';
	my $proctitle = 'initializing';

	# This interface is a bit silly, if an additional variable
	# is added, a more reasonable interface needs to be thought up.
	# Perhaps something like:
	# log_set(name => ..., proctitle => ..., context => ..., context => ...);
	# Luckily it's just for internal use.

	sub log_set_proctitle {
		$proctitle = shift;
		my $pfx = $logcontext eq '' ? '' : " $logcontext";
		$0 = "$logname: $proctitle$pfx";
	}

	sub log_set_context {
		$logcontext = join(' ', map { "[$_]" } grep { defined } @_);
		log_set_proctitle($proctitle);
	}

	sub log_set_proctitle_and_context {
		$proctitle = shift;
		log_set_context(@_);
	}

	sub log_set_name {
		$logname = shift;
		openlog($logname, 'nofatal,pid', Sys::Syslog::LOG_DAEMON);
		log_set_proctitle($proctitle);
	}

	sub log_set_name_and_proctitle {
		$proctitle = shift;
		$logname = shift;
		log_set_proctitle($proctitle);
	}

	sub log_set_name_and_proctitle_and_context {
		$logname = shift;
		$proctitle = shift;
		log_set_context(@_);
	}

	sub log_warning {
		my $msg = join('', @_);
		my $pfx = $logcontext eq '' ? '' : "$logcontext ";
		my $level = Sys::Syslog::LOG_WARNING;
		my $nextlevel = ($msg =~ /^.* at (\S+) line (\d+)(?:, *<[^>]*> line \d+)?\.?\n/)
			? Sys::Syslog::LOG_DEBUG
			: $level;

		foreach my $line (split("\n", $msg)) {
			$line =~ s/\s+$//;
			next unless $line;
			$line =~ s/\t/    /g;
			$line = $pfx.$line;
			utf8::encode($line);
			syslog($level, '%s', $line);
			$level = $nextlevel;
		}
	}

	log_set_name($logname);
	$SIG{__WARN__} = \&log_warning;
}

my $file = $ARGV[0];
die "Usage: xyzzy <configfile>\n"
	unless $file;

use FCGI;
use IO::Handle;

use Xyzzy;
use Xyzzy::Header;
use Xyzzy::Status;
use Xyzzy::Request::Root;

# Built-in process manager
if(my $numprocs = $ENV{XYZZY_FCGI_CHILDREN} // $ENV{PHP_FCGI_CHILDREN}) {
	die "Number of processes ($numprocs) should be an integer number\n"
		unless $numprocs =~ /^[1-9]\d*$/;
	$numprocs = int($numprocs);

	my $manager = 1;
	my $queue = $numprocs;
	my $attempt;
	my $panic;
	my $backoff = 0;
	my %running;
	my $last = 0;
	my $now = time;
	my $debug = $ENV{XYZZY_FCGI_DEBUG};
	my $alarm;
	my $sigalarm;
	my $sigwinch;
	local our $alarmdie;

	log_set_context($file);

	setsid();

	sub startproc {
		my $pid = fork;
		if($pid) {
			my $p = {pid => $pid, start => $now};
			$running{$pid} = $p;
			$last = $now;
			$queue--;
			return $p;
		} elsif(defined $pid) {
			$wall_proc_start = clock_gettime(CLOCK_MONOTONIC);
			log_set_name_and_proctitle_and_context('Xyzzy', 'forking');
			$manager = 0;
			return undef;
		}
		die "fork: $!\n";
	}

	sub numeric_max {
		my ($a, $b) = @_;
		return $a > $b ? $a : $b;
	}

	eval {
		local $SIG{WINCH} = sub { $backoff = 0; kill(POSIX::SIGTERM, keys %running) };
		local $SIG{HUP} = sub { $backoff = 0; kill(POSIX::SIGHUP, keys %running) };
		local $SIG{TERM} =
		local $SIG{INT} = sub { my $sig = shift; die "SIG$sig received, exiting\n" };
		local $SIG{ALRM} = sub { setitimer(0, 0, 0); $sigalarm = 1; die "Timeout\n" if $alarmdie };

		while($manager) {
			die "Internal error" if $panic && !$attempt && !$queue;

			undef $alarm;
			undef $sigalarm;
			if($attempt) {
				$alarm = numeric_max($attempt->{start} + 5 - $now, 0);
				warn "Attempt is underway, setting alarm to $alarm seconds\n" if $debug;
				log_set_proctitle("trying to recover");
			} elsif($panic) {
				$alarm = numeric_max($panic + $backoff - $now, 0);
				warn "In panic mode, setting alarm to $alarm seconds (backoff = $backoff)\n" if $debug;
				log_set_proctitle("waiting before trying again to recover");
			} elsif($queue) {
				my $numrunning = $numprocs - $queue;
				log_set_proctitle("$numrunning/$numprocs processes");
				$alarm = numeric_max($last + 1 - $now, 0);
			} else {
				log_set_proctitle("$numprocs processes");
			}

			warn "count=".scalar(values %running)." queue=$queue\n" if $debug;

			my $pid = -1;
			if(defined $alarm && $alarm == 0) {
				$sigalarm = 1;
			} else {
				eval {
					local $alarmdie = 1;
					setitimer(0, $alarm, 0.1) if $alarm;
					if(%running) {
						$pid = waitpid(-1, 0);
					} elsif($alarm) {
						sleep;
					} else {
						sleep .1;
					}
				};
				my $e = $@;
				setitimer(0, 0, 0) if $alarm;
				if($e && !$sigalarm) {
					delete $running{$pid} if $pid != -1;
					die $e;
				}
			}
			$now = time;

			if($sigalarm) {
				if($attempt) {
					if($attempt->{start} >= $panic && $attempt->{start} + 5 < $now) {
						warn "Panic mode canceled\n" if $debug;
						undef $attempt;
						undef $panic;
						$backoff = 0;
					}
				} elsif($panic) {
					if($queue && $now > $panic + $backoff) {
						warn "Starting probation process\n" if $debug;
						return unless $attempt = startproc();
					} else {
						warn "Not starting probation process\n" if $debug;
					}
				} elsif($queue) {
					if($last + 1 < $now) {
						return unless startproc();
					}
				} else {
					warn "No panic?\n" if $debug;
				}
			}

			if($pid == -1) {
				warn "No processes?\n" if $debug;
			} elsif(!$running{$pid}) {
				warn "Unknown process $pid?\n" if $debug;
			} elsif($running{$pid}{start} > $now - 5) {
				warn "Process $pid exited too fast!\n" if $debug;
				if($panic) {
					if($attempt && $attempt->{pid} == $pid) {
						$backoff++;
						undef $attempt;
						warn "Probation process $pid failed\n" if $debug;
					}
					$panic = numeric_max($panic, $running{$pid}{start});
				} else {
					$backoff = 1;
					$panic = $running{$pid}{start};
					warn "Panic mode activated\n" if $debug;
				}
				delete $running{$pid};
				$queue++;
			} else {
				warn "Process exited normally, queueing\n" if $debug;
				delete $running{$pid};
				$queue++;
			}
		}
	};
	if($@) {
		warn $@;
		if($manager) {
			log_set_proctitle("exiting");
			if(%running) {
				warn "Sending processes the TERM signal\n" if $debug;
				foreach my $pid (keys %running) {
					delete $running{$pid}
						if kill(POSIX::SIGTERM, $pid) != 1;
				}
				local $SIG{ALRM} = sub { return };
				eval {
					local $SIG{ALRM} = sub { setitimer(0, 0, 0); die "Timeout\n" };
					setitimer(0, 2, 0.1);
					while(%running) {
						my $pid = waitpid(-1, 0);
						last if $pid == -1;
						delete $running{$pid};
					}
				};
				setitimer(0, 0, 0);
				if(%running) {
					warn "Sending processes the KILL signal\n" if $debug;
					kill(POSIX::SIGKILL, keys %running);
				}
			}
		}
		exit 2;
	}
	exit 0 if $manager;
}

sub concat {
	return undef if grep !defined, @_;
	return join('/', @_);
}

my $benchmark;
my $numreqs;
my $handler;
my $hupped;
my $termed;

sub load_config {
	my $cfg = new Xyzzy;
	$cfg->include($file);
	$benchmark = $cfg->benchmark;
	$handler = $cfg->handler;
	log_set_name(ref $cfg);
}

sub reload_config {
	undef $hupped;
	eval { load_config() };
	if(my $e = $@) {
		warn "While reloading configuration from $file:\n";
		warn $e;
		warn "Proceeding using previous configuration\n";
		log_set_proctitle("reload failed");
	} else {
		warn "Configuration reloaded\n";
		log_set_proctitle("reloaded");
	}
}

sub sighup {
	$hupped = 1;
}

sub sigterm {
	die "SIGTERM received, exiting\n" if our $termok;
	$termed = 1;
}

eval {
	load_config();

	$SIG{HUP} = \&sighup;
	$SIG{TERM} = \&sigterm;

	log_set_proctitle_and_context("new");

	my $fcgi = FCGI::Request(new IO::Handle, new IO::Handle, new IO::Handle, {}, 0, FCGI::FAIL_ACCEPT_ON_INTR);

	if($benchmark) {
		my $cpu_proc_init = clock_gettime(CLOCK_PROCESS_CPUTIME_ID);
		my $wall_proc_init = clock_gettime(CLOCK_MONOTONIC) - $wall_proc_start;

		warn sprintf("initial: wall=\%.3fs cpu=\%.3fs (\%.2f\%\%)\n", $wall_proc_init, $cpu_proc_init, 100.0 * $cpu_proc_init / $wall_proc_init);
	}

	for(;;) {
		$SIG{__WARN__} = \&log_warning;
		$SIG{HUP} = \&sighup unless $SIG{HUP} == \&sighup;
		$SIG{TERM} = \&sigterm unless $SIG{TERM} == \&sigterm;
		$! = - do {
			# try to prevent signal race conditions
			local our $termok = 1;
			die "SIGTERM received, exiting\n" if $termed;
			reload_config() while $hupped;
			$fcgi->Accept;
		};
		if($!) {
			next if $!{EINTR};
			die "FCGI: accept(): $!\n";
		}
		$numreqs++;
		log_set_proctitle("serving #$numreqs");

		my $wall_req_start = clock_gettime(CLOCK_MONOTONIC);
		my $cpu_req_start = clock_gettime(CLOCK_PROCESS_CPUTIME_ID);

		my ($in, $out, $err, $env);
		if($fcgi->IsFastCGI) {
			($in, $out, $err) = $fcgi->GetHandles;
			$env = $fcgi->GetEnvironment;
		} else {
			($in, $out, $err) = (*STDIN{IO}, *STDOUT{IO}, *STDERR{IO});
			*STDIN = new IO::File('/dev/null', '<');
			*STDOUT =
			*STDERR = new IO::File('/dev/null', '>');
			$env = {%ENV};
			%ENV = (PATH => delete $env->{PATH} // '/usr/local/bin:/usr/bin:/bin');
		}

		my $output = eval {
			my $ctx = new Xyzzy::Request::Root(in => $in, out => $out, err => $err, env => $env, cfg => $handler);
			log_set_context($ctx->remote_addr);
			my $res = eval { $handler->handle($ctx) };
			if($res) {
				die "not a Xyzzy::Response object\n"
					unless UNIVERSAL::isa($res, 'Xyzzy::Response');
			} else {
				die "no response from handler\n" unless $@;
				die $@ unless UNIVERSAL::isa($@, 'Xyzzy::Response');
				$res = $@;
				undef $@;
			}

			$res->as_cgi;
		};
		if($@) {
			warn $@;
			my $status = new Xyzzy::Status(500);
			$output = $status->response->as_cgi;
		}

		log_set_proctitle("output #$numreqs");

		foreach(@$output) { $out->write($_) }

		$fcgi->Flush;
		$fcgi->Finish;

		if($benchmark) {
			my $cpu_req_total = clock_gettime(CLOCK_PROCESS_CPUTIME_ID) - $cpu_req_start;
			my $wall_req_total = clock_gettime(CLOCK_MONOTONIC) - $wall_req_start;

			warn sprintf("request: wall=\%.3fs cpu=\%.3fs (\%.2f\%\%)\n", $wall_req_total, $cpu_req_total, 100.0 * $cpu_req_total / $wall_req_total);
		}

		log_set_proctitle_and_context("idle #$numreqs");
	}
};

my $err = $@;

if($benchmark) {
	my $cpu_proc_total = clock_gettime(CLOCK_PROCESS_CPUTIME_ID);
	my $wall_proc_total = clock_gettime(CLOCK_MONOTONIC) - $wall_proc_start;

	warn sprintf("process: wall=\%.3fs cpu=\%.3fs (\%.2f\%\%)\n", $wall_proc_total, $cpu_proc_total, 100.0 * $cpu_proc_total / $wall_proc_total);
}

if($err) {
	warn $err;
	exit(1);
}
