<?php

/**
 * Copyright 2013 François Kooman <fkooman@tuxed.net>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

require_once 'vendor/autoload.php';

use fkooman\Config\Config;

class ConfigTest extends PHPUnit_Framework_TestCase
{

    public function testInvoice()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $this->assertEquals("34843", $c->getValue("invoice", true));
    }

    public function testSection()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $d = $c->getSection("bill-to");
        $this->assertEquals("Chris", $d->getValue("given", true));
    }

    public function testIniFile()
    {
        $c = Config::fromIniFile("tests/data/1.ini");
        $this->assertEquals("bar", $c->getValue("foo", true));
    }

    public function testJsonFile()
    {
        $c = Config::fromJsonFile("tests/data/sample.json");
        $this->assertEquals("bar", $c->getValue("foo", true));
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage unable to read configuration file 'missing.txt'
     */
    public function testMissingYamlFile()
    {
        $c = Config::fromYamlFile("missing.txt");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage unable to read configuration file 'missing.txt'
     */
    public function testMissingIniFile()
    {
        $c = Config::fromIniFile("missing.txt");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage unable to read configuration file 'missing.txt'
     */
    public function testMissingJsonFile()
    {
        $c = Config::fromJsonFile("missing.txt");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage unable to parse configuration file 'tests/data/raw.dat'
     */
    public function testBrokenYamlFile()
    {
        $c = Config::fromYamlFile("tests/data/raw.dat");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage unable to parse configuration file 'tests/data/raw.dat'
     */
    public function testBrokenIniFile()
    {
        $c = Config::fromIniFile("tests/data/raw.dat");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage unable to parse configuration file 'tests/data/raw.dat'
     */
    public function testBrokenJsonFile()
    {
        $c = Config::fromJsonFile("tests/data/raw.dat");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage subtree 'foo' does not exist
     */
    public function testNonExistingRequiredSection()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $d = $c->getSection("foo", true);
    }

    public function testNonExistingNotRequiredSection()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $this->assertEquals(null, $c->getSection("foo")->getValue("foo"));
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage 'invoice' is not a subtree
     */
    public function testNotASection()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $d = $c->getSection("invoice");
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage required leaf 'foo' does not exist
     */
    public function testNonExistingRequiredKey()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $c->getValue("foo", true);
    }

    public function testNonExistingNotRequiredKeyDefault()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $this->assertEquals(123, $c->getValue("foo", false, 123));
    }

    /**
     * @expectedException \fkooman\Config\ConfigException
     * @expectedExceptionMessage 'bill-to' is a subtree
     */
    public function testGetAValueAsSection()
    {
        $c = Config::fromYamlFile("tests/data/invoice.yaml");
        $c->getValue("bill-to");
    }

    public function testToArray()
    {
        $c = Config::fromIniFile("tests/data/1.ini");
        $this->assertEquals(array("one", "two", "three"), $c->getSection("two")->getSection("list")->toArray());
        $this->assertEquals(array("one", "two", "three"), $c->s("two")->s("list")->toArray());
        $this->assertEquals('foo', $c->s("two")->l("bar"));
        $this->assertEquals('foo', $c->getSubtree("two")->getLeaf("bar"));
    }

}
