# $Id: Squarepeg.pm 39309 2013-04-26 12:41:26Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/squarepeg/lib/UvT/Squarepeg.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Squarepeg::Resources;

use UvT::Squarepeg::LDAP;
use UvT::Squarepeg::Quota;

use UvT::Squarepeg::Handler -self;

field ldap => sub { new UvT::Squarepeg::LDAP(cfg => shift->cfg) };
field quota => sub { new UvT::Squarepeg::Quota(cfg => shift->cfg) };

package UvT::Squarepeg;

use Xyzzy::CAS;
use Xyzzy::Database;
use Xyzzy::Directory;
use Xyzzy::Method;
use Xyzzy::ReST;

use UvT::Squarepeg::ACL;
use UvT::Squarepeg::Ajax;
use UvT::Squarepeg::Box;
use UvT::Squarepeg::Boxes;
use UvT::Squarepeg::Portal;
use UvT::Squarepeg::Root;
use UvT::Squarepeg::Static;
use UvT::Squarepeg::User;
use UvT::Squarepeg::Users;

use Xyzzy -self;

use Xyzzy::CAS::Config -mixin;
use Xyzzy::Database::Config -mixin;
use UvT::Squarepeg::Config -mixin;
use UvT::Squarepeg::LDAP::Config -mixin;
use UvT::Squarepeg::Quota::Config -mixin;

sub handler {
	my $rsc = new UvT::Squarepeg::Resources(cfg => $self);

	my $link = new Xyzzy::Method(cfg => $rsc, methods => {
		GET => new UvT::Squarepeg::Ajax::User::Box::Get(cfg => $rsc),
		PUT => new UvT::Squarepeg::Ajax::User::Box::Modify(cfg => $rsc),
		DELETE => new UvT::Squarepeg::Ajax::User::Box::Delete(cfg => $rsc),
	});

	# generic:  /users
	# specific: /users/bob

	my $rest = new Xyzzy::ReST(cfg => $rsc,
		generic => {
			users => new Xyzzy::Method(cfg => $rsc, methods => { # /users
				GET => new UvT::Squarepeg::Ajax::Users::List(cfg => $rsc),
				POST => new UvT::Squarepeg::Ajax::Users::Create(cfg => $rsc),
			}),
			boxes => new Xyzzy::Method(cfg => $rsc, methods => { # /boxes
				GET => new UvT::Squarepeg::Ajax::Boxes::List(cfg => $rsc),
				POST => new UvT::Squarepeg::Ajax::Boxes::Create(cfg => $rsc),
			})
		},
		specific => {
			users => new Xyzzy::ReST(cfg => $rsc,
				handler => new Xyzzy::Method(cfg => $rsc, methods => { # /users/bob
					GET => new UvT::Squarepeg::Ajax::User::Get(cfg => $rsc),
					PUT => new UvT::Squarepeg::Ajax::User::Modify(cfg => $rsc),
					DELETE => new UvT::Squarepeg::Ajax::User::Delete(cfg => $rsc),
				}),
				generic => {
					boxes => new Xyzzy::Method(cfg => $rsc, methods => { # /users/bob/boxes
						GET => new UvT::Squarepeg::Ajax::User::Boxes::List(cfg => $rsc),
						POST => new UvT::Squarepeg::Ajax::User::Boxes::Create(cfg => $rsc),
					}),
				},
				specific => { boxes => $link } # /users/bob/boxes/sales
			),
			boxes => new Xyzzy::ReST(cfg => $rsc,
				handler => new Xyzzy::Method(cfg => $rsc, methods => { # /boxes/sales
					GET => new UvT::Squarepeg::Ajax::Box::Get(cfg => $rsc),
					PUT => new UvT::Squarepeg::Ajax::Box::Modify(cfg => $rsc),
					DELETE => new UvT::Squarepeg::Ajax::Box::Delete(cfg => $rsc),
				}),
				generic => {
					users => new Xyzzy::Method(cfg => $rsc, methods => { # /boxes/sales/users
						GET => new UvT::Squarepeg::Ajax::Box::Users::List(cfg => $rsc),
						POST => new UvT::Squarepeg::Ajax::Box::Users::Create(cfg => $rsc),
					}),
				},
				specific => { users => $link } # /boxes/sales/users/bob
			)
		},
	);

	my $ajax = new UvT::Squarepeg::Ajax(cfg => $rsc,
		subdirs => {
			nonce => new UvT::Squarepeg::Ajax::Nonce(cfg => $rsc),
			rest => $rest,
		}
	);

	my $handler = new Xyzzy::Directory(cfg => $rsc,
		subdirs => {
			user => new UvT::Squarepeg::User(cfg => $rsc),
			users => new UvT::Squarepeg::Users(cfg => $rsc),
			box => new UvT::Squarepeg::Box(cfg => $rsc),
			boxes => new UvT::Squarepeg::Boxes(cfg => $rsc),
			ajax => $ajax,
		},
		fallback => new UvT::Squarepeg::Static(cfg => $rsc),
		handler => new UvT::Squarepeg::Portal(cfg => $rsc)
	);

	# wrap the handler in successively more generic decorators
	$handler = new UvT::Squarepeg::ACL(cfg => $rsc, handler => $handler);
	$handler = new Xyzzy::CAS(cfg => $rsc, handler => $handler);
	$handler = new Xyzzy::Database(cfg => $rsc, handler => $handler);
	$handler = new UvT::Squarepeg::Root(cfg => $rsc, handler => $handler);

	return $handler;
}
