'''
Copyright 2014 Cisco Systems, Inc.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
'''
import base64
from UcsSdk import *


from UcsNagiosFileHandler import UcsNagiosFileHandler
from base64 import b64decode


def encode(key, clear):
    enc = []
    for i in range(len(clear)):
        key_c = key[i % len(key)]
        enc_c = chr((ord(clear[i]) + ord(key_c)) % 256)
        enc.append(enc_c)
    return base64.urlsafe_b64encode("".join(enc))

def decode(key, enc):
    dec = []
    enc = base64.urlsafe_b64decode(enc)
    for i in range(len(enc)):
        key_c = key[i % len(key)]
        dec_c = chr((256 + ord(enc[i]) - ord(key_c)) % 256)
        dec.append(dec_c)
    return "".join(dec)

def word_to_upper(word):
    '''
    Method used for converting class and attributes passed
    by end user to the format expected by pythonSDK
    :param word: A string containing class or attribute
    '''
    return (word[0].upper() + word[1:])

## Class for handling and discovering the UCS server and creating a dictionary
## format for further processing


class UcsNagiosDiscovery(UcsNagiosFileHandler):
    '''
    Nagios auto discovery class for discovering
    Cisco UCS and fetching the required inventory
    information
    '''

    def __init__(self,
                 ucsmip,
                 user,
                 password,
                 plugin_cfg_dict,
                 no_ssl=False,
                 port=None,
                 proxy=None,
                 debug=False):
        '''
        Constructor
        '''
        # Class variables
        self.ucs_ip = ucsmip
        self.user = user
        self.password = password
        self.port = port
        self.no_ssl = no_ssl
        self.proxy = proxy
        self.handle = None
        self.debug = debug
        self.plugin_cfg_dict = plugin_cfg_dict
        self.ucs_dict = dict()
        UcsNagiosFileHandler.__init__(self,
                                      self.get_ucs_handle(),
                                      self.ucs_dict)

    def get_ucs_handle(self):
        if self.handle is None:
            self.handle = UcsHandle()
            if self.debug :
                self.handle.SetDumpXml()
            self.handle.Login(name=self.ucs_ip,
                                username=self.user,
                                password=b64decode(self.password),
                                noSsl=self.no_ssl,
                                port=self.port,
                                proxy=self.proxy,
                                autoRefresh=False)

        return self.handle

    def close_ucs_handle(self):
        try:
            print "Closing Connection with : " + self.handle._name
            self.handle.Logout()
        except Exception, err:
            self.handle.Logout()
            raise err

    def discover_ucs(self,host_creation_flag):
        handle = None
        try:
            handle = self.get_ucs_handle()
            class_id_list = self.plugin_cfg_dict['DISCOVERY_CLASS_LIST'].split(",")
            idset = ClassIdSet()
            for class_id in class_id_list:
                # Changes done to support UCS Mini as the IOM class are 
                # different in UCS and UCS Mini.Initially we used a single call for fetching the 
                # class data but with the new class (equipmentSwitchIOCard) in place 
                # we are ignoring the failure of this class when dealing with UCS Mini 
#                 cid = ClassId()
#                 cid.setattr("Value", UcsUtils.WordL(class_id))
#                 idset.AddChild(cid)
                class_id = class_id.strip()
                get_response = handle.ConfigResolveClass(class_id,None)
                if get_response.OutConfigs:
                    for mo in get_response.OutConfigs.GetChild():
                        self.get_mo_details(mo,host_creation_flag,class_id)
                else:
                    error_code=get_response.errorCode
                    error_descr=get_response.errorDescr
                    error_text=str(error_code)+" : "+str(error_descr)
                    #raise Exception(error_text)
                    if self.debug :
                        print "Error while trying to discover class ", class_id
                        print error_text  
        except Exception, err:
            handle.Logout()
            raise Exception(str(err))

    def get_mo_details(self, mo,host_creation_flag,class_id):
        class_id_updated=word_to_upper(mo.classId)
        # FI
        if (class_id_updated == "NetworkElement"):
            key = mo.OobIfIp + "_fi_" + mo.Id
            if host_creation_flag:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
            else:
                self.ucs_dict[key, 'host_name'] = key
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip, key])
            self.ucs_dict[key, 'alias'] = "_fi_" + mo.Id
            self.ucs_dict[key, 'host_ip'] = self.ucs_ip
            self.ucs_dict[key, 'Dn'] = mo.Dn
            self.ucs_dict[key, 'parent'] = self.ucs_ip
            self.ucs_dict[key, 'Operability'] = mo.Operability
            self.ucs_dict[key, 'class_id'] = class_id

        # Blade
        elif (class_id_updated == "ComputeBlade"):
            only_active = ""
            try:
                only_active = self.plugin_cfg_dict['ASSOCIATED_BLADES_ONLY'].strip()
            except Exception:
                only_active = "False"
            try: 
                sp_name = mo.AssignedToDn.split("ls-")[1]
            except IndexError:
                sp_name = ""
            
            if only_active.lower() != "true":
                if sp_name.strip() == "":
                    key = self.ucs_ip + "_blade_" + mo.ChassisId + "_" + mo.SlotId
                    alias = "_blade_" + mo.ChassisId + "_" + mo.SlotId
                else:
                    key = self.ucs_ip + "_blade_" + mo.ChassisId + "_" + mo.SlotId + "_" + sp_name
                    alias = "_blade_" + mo.ChassisId + "_" + mo.SlotId + "_" + sp_name
            elif only_active.lower() == "true" and mo.Association == "associated":
                key = self.ucs_ip + "_blade_" + mo.ChassisId + "_" + mo.SlotId + "_" + sp_name
                alias = "_blade_" + mo.ChassisId + "_" + mo.SlotId + "_" + sp_name
            else:
                return
                
            self.ucs_dict[key, 'alias'] = alias
            if host_creation_flag:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
            else:
                self.ucs_dict[key, 'host_name'] = key
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip + '_chassis_' + mo.ChassisId, key])
            self.ucs_dict[key, 'host_ip'] = self.ucs_ip
            self.ucs_dict[key, 'Dn'] = mo.Dn
            self.ucs_dict[key, 'parent'] = self.ucs_ip + '_chassis_' + mo.ChassisId
            self.ucs_dict[key, 'Operability'] = mo.Operability
            self.ucs_dict[key, 'class_id'] = class_id

        elif (class_id_updated == "EquipmentIOCard"):
            # As we don't want the FEX IOM details so we will skip them
            if not re.search('fex', mo.Dn):
                key = self.ucs_ip + "_iom_" + mo.ChassisId + "_" + mo.Id
                self.ucs_dict[key, 'alias'] = "_iom_" + mo.ChassisId + "_" + mo.Id
                if host_creation_flag:
                    self.ucs_dict[key, 'host_name'] = self.ucs_ip
                    self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
                else:
                    self.ucs_dict[key, 'host_name'] = self.ucs_ip + "_iom_" + mo.ChassisId + "_" + mo.Id
                    self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip + '_chassis_' + mo.ChassisId, self.ucs_ip + "_iom_" + mo.ChassisId + "_" + mo.Id])
                self.ucs_dict[key, 'host_ip'] = self.ucs_ip
                self.ucs_dict[key, 'Dn'] = mo.Dn
                self.ucs_dict[key, 'Operability'] = mo.Operability
                self.ucs_dict[key, 'class_id'] = class_id
                self.ucs_dict[key, 'parent'] = self.ucs_ip + '_chassis_' + mo.ChassisId
        elif (class_id_updated == "EquipmentSwitchIOCard"):
            # As we don't want the FEX IOM details so we will skip them
            if not re.search('fex', mo.Dn):
                key = self.ucs_ip + "_iom_" + mo.ChassisId + "_" + mo.Id
                self.ucs_dict[key, 'alias'] = "_iom_" + mo.ChassisId + "_" + mo.Id
                if host_creation_flag:
                    self.ucs_dict[key, 'host_name'] = self.ucs_ip
                    self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
                else:
                    self.ucs_dict[key, 'host_name'] = self.ucs_ip + "_iom_" + mo.ChassisId + "_" + mo.Id
                    self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip + '_chassis_' + mo.ChassisId, self.ucs_ip + "_iom_" + mo.ChassisId + "_" + mo.Id])
                self.ucs_dict[key, 'host_ip'] = self.ucs_ip
                self.ucs_dict[key, 'Dn'] = mo.Dn
                self.ucs_dict[key, 'Operability'] = mo.Operability
                self.ucs_dict[key, 'class_id'] = class_id
                self.ucs_dict[key, 'parent'] = self.ucs_ip + '_chassis_' + mo.ChassisId

        # Chassis
        elif (class_id_updated == "EquipmentChassis"):
            key = self.ucs_ip + "_chassis_" + mo.Id
            self.ucs_dict[key, 'alias'] = "_chassis_" + mo.Id
            if host_creation_flag:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
            else:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip + "_chassis_" + mo.Id
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip, self.ucs_ip + "_chassis_" + mo.Id])
            self.ucs_dict[key, 'host_ip'] = self.ucs_ip
            self.ucs_dict[key, 'Dn'] = mo.Dn
            self.ucs_dict[key, 'parent'] = self.ucs_ip
            self.ucs_dict[key, 'Operability'] = mo.Operability
            self.ucs_dict[key, 'class_id'] = class_id

        # FEX
        elif (class_id_updated == "EquipmentFex"):
            key = self.ucs_ip + "_fex_" + mo.Id
            self.ucs_dict[key, 'alias'] = "_fex_" + mo.Id
            if host_creation_flag:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
            else:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip + "_fex_" + mo.Id
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip, self.ucs_ip + "_fex_" + mo.Id])
            self.ucs_dict[key, 'host_ip'] = self.ucs_ip
            self.ucs_dict[key, 'Dn'] = mo.Dn
            self.ucs_dict[key, 'Operability'] = mo.Operability
            self.ucs_dict[key, 'class_id'] = class_id
            self.ucs_dict[key, 'parent'] = self.ucs_ip

        # Rack-Unit
        elif (class_id_updated == 'ComputeRackUnit'):
            only_active = ""
            try:
                only_active = self.plugin_cfg_dict['ASSOCIATED_BLADES_ONLY'].strip()
            except Exception:
                only_active = "False"
            try:
                sp_name = mo.AssignedToDn.split("ls-")[1]
            except:
                sp_name = ""

            if only_active.lower() != "true":
                if sp_name.strip() == "":
                    key = self.ucs_ip + "_rack_" + mo.Id
                    alias = "_rack_" + mo.Id
                else:
                    key = self.ucs_ip + "_rack_" + mo.Id + "_" + sp_name
                    alias = "_rack_" + mo.Id + "_" + sp_name
            elif only_active.lower() == "true" and mo.Association == "associated":
                key = self.ucs_ip + "_rack_" + mo.Id + "_" + sp_name
                alias = "_rack_" + mo.Id + "_" + sp_name
            else:
                return
            self.ucs_dict[key, 'alias'] = alias
            if host_creation_flag:
                self.ucs_dict[key, 'host_name'] = self.ucs_ip
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
            else:
                self.ucs_dict[key, 'host_name'] = key
                self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip, key])
            self.ucs_dict[key, 'host_ip'] = self.ucs_ip
            self.ucs_dict[key, 'Dn'] = mo.Dn
            self.ucs_dict[key, 'Operability'] = mo.Operability
            self.ucs_dict[key, 'class_id'] = class_id
            self.ucs_dict[key, 'parent'] = self.ucs_ip
        else:
            if mo.Dn:
                key = self.ucs_ip + "_" + "_".join((mo.Dn).split("/"))
                self.ucs_dict[key, 'alias'] = "_" + "_".join((mo.Dn).split("/"))
                if host_creation_flag:
                    self.ucs_dict[key, 'host_name'] = self.ucs_ip
                    self.ucs_dict[self.ucs_ip, 'host_group'].append([self.ucs_ip,self.ucs_ip])
                else:
                    self.ucs_dict[key, 'host_name'] = key 
                    self.ucs_dict[self.ucs_ip, 'host_group'].append(["",self.ucs_ip])
                self.ucs_dict[key, 'host_ip'] = self.ucs_ip
                self.ucs_dict[key, 'Dn'] = mo.Dn
                self.ucs_dict[key, 'Operability'] = "dummy"
                self.ucs_dict[key, 'class_id'] = class_id
                self.ucs_dict[key, 'parent'] = self.ucs_ip
