'''
Copyright 2014 Cisco Systems, Inc.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
'''

import csv
import operator

# Wrapper class for ignoring comments and empty lines
# in the configuration files
class CheckForCommentOrBlank:
    def __init__(self, fp):
        self.fp = fp

    def __iter__(self):
        return self

    def next(self):
        line = self.fp.next()
        if not line.strip() or line.startswith('#'):
            return self.next()
        return line

    def close(self):
        self.fp.close()


## Class for handling the property files

class PropertyHandlerClass(object):

    _instance = None
    def __new__(cls, *args, **kwargs):
        if not cls._instance:
            cls._instance = super(PropertyHandlerClass, cls).__new__(
                            cls, *args, **kwargs)
        return cls._instance


# Use thread lock so as to be on a safer side
    def read_properties(self, filename):
        """ Reads a given properties file with each line of the format key=value.  Returns a dictionary containing the pairs.
        Keyword arguments:
            filename -- the name of the file to be read
        """
        result = dict()
        csvfile = CheckForCommentOrBlank(open(filename, 'rb')) 
        #with open(filename, "rb") as csvfile:
        reader = csv.reader(csvfile, delimiter='=', escapechar='\\', quoting=csv.QUOTE_NONE)
        for row in reader:
            if len(row) != 2:
                raise Exception('Check if configuration properties are in the format PROPERTY_NAME = VALUE.\n'
                'Also check the VALUE should be in a single line.')
            result[row[0].strip()] = row[1].strip()
        csvfile.close()
        return result

    def write_properties(self, filename, dictionary):
        """ Writes the provided dictionary in key-sorted order to a properties file with each line of the format key=value
        Keyword arguments:
            filename -- the name of the file to be written
            dictionary -- a dictionary containing the key/value pairs.
        """
        csvfile = open(filename, 'wb')
        #with open(filename, "wb") as csvfile:
        writer = csv.writer(csvfile, delimiter='=', escapechar='\\', quoting=csv.QUOTE_NONE)
        for key, value in sorted(dictionary.items(), key=operator.itemgetter(0)):
            writer.writerow([ key.strip(), value.strip()])

