<?php
/**
 * @file
 * The plugin query object for MediaMosa allows REST calls inside views. It
 * uses the Views 3 plugin option to query another source instead of the normal
 * Drupal database.
 */

abstract class mediamosa_ck_views_rest extends views_plugin_query {
  // ---------------------------------------------------------------- Variables.
  public $requests;
  public $aliases;
  public $params;
  public $orderby;
  public $required_fields;

  /**
   * Execute the REST call.
   *
   * @param object $view
   *   The view object.
   * @param array $params
   *   Array of view options and settings.
   * @param array $options
   *   Options that can be used for overrides on the view params and REST call
   *   related options.
   */
  abstract protected function do_rest_call($view, array $params, array $options = array());

  /**
   * Process the XML and fill the view data.
   *
   * @param mediamosa_connector_response $response
   *   The response object.
   * @param object $view
   *   The view object.
   * @param array $params
   *   Array of view options and settings.
   * @param array $options
   *   Options that can be used for overrides on the view params and REST call
   *   related options.
   */
  abstract protected function process_rest_call(mediamosa_connector_response $response, $view, array $params, array $options = array());

  // ---------------------------------------------------------------- Functions.
  /**
   * Builds the necessary info to execute the query.
   *
   * @param object $view
   *   The view object.
   */
  public function build(&$view) {
    // Setup the view object.
    $view->init_pager();

    // Let the pager modify the query to add limits.
    $this->pager->query();

    // Set aliases of the fields.
    foreach ($view->field as $field_name => &$field) {
      $field->field_alias = $field_name;
      $field->aliases['entity_type'] = 'entity_type';
    }

    // Add fields to the query so they will be shown in solr document.
    $this->params['fl'] = array_keys($view->field);
  }

  /**
   * Add a field to the query table, possibly with an alias.
   *
   * This will automatically call ensure_table to make sure the required table
   * exists, *unless* $table is unset.
   *
   * @param string $table
   *   The table this field is attached to, here it is the ID of the REST view
   *   class.
   * @param string $field
   *   The name of the field to add.
   * @param boolean $required
   *   TRUE or FALSE, will add the field to required listing as input.
   *
   * @return string
   *   The name that this field can be referred to as. Usually this is the
   *   alias.
   */
  public function add_field($table, $field, $required = FALSE) {
    $alias = "{$table}_{$field}";
    $this->aliases[$table][$field] = $alias;

    if ($required) {
      $this->required_fields[$alias] = $alias;
    }

    return $alias;
  }

  /**
   * Is called when sort option is added/used.
   */
  public function add_orderby($table, $field = NULL, $order = 'ASC', $alias = '', $params = array()) {
    $this->orderby[$table . '.' . $field] = array(
      'orderby' => $field,
      'order' => $order,
      'params' => $params);
  }

  /**
   * Provide a form to edit options for this plugin.
   */
  public function options_form(&$form, &$form_state) {
    // Call parent.
    parent::options_form($form, $form_state);

    // Checkbox to allow show only public marked assets.
    $form['mediamosa_ck_view_show_drupal_error_messages'] = array(
      '#title' => t('Show Drupal error messages.'),
      '#description' => t('Enable this switch to show errors as Drupal error messages.'),
      '#type' => 'checkbox',
      '#default_value' => $this->mediamosa_option_show_drupal_error_messages(),
    );
  }

  /**
   * Returns the value for mediamosa_is_public_list option.
   *
   * @return boolean
   *   The value of mediamosa_is_public_list.
   */
  public function mediamosa_option_show_drupal_error_messages() {

    if (isset($this->options['mediamosa_show_drupal_error_messages'])) {
      return $this->options['mediamosa_show_drupal_error_messages'];
    }

    // By defaut do not show the drupal error.
    return FALSE;
  }

  /**
   * Executes the query.
   *
   * And fills the associated view object with according values.
   * Values to set:
   * $view->result,
   * $view->total_rows,
   * $view->execute_time,
   * $view->pager['current_page'].
   */
  public function execute(&$view) {
    try {
      $start = microtime(TRUE);

      // Setup the params.
      $params = array();
      $params['rows'] = $this->pager->options['items_per_page'];
      $params['start'] = $this->pager->current_page * $this->pager->options['items_per_page'];

      // Params to copy.
      $param_fields = array(
        'q' => 'q',
        'fl' => 'fl',
        'filters' => 'filters',
        'sort' => 'sort',
      );

      // Copy needed fields.
      foreach ($param_fields as $param_field) {
        if (!empty($this->params[$param_field])) {
          $params[$param_field] = $this->params[$param_field];
        }
      }

      // Perform request to MediaMosa.
      $response = $this->do_rest_call($view, $params, $this->options);

      // Now process the xml into the view data.
      $view->total_rows = 0;
      if ($response && empty($this->errors)) {
        $this->process_rest_call($response, $view, $params, $this->options);
      }

      // Store the results.
      $this->pager->total_items = $view->total_rows;
      $this->pager->update_page_info();

      // We shouldn't use $results['performance']['complete'] here, since
      // extracting the results probably takes considerable time as well.
      $view->execute_time = microtime(TRUE) - $start;
    }
    catch (Exception $e) {
      $this->errors[] = $e->getMessage();
    }

    if (!empty($this->errors)) {
      foreach ($this->errors as $message) {
        mediamosa_ck::watchdog('Caught error during REST view: @message', array('@message' => $message), WATCHDOG_ERROR);
        if ($this->mediamosa_option_show_drupal_error_messages()) {
          drupal_set_message($message, 'error');
        }
      }
      $view->result = array();
      $view->total_rows = 0;
      $view->execute_time = 0;
    }
  }

  /**
   * Add filter.
   */
  public function add_filter($type, $value, $exclude = FALSE) {
    $this->params['filters'][$type] = ($exclude ? '-' : '') . $value;
  }

  /**
   * Add string filter.
   */
  public function add_filter_string($string) {
    $this->params['q.alt'][] = $string;
  }

  /**
   * Add sort.
   */
  public function add_sort($field, $order) {
    $this->params['sort'][] = $field . ' ' . $order;
  }

  /**
   * Get parameters.
   */
  public function get_params() {
    return $this->params;
  }

  /**
   * Nowhere.
   */
  public function add_where($group, $field, $value = NULL, $operator = NULL) {
  }

  /**
   * Generates a unique placeholder used in the db query.
   */
  public function placeholder($base = 'views') {
    static $placeholders = array();
    if (!isset($placeholders[$base])) {
      $placeholders[$base] = 0;
      return ':' . $base;
    }
    else {
      return ':' . $base . ++$placeholders[$base];
    }
  }
  // -------------------------------------------------------- MediaMosa related.
  /**
   * Add metadata fields to the view data.
   *
   * @param array $data
   *   The data array to add metadata definitions to.
   */
  public static function get_views_data_metadata(&$data) {
    // Add metadata.
    foreach (mediamosa_ck::get_metadata_fields() as $set => $metadata) {
      // Add the whole set as a block.
      $data[$set] = array(
        'title' => $metadata['title'],
        'help' => t('Metadata block for set @set_label.', array('@set_label' => $metadata['title'])),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text_metadata',
        ),
      );

      foreach ($metadata['fields'] as $name => $type) {
        // 3 types; date, text and integer.
        switch ($type) {
          case 'DATETIME':
            $data[$set . '_' . $name] = array(
              'title' => $set . '.' . $name,
              'help' => t('Metadata date field in @set_label.', array('@set_label' => $metadata['title'])),
              'field' => array(
                'handler' => 'mediamosa_ck_views_field_date',
                'click sortable' => TRUE,
              ),
              'sort' => array(
                'handler' => 'mediamosa_ck_views_sort',
              ),
            );
            break;

          case 'CHAR':
          default:
            $data[$set . '_' . $name] = array(
              'title' => $set . '.' . $name,
              'help' => t('Metadata field in @set_label.', array('@set_label' => $metadata['title'])),
              'field' => array(
                'handler' => 'mediamosa_ck_views_field_text_metadata',
                'click sortable' => TRUE,
              ),
              'filter' => array(
                'handler' => 'mediamosa_ck_views_filter_text',
              ),
              'sort' => array(
                'handler' => 'mediamosa_ck_views_sort',
              ),
            );
            break;

          case 'INT':
            $data[$set . '_' . $name] = array(
              'title' => $set . '.' . $name,
              'help' => t('Metadata field in @set_label.', array('@set_label' => $metadata['title'])),
              'field' => array(
                'handler' => 'mediamosa_ck_views_field_text',
                'click sortable' => TRUE,
              ),
              'sort' => array(
                'handler' => 'mediamosa_ck_views_sort',
              ),
            );
            break;
        }
      }
    }
  }

  /**
   * Simple conversion from the data in get_views_data to fields in the output.
   *
   * For now this only works on data that is in the root of the output.
   */
  public static function get_fields($data) {
    $fields = array();

    foreach ($data as $field_name => $field) {
      // Include all that have a field handler.
      if (empty($field['field']['handler'])) {
        continue;
      }

      $fields[] = array('name' => $field_name);
    }

    return $fields;
  }

  /**
   * Add fields for owner and group.
   */
  public static function get_views_data_owner_stuff(&$data) {
    $data += array(
      'owner_id' => array(
        'title' => t('Owner'),
        'help' => t('The owner of the object.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_owner',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_owner',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'owned_id' => array(
        'title' => t('My assets'),
        'help' => t('Logged in user ownes the object.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_owned',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_owned',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'group_id' => array(
        'title' => t('Group'),
        'help' => t('The user group of the object.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
      ),

      'is_owner' => array(
        'title' => t('Access - Is owner'),
        'help' => t('A boolean value to indicate ownership of the object. TRUE / Yes will allow the user to edit or delete the object.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_boolean',
        ),
      ),

      'mediafile_mime_type' => array(
        'title' => t('Mime Type of Original'),
        'help' => t('Mime type of original mediafile. This filter is available since MediaMosa 3.5.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_mimetype_original',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'mime_type' => array(
        'title' => t('Mime Type'),
        'help' => t('Search if an assets contains files with given mimetype.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_mimetype',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'access_edit' => array(
        'title' => t('Access - Edit'),
        'help' => t('A boolean value to indicate that the current user can edit the object. TRUE / Yes will allow the user to edit the object.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_boolean',
        ),
      ),

      'access_delete' => array(
        'title' => t('Access - Delete'),
        'help' => t('A boolean value to indicate that the current user can delete the object. TRUE / Yes will allow the user to delete the object.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_boolean',
        ),
      ),
    );
  }

  public static function get_views_data_collection_common(&$data) {
    $data += array(
      // Fields.
      'coll_id' => array(
        'title' => t('Collection ID'),
        'help' => t('Unique Identifier of each collection.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => TRUE,
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'title' => array(
        'title' => t('Title'),
        'help' => t('The title of the collection.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'description' => array(
        'title' => t('Description'),
        'help' => t('The description of the collection.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_formatted_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'private' => array(
        'title' => t('private'),
        'help' => t('Collection is marked private.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
      ),

      'public' => array(
        'title' => t('public'),
        'help' => t('Collection is marked public.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
      ),

      'category' => array(
        'title' => t('category'),
        'help' => t('Collection is marked for main category.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
      ),

      'is_unappropriate' => array(
        'title' => t('is_unappropriate'),
        'help' => t('Collection is marked for not appropriate.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
      ),

      'numofvideos' => array(
        'title' => t('numofvideos'),
        'help' => t('The number of media items in the collection.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_text',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'created' => array(
        'title' => t('Created'),
        'help' => t('The date when collection was created.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_date',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'changed' => array(
        'title' => t('Changed'),
        'help' => t('The date when collection was changed.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_date',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'actions' => array(
        'title' => t('Actions'),
        'help' => t('Allowed administration links.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_collection_actions',
        ),
      ),
    );
  }

  /**
   * Add common fields for asset.
   *
   * @param array $data
   *   The data array to add asset definitions to.
   */
  public static function get_views_data_asset_common(&$data) {
    $data += array(
      // Fields.
      'asset_id' => array(
        'title' => t('Asset ID'),
        'help' => t('Unique Identifier of each asset.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => TRUE,
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'granted' => array(
        'title' => t('Granted'),
        'help' => t('Granted value indicates if you are allowed to see and play mediafile.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
        ),
        'filter' => array(
          'handler' => 'mediamosa_ck_views_filter_boolean',
        ),
      ),

      'viewed' => array(
        'title' => t('Viewed'),
        'help' => t('Number of times viewed.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => TRUE,
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'played' => array(
        'title' => t('Played'),
        'help' => t('Number of times played.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => TRUE,
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'videotimestamp' => array(
        'title' => t('Created'),
        'help' => t('When asset got media.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_date',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      'videotimestampmodified' => array(
        'title' => t('Changed'),
        'help' => t('When asset was modified.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_date',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),

      // Fields.
      'title' => array(
        'title' => t('Title'),
        'help' => t('The title of the asset. Is for display purpose only.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => FALSE,
        ),
      ),

      'fid' => array(
        'title' => t('File id'),
        'help' => t('The file id of the asset.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => FALSE,
        ),
      ),

      'description' => array(
        'title' => t('Description'),
        'help' => t('The description of the asset. Is for display purpose only.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_text',
          'click sortable' => FALSE,
        ),
      ),

      'still_url' => array(
        'title' => t('Still'),
        'help' => t('The thumbnail of the media.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_still',
        ),
      ),

      'player' => array(
        'title' => t('Media player'),
        'help' => t('The Media player allows you to get either a play link (URL) or the HTML object code to include a player.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_player',
        ),
      ),

      'playback_range' => array(
        'title' => t('Playback range'),
        'help' => t('Used in combination with player, start and endtime of media. example: t=10,20 (specifies that the video should play the range 10 seconds through 20 seconds).'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_playback_range',
        ),
      ),

      'mediafile_duration' => array(
        'title' => t('Mediafile duration'),
        'help' => t('The duration of the original mediafile.'),
        'field' => array(
          'handler' => 'mediamosa_ck_views_field_mediafile_duration',
        ),
        'sort' => array(
          'handler' => 'mediamosa_ck_views_sort',
        ),
      ),
    );
  }

  /**
   * Fields that only are in the asset call.
   *
   * @param array $data
   *   The data array to add asset definitions to.
   */
  public static function get_views_data_asset_specific(&$data) {
    $data += array();
  }

  /**
   * Simple conversion from xmlobject to array.
   */
  public static function xmlobject2array($simplexmlelement) {
    $array = json_decode(json_encode($simplexmlelement), TRUE);
    self::_xmlobject2array_helper($array);
    return $array;
  }

  /**
   * Fixes empty array's.
   */
  private static function _xmlobject2array_helper(&$array) {
    foreach ($array as $key => &$child_array) {
      if (is_array($child_array)) {
        if (empty($child_array)) {
          $array[$key] = '';
        }
        else {
          self::_xmlobject2array_helper($child_array);
        }
      }
    }
  }
}
