# $Id: Canonical.pm 43961 2015-10-05 11:54:50Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/libnet-ldap-schema-canonical-perl/lib/Net/LDAP/Schema/Canonical.pm $

use strict;
use warnings FATAL => 'all';

package Net::LDAP::Schema;

use Unicode::Normalize;
use Encode;

my %rfc4519_namemap = (
	'2.5.4.15' => 'businessCategory',
	'2.5.4.6' => 'c',
	'2.5.4.3' => 'cn',
	'0.9.2342.19200300.100.1.25' => 'dc',
	'2.5.4.13' => 'description',
	'2.5.4.27' => 'destinationIndicator',
	'2.5.4.49' => 'distinguishedName',
	'2.5.4.46' => 'dnQualifier',
	'2.5.4.47' => 'enhancedSearchGuide',
	'2.5.4.23' => 'facsimileTelephoneNumber',
	'2.5.4.44' => 'generationQualifier',
	'2.5.4.42' => 'givenName',
	'2.5.4.51' => 'houseIdentifier',
	'2.5.4.43' => 'initials',
	'2.5.4.25' => 'internationalISDNNumber',
	'2.5.4.7' => 'l',
	'2.5.4.31' => 'member',
	'2.5.4.41' => 'name',
	'2.5.4.10' => 'o',
	'2.5.4.11' => 'ou',
	'2.5.4.32' => 'owner',
	'2.5.4.19' => 'physicalDeliveryOfficeName',
	'2.5.4.16' => 'postalAddress',
	'2.5.4.17' => 'postalCode',
	'2.5.4.18' => 'postOfficeBox',
	'2.5.4.28' => 'preferredDeliveryMethod',
	'2.5.4.26' => 'registeredAddress',
	'2.5.4.33' => 'roleOccupant',
	'2.5.4.14' => 'searchGuide',
	'2.5.4.34' => 'seeAlso',
	'2.5.4.5' => 'serialNumber',
	'2.5.4.4' => 'sn',
	'2.5.4.8' => 'st',
	'2.5.4.9' => 'street',
	'2.5.4.20' => 'telephoneNumber',
	'2.5.4.22' => 'teletexTerminalIdentifier',
	'2.5.4.21' => 'telexNumber',
	'2.5.4.12' => 'title',
	'0.9.2342.19200300.100.1.1' => 'uid',
	'2.5.4.50' => 'uniqueMember',
	'2.5.4.35' => 'userPassword',
	'2.5.4.24' => 'x121Address',
	'2.5.4.45' => 'x500UniqueIdentifier',
	businesscategory => 'businessCategory',
	c => 'c',
	countryname => 'c',
	cn => 'cn',
	commonname => 'cn',
	dc => 'dc',
	domaincomponent => 'dc',
	description => 'description',
	destinationindicator => 'destinationIndicator',
	distinguishedname => 'distinguishedName',
	dnqualifier => 'dnQualifier',
	enhancedsearchguide => 'enhancedSearchGuide',
	facsimiletelephonenumber => 'facsimileTelephoneNumber',
	generationqualifier => 'generationQualifier',
	givenname => 'givenName',
	gn => 'givenName',
	houseidentifier => 'houseIdentifier',
	initials => 'initials',
	internationalisdnnumber => 'internationalISDNNumber',
	l => 'l',
	localityname => 'l',
	member => 'member',
	name => 'name',
	o => 'o',
	organizationName => 'o',
	organizationalunitname => 'ou',
	ou => 'ou',
	owner => 'owner',
	physicaldeliveryofficename => 'physicalDeliveryOfficeName',
	postaladdress => 'postalAddress',
	postalcode => 'postalCode',
	postofficebox => 'postOfficeBox',
	preferreddeliverymethod => 'preferredDeliveryMethod',
	registeredaddress => 'registeredAddress',
	roleoccupant => 'roleOccupant',
	searchguide => 'searchGuide',
	seealso => 'seeAlso',
	serialnumber => 'serialNumber',
	sn => 'sn',
	st => 'st',
	street => 'street',
	surname => 'sn',
	telephonenumber => 'telephoneNumber',
	teletexterminalidentifier => 'teletexTerminalIdentifier',
	telexnumber => 'telexNumber',
	title => 'title',
	uid => 'uid',
	uniquemember => 'uniqueMember',
	userid => 'uid',
	userpassword => 'userPassword',
	x121address => 'x121Address',
	x500uniqueidentifier => 'x500UniqueIdentifier',
);

my %rfc4519_typemap = (
	businessCategory => 'caseIgnoreMatch',
	c => 'caseIgnoreMatch',
	cn => 'caseIgnoreMatch',
	dc => 'caseIgnoreIA5Match',
	description => 'caseIgnoreMatch',
	destinationIndicator => 'caseIgnoreMatch',
	distinguishedName => 'distinguishedNameMatch',
	dnQualifier => 'caseIgnoreMatch',
	enhancedSearchGuide => '',
	facsimileTelephoneNumber => '',
	generationQualifier => 'caseIgnoreMatch',
	givenName => 'caseIgnoreMatch',
	houseIdentifier => 'caseIgnoreMatch',
	initials => 'caseIgnoreMatch',
	internationalISDNNumber => 'numericStringMatch',
	l => 'caseIgnoreMatch',
	member => 'distinguishedNameMatch',
	name => 'caseIgnoreMatch',
	o => 'caseIgnoreMatch',
	ou => 'caseIgnoreMatch',
	owner => 'distinguishedNameMatch',
	physicalDeliveryOfficeName => 'caseIgnoreMatch',
	postalAddress => 'caseIgnoreListMatch',
	postalCode => 'caseIgnoreMatch',
	postOfficeBox => 'caseIgnoreMatch',
	preferredDeliveryMethod => '',
	registeredAddress => 'caseIgnoreListMatch',
	roleOccupant => 'distinguishedNameMatch',
	searchGuide => '',
	seeAlso => 'distinguishedNameMatch',
	serialNumber => 'caseIgnoreMatch',
	sn => 'caseIgnoreMatch',
	st => 'caseIgnoreMatch',
	street => 'caseIgnoreMatch',
	telephoneNumber => 'telephoneNumberMatch',
	teletexTerminalIdentifier => '',
	telexNumber => '',
	title => 'caseIgnoreMatch',
	uid => 'caseIgnoreMatch',
	uniqueMember => 'uniqueMemberMatch',
	userPassword => 'octetStringMatch',
	x121Address => 'numericStringMatch',
	x500UniqueIdentifier => 'bitStringMatch',

	# FIXME: add all syntax identifiers from section 3.3
	# FIXME: add all *Match identifiers from section 4.2
);

sub canonicalize_name {
	my ($self, $name) = @_;

	$name = lc($name);
	$name =~ s/(;.*)//;
	my $options = $1 // '';

	return "$rfc4519_namemap{$name}$options"
		if exists $rfc4519_namemap{$name};

	if(my $c = ref $self && $self->attribute($name)) {
		return $c->{name}.$options;
	} else {
		return $name.$options;
	}
}

# RFC 3454 appendix B.2
my %rfc3454_casemap = (
	"\x{0041}" => "\x{0061}",
	"\x{0042}" => "\x{0062}",
	"\x{0043}" => "\x{0063}",
	"\x{0044}" => "\x{0064}",
	"\x{0045}" => "\x{0065}",
	"\x{0046}" => "\x{0066}",
	"\x{0047}" => "\x{0067}",
	"\x{0048}" => "\x{0068}",
	"\x{0049}" => "\x{0069}",
	"\x{004A}" => "\x{006A}",
	"\x{004B}" => "\x{006B}",
	"\x{004C}" => "\x{006C}",
	"\x{004D}" => "\x{006D}",
	"\x{004E}" => "\x{006E}",
	"\x{004F}" => "\x{006F}",
	"\x{0050}" => "\x{0070}",
	"\x{0051}" => "\x{0071}",
	"\x{0052}" => "\x{0072}",
	"\x{0053}" => "\x{0073}",
	"\x{0054}" => "\x{0074}",
	"\x{0055}" => "\x{0075}",
	"\x{0056}" => "\x{0076}",
	"\x{0057}" => "\x{0077}",
	"\x{0058}" => "\x{0078}",
	"\x{0059}" => "\x{0079}",
	"\x{005A}" => "\x{007A}",
	"\x{00B5}" => "\x{03BC}",
	"\x{00C0}" => "\x{00E0}",
	"\x{00C1}" => "\x{00E1}",
	"\x{00C2}" => "\x{00E2}",
	"\x{00C3}" => "\x{00E3}",
	"\x{00C4}" => "\x{00E4}",
	"\x{00C5}" => "\x{00E5}",
	"\x{00C6}" => "\x{00E6}",
	"\x{00C7}" => "\x{00E7}",
	"\x{00C8}" => "\x{00E8}",
	"\x{00C9}" => "\x{00E9}",
	"\x{00CA}" => "\x{00EA}",
	"\x{00CB}" => "\x{00EB}",
	"\x{00CC}" => "\x{00EC}",
	"\x{00CD}" => "\x{00ED}",
	"\x{00CE}" => "\x{00EE}",
	"\x{00CF}" => "\x{00EF}",
	"\x{00D0}" => "\x{00F0}",
	"\x{00D1}" => "\x{00F1}",
	"\x{00D2}" => "\x{00F2}",
	"\x{00D3}" => "\x{00F3}",
	"\x{00D4}" => "\x{00F4}",
	"\x{00D5}" => "\x{00F5}",
	"\x{00D6}" => "\x{00F6}",
	"\x{00D8}" => "\x{00F8}",
	"\x{00D9}" => "\x{00F9}",
	"\x{00DA}" => "\x{00FA}",
	"\x{00DB}" => "\x{00FB}",
	"\x{00DC}" => "\x{00FC}",
	"\x{00DD}" => "\x{00FD}",
	"\x{00DE}" => "\x{00FE}",
	"\x{00DF}" => "\x{0073}\x{0073}",
	"\x{0100}" => "\x{0101}",
	"\x{0102}" => "\x{0103}",
	"\x{0104}" => "\x{0105}",
	"\x{0106}" => "\x{0107}",
	"\x{0108}" => "\x{0109}",
	"\x{010A}" => "\x{010B}",
	"\x{010C}" => "\x{010D}",
	"\x{010E}" => "\x{010F}",
	"\x{0110}" => "\x{0111}",
	"\x{0112}" => "\x{0113}",
	"\x{0114}" => "\x{0115}",
	"\x{0116}" => "\x{0117}",
	"\x{0118}" => "\x{0119}",
	"\x{011A}" => "\x{011B}",
	"\x{011C}" => "\x{011D}",
	"\x{011E}" => "\x{011F}",
	"\x{0120}" => "\x{0121}",
	"\x{0122}" => "\x{0123}",
	"\x{0124}" => "\x{0125}",
	"\x{0126}" => "\x{0127}",
	"\x{0128}" => "\x{0129}",
	"\x{012A}" => "\x{012B}",
	"\x{012C}" => "\x{012D}",
	"\x{012E}" => "\x{012F}",
	"\x{0130}" => "\x{0069}\x{0307}",
	"\x{0132}" => "\x{0133}",
	"\x{0134}" => "\x{0135}",
	"\x{0136}" => "\x{0137}",
	"\x{0139}" => "\x{013A}",
	"\x{013B}" => "\x{013C}",
	"\x{013D}" => "\x{013E}",
	"\x{013F}" => "\x{0140}",
	"\x{0141}" => "\x{0142}",
	"\x{0143}" => "\x{0144}",
	"\x{0145}" => "\x{0146}",
	"\x{0147}" => "\x{0148}",
	"\x{0149}" => "\x{02BC}\x{006E}",
	"\x{014A}" => "\x{014B}",
	"\x{014C}" => "\x{014D}",
	"\x{014E}" => "\x{014F}",
	"\x{0150}" => "\x{0151}",
	"\x{0152}" => "\x{0153}",
	"\x{0154}" => "\x{0155}",
	"\x{0156}" => "\x{0157}",
	"\x{0158}" => "\x{0159}",
	"\x{015A}" => "\x{015B}",
	"\x{015C}" => "\x{015D}",
	"\x{015E}" => "\x{015F}",
	"\x{0160}" => "\x{0161}",
	"\x{0162}" => "\x{0163}",
	"\x{0164}" => "\x{0165}",
	"\x{0166}" => "\x{0167}",
	"\x{0168}" => "\x{0169}",
	"\x{016A}" => "\x{016B}",
	"\x{016C}" => "\x{016D}",
	"\x{016E}" => "\x{016F}",
	"\x{0170}" => "\x{0171}",
	"\x{0172}" => "\x{0173}",
	"\x{0174}" => "\x{0175}",
	"\x{0176}" => "\x{0177}",
	"\x{0178}" => "\x{00FF}",
	"\x{0179}" => "\x{017A}",
	"\x{017B}" => "\x{017C}",
	"\x{017D}" => "\x{017E}",
	"\x{017F}" => "\x{0073}",
	"\x{0181}" => "\x{0253}",
	"\x{0182}" => "\x{0183}",
	"\x{0184}" => "\x{0185}",
	"\x{0186}" => "\x{0254}",
	"\x{0187}" => "\x{0188}",
	"\x{0189}" => "\x{0256}",
	"\x{018A}" => "\x{0257}",
	"\x{018B}" => "\x{018C}",
	"\x{018E}" => "\x{01DD}",
	"\x{018F}" => "\x{0259}",
	"\x{0190}" => "\x{025B}",
	"\x{0191}" => "\x{0192}",
	"\x{0193}" => "\x{0260}",
	"\x{0194}" => "\x{0263}",
	"\x{0196}" => "\x{0269}",
	"\x{0197}" => "\x{0268}",
	"\x{0198}" => "\x{0199}",
	"\x{019C}" => "\x{026F}",
	"\x{019D}" => "\x{0272}",
	"\x{019F}" => "\x{0275}",
	"\x{01A0}" => "\x{01A1}",
	"\x{01A2}" => "\x{01A3}",
	"\x{01A4}" => "\x{01A5}",
	"\x{01A6}" => "\x{0280}",
	"\x{01A7}" => "\x{01A8}",
	"\x{01A9}" => "\x{0283}",
	"\x{01AC}" => "\x{01AD}",
	"\x{01AE}" => "\x{0288}",
	"\x{01AF}" => "\x{01B0}",
	"\x{01B1}" => "\x{028A}",
	"\x{01B2}" => "\x{028B}",
	"\x{01B3}" => "\x{01B4}",
	"\x{01B5}" => "\x{01B6}",
	"\x{01B7}" => "\x{0292}",
	"\x{01B8}" => "\x{01B9}",
	"\x{01BC}" => "\x{01BD}",
	"\x{01C4}" => "\x{01C6}",
	"\x{01C5}" => "\x{01C6}",
	"\x{01C7}" => "\x{01C9}",
	"\x{01C8}" => "\x{01C9}",
	"\x{01CA}" => "\x{01CC}",
	"\x{01CB}" => "\x{01CC}",
	"\x{01CD}" => "\x{01CE}",
	"\x{01CF}" => "\x{01D0}",
	"\x{01D1}" => "\x{01D2}",
	"\x{01D3}" => "\x{01D4}",
	"\x{01D5}" => "\x{01D6}",
	"\x{01D7}" => "\x{01D8}",
	"\x{01D9}" => "\x{01DA}",
	"\x{01DB}" => "\x{01DC}",
	"\x{01DE}" => "\x{01DF}",
	"\x{01E0}" => "\x{01E1}",
	"\x{01E2}" => "\x{01E3}",
	"\x{01E4}" => "\x{01E5}",
	"\x{01E6}" => "\x{01E7}",
	"\x{01E8}" => "\x{01E9}",
	"\x{01EA}" => "\x{01EB}",
	"\x{01EC}" => "\x{01ED}",
	"\x{01EE}" => "\x{01EF}",
	"\x{01F0}" => "\x{006A}\x{030C}",
	"\x{01F1}" => "\x{01F3}",
	"\x{01F2}" => "\x{01F3}",
	"\x{01F4}" => "\x{01F5}",
	"\x{01F6}" => "\x{0195}",
	"\x{01F7}" => "\x{01BF}",
	"\x{01F8}" => "\x{01F9}",
	"\x{01FA}" => "\x{01FB}",
	"\x{01FC}" => "\x{01FD}",
	"\x{01FE}" => "\x{01FF}",
	"\x{0200}" => "\x{0201}",
	"\x{0202}" => "\x{0203}",
	"\x{0204}" => "\x{0205}",
	"\x{0206}" => "\x{0207}",
	"\x{0208}" => "\x{0209}",
	"\x{020A}" => "\x{020B}",
	"\x{020C}" => "\x{020D}",
	"\x{020E}" => "\x{020F}",
	"\x{0210}" => "\x{0211}",
	"\x{0212}" => "\x{0213}",
	"\x{0214}" => "\x{0215}",
	"\x{0216}" => "\x{0217}",
	"\x{0218}" => "\x{0219}",
	"\x{021A}" => "\x{021B}",
	"\x{021C}" => "\x{021D}",
	"\x{021E}" => "\x{021F}",
	"\x{0220}" => "\x{019E}",
	"\x{0222}" => "\x{0223}",
	"\x{0224}" => "\x{0225}",
	"\x{0226}" => "\x{0227}",
	"\x{0228}" => "\x{0229}",
	"\x{022A}" => "\x{022B}",
	"\x{022C}" => "\x{022D}",
	"\x{022E}" => "\x{022F}",
	"\x{0230}" => "\x{0231}",
	"\x{0232}" => "\x{0233}",
	"\x{0345}" => "\x{03B9}",
	"\x{037A}" => "\x{0020}\x{03B9}",
	"\x{0386}" => "\x{03AC}",
	"\x{0388}" => "\x{03AD}",
	"\x{0389}" => "\x{03AE}",
	"\x{038A}" => "\x{03AF}",
	"\x{038C}" => "\x{03CC}",
	"\x{038E}" => "\x{03CD}",
	"\x{038F}" => "\x{03CE}",
	"\x{0390}" => "\x{03B9}\x{0308}\x{0301}",
	"\x{0391}" => "\x{03B1}",
	"\x{0392}" => "\x{03B2}",
	"\x{0393}" => "\x{03B3}",
	"\x{0394}" => "\x{03B4}",
	"\x{0395}" => "\x{03B5}",
	"\x{0396}" => "\x{03B6}",
	"\x{0397}" => "\x{03B7}",
	"\x{0398}" => "\x{03B8}",
	"\x{0399}" => "\x{03B9}",
	"\x{039A}" => "\x{03BA}",
	"\x{039B}" => "\x{03BB}",
	"\x{039C}" => "\x{03BC}",
	"\x{039D}" => "\x{03BD}",
	"\x{039E}" => "\x{03BE}",
	"\x{039F}" => "\x{03BF}",
	"\x{03A0}" => "\x{03C0}",
	"\x{03A1}" => "\x{03C1}",
	"\x{03A3}" => "\x{03C3}",
	"\x{03A4}" => "\x{03C4}",
	"\x{03A5}" => "\x{03C5}",
	"\x{03A6}" => "\x{03C6}",
	"\x{03A7}" => "\x{03C7}",
	"\x{03A8}" => "\x{03C8}",
	"\x{03A9}" => "\x{03C9}",
	"\x{03AA}" => "\x{03CA}",
	"\x{03AB}" => "\x{03CB}",
	"\x{03B0}" => "\x{03C5}\x{0308}\x{0301}",
	"\x{03C2}" => "\x{03C3}",
	"\x{03D0}" => "\x{03B2}",
	"\x{03D1}" => "\x{03B8}",
	"\x{03D2}" => "\x{03C5}",
	"\x{03D3}" => "\x{03CD}",
	"\x{03D4}" => "\x{03CB}",
	"\x{03D5}" => "\x{03C6}",
	"\x{03D6}" => "\x{03C0}",
	"\x{03D8}" => "\x{03D9}",
	"\x{03DA}" => "\x{03DB}",
	"\x{03DC}" => "\x{03DD}",
	"\x{03DE}" => "\x{03DF}",
	"\x{03E0}" => "\x{03E1}",
	"\x{03E2}" => "\x{03E3}",
	"\x{03E4}" => "\x{03E5}",
	"\x{03E6}" => "\x{03E7}",
	"\x{03E8}" => "\x{03E9}",
	"\x{03EA}" => "\x{03EB}",
	"\x{03EC}" => "\x{03ED}",
	"\x{03EE}" => "\x{03EF}",
	"\x{03F0}" => "\x{03BA}",
	"\x{03F1}" => "\x{03C1}",
	"\x{03F2}" => "\x{03C3}",
	"\x{03F4}" => "\x{03B8}",
	"\x{03F5}" => "\x{03B5}",
	"\x{0400}" => "\x{0450}",
	"\x{0401}" => "\x{0451}",
	"\x{0402}" => "\x{0452}",
	"\x{0403}" => "\x{0453}",
	"\x{0404}" => "\x{0454}",
	"\x{0405}" => "\x{0455}",
	"\x{0406}" => "\x{0456}",
	"\x{0407}" => "\x{0457}",
	"\x{0408}" => "\x{0458}",
	"\x{0409}" => "\x{0459}",
	"\x{040A}" => "\x{045A}",
	"\x{040B}" => "\x{045B}",
	"\x{040C}" => "\x{045C}",
	"\x{040D}" => "\x{045D}",
	"\x{040E}" => "\x{045E}",
	"\x{040F}" => "\x{045F}",
	"\x{0410}" => "\x{0430}",
	"\x{0411}" => "\x{0431}",
	"\x{0412}" => "\x{0432}",
	"\x{0413}" => "\x{0433}",
	"\x{0414}" => "\x{0434}",
	"\x{0415}" => "\x{0435}",
	"\x{0416}" => "\x{0436}",
	"\x{0417}" => "\x{0437}",
	"\x{0418}" => "\x{0438}",
	"\x{0419}" => "\x{0439}",
	"\x{041A}" => "\x{043A}",
	"\x{041B}" => "\x{043B}",
	"\x{041C}" => "\x{043C}",
	"\x{041D}" => "\x{043D}",
	"\x{041E}" => "\x{043E}",
	"\x{041F}" => "\x{043F}",
	"\x{0420}" => "\x{0440}",
	"\x{0421}" => "\x{0441}",
	"\x{0422}" => "\x{0442}",
	"\x{0423}" => "\x{0443}",
	"\x{0424}" => "\x{0444}",
	"\x{0425}" => "\x{0445}",
	"\x{0426}" => "\x{0446}",
	"\x{0427}" => "\x{0447}",
	"\x{0428}" => "\x{0448}",
	"\x{0429}" => "\x{0449}",
	"\x{042A}" => "\x{044A}",
	"\x{042B}" => "\x{044B}",
	"\x{042C}" => "\x{044C}",
	"\x{042D}" => "\x{044D}",
	"\x{042E}" => "\x{044E}",
	"\x{042F}" => "\x{044F}",
	"\x{0460}" => "\x{0461}",
	"\x{0462}" => "\x{0463}",
	"\x{0464}" => "\x{0465}",
	"\x{0466}" => "\x{0467}",
	"\x{0468}" => "\x{0469}",
	"\x{046A}" => "\x{046B}",
	"\x{046C}" => "\x{046D}",
	"\x{046E}" => "\x{046F}",
	"\x{0470}" => "\x{0471}",
	"\x{0472}" => "\x{0473}",
	"\x{0474}" => "\x{0475}",
	"\x{0476}" => "\x{0477}",
	"\x{0478}" => "\x{0479}",
	"\x{047A}" => "\x{047B}",
	"\x{047C}" => "\x{047D}",
	"\x{047E}" => "\x{047F}",
	"\x{0480}" => "\x{0481}",
	"\x{048A}" => "\x{048B}",
	"\x{048C}" => "\x{048D}",
	"\x{048E}" => "\x{048F}",
	"\x{0490}" => "\x{0491}",
	"\x{0492}" => "\x{0493}",
	"\x{0494}" => "\x{0495}",
	"\x{0496}" => "\x{0497}",
	"\x{0498}" => "\x{0499}",
	"\x{049A}" => "\x{049B}",
	"\x{049C}" => "\x{049D}",
	"\x{049E}" => "\x{049F}",
	"\x{04A0}" => "\x{04A1}",
	"\x{04A2}" => "\x{04A3}",
	"\x{04A4}" => "\x{04A5}",
	"\x{04A6}" => "\x{04A7}",
	"\x{04A8}" => "\x{04A9}",
	"\x{04AA}" => "\x{04AB}",
	"\x{04AC}" => "\x{04AD}",
	"\x{04AE}" => "\x{04AF}",
	"\x{04B0}" => "\x{04B1}",
	"\x{04B2}" => "\x{04B3}",
	"\x{04B4}" => "\x{04B5}",
	"\x{04B6}" => "\x{04B7}",
	"\x{04B8}" => "\x{04B9}",
	"\x{04BA}" => "\x{04BB}",
	"\x{04BC}" => "\x{04BD}",
	"\x{04BE}" => "\x{04BF}",
	"\x{04C1}" => "\x{04C2}",
	"\x{04C3}" => "\x{04C4}",
	"\x{04C5}" => "\x{04C6}",
	"\x{04C7}" => "\x{04C8}",
	"\x{04C9}" => "\x{04CA}",
	"\x{04CB}" => "\x{04CC}",
	"\x{04CD}" => "\x{04CE}",
	"\x{04D0}" => "\x{04D1}",
	"\x{04D2}" => "\x{04D3}",
	"\x{04D4}" => "\x{04D5}",
	"\x{04D6}" => "\x{04D7}",
	"\x{04D8}" => "\x{04D9}",
	"\x{04DA}" => "\x{04DB}",
	"\x{04DC}" => "\x{04DD}",
	"\x{04DE}" => "\x{04DF}",
	"\x{04E0}" => "\x{04E1}",
	"\x{04E2}" => "\x{04E3}",
	"\x{04E4}" => "\x{04E5}",
	"\x{04E6}" => "\x{04E7}",
	"\x{04E8}" => "\x{04E9}",
	"\x{04EA}" => "\x{04EB}",
	"\x{04EC}" => "\x{04ED}",
	"\x{04EE}" => "\x{04EF}",
	"\x{04F0}" => "\x{04F1}",
	"\x{04F2}" => "\x{04F3}",
	"\x{04F4}" => "\x{04F5}",
	"\x{04F8}" => "\x{04F9}",
	"\x{0500}" => "\x{0501}",
	"\x{0502}" => "\x{0503}",
	"\x{0504}" => "\x{0505}",
	"\x{0506}" => "\x{0507}",
	"\x{0508}" => "\x{0509}",
	"\x{050A}" => "\x{050B}",
	"\x{050C}" => "\x{050D}",
	"\x{050E}" => "\x{050F}",
	"\x{0531}" => "\x{0561}",
	"\x{0532}" => "\x{0562}",
	"\x{0533}" => "\x{0563}",
	"\x{0534}" => "\x{0564}",
	"\x{0535}" => "\x{0565}",
	"\x{0536}" => "\x{0566}",
	"\x{0537}" => "\x{0567}",
	"\x{0538}" => "\x{0568}",
	"\x{0539}" => "\x{0569}",
	"\x{053A}" => "\x{056A}",
	"\x{053B}" => "\x{056B}",
	"\x{053C}" => "\x{056C}",
	"\x{053D}" => "\x{056D}",
	"\x{053E}" => "\x{056E}",
	"\x{053F}" => "\x{056F}",
	"\x{0540}" => "\x{0570}",
	"\x{0541}" => "\x{0571}",
	"\x{0542}" => "\x{0572}",
	"\x{0543}" => "\x{0573}",
	"\x{0544}" => "\x{0574}",
	"\x{0545}" => "\x{0575}",
	"\x{0546}" => "\x{0576}",
	"\x{0547}" => "\x{0577}",
	"\x{0548}" => "\x{0578}",
	"\x{0549}" => "\x{0579}",
	"\x{054A}" => "\x{057A}",
	"\x{054B}" => "\x{057B}",
	"\x{054C}" => "\x{057C}",
	"\x{054D}" => "\x{057D}",
	"\x{054E}" => "\x{057E}",
	"\x{054F}" => "\x{057F}",
	"\x{0550}" => "\x{0580}",
	"\x{0551}" => "\x{0581}",
	"\x{0552}" => "\x{0582}",
	"\x{0553}" => "\x{0583}",
	"\x{0554}" => "\x{0584}",
	"\x{0555}" => "\x{0585}",
	"\x{0556}" => "\x{0586}",
	"\x{0587}" => "\x{0565}\x{0582}",
	"\x{1E00}" => "\x{1E01}",
	"\x{1E02}" => "\x{1E03}",
	"\x{1E04}" => "\x{1E05}",
	"\x{1E06}" => "\x{1E07}",
	"\x{1E08}" => "\x{1E09}",
	"\x{1E0A}" => "\x{1E0B}",
	"\x{1E0C}" => "\x{1E0D}",
	"\x{1E0E}" => "\x{1E0F}",
	"\x{1E10}" => "\x{1E11}",
	"\x{1E12}" => "\x{1E13}",
	"\x{1E14}" => "\x{1E15}",
	"\x{1E16}" => "\x{1E17}",
	"\x{1E18}" => "\x{1E19}",
	"\x{1E1A}" => "\x{1E1B}",
	"\x{1E1C}" => "\x{1E1D}",
	"\x{1E1E}" => "\x{1E1F}",
	"\x{1E20}" => "\x{1E21}",
	"\x{1E22}" => "\x{1E23}",
	"\x{1E24}" => "\x{1E25}",
	"\x{1E26}" => "\x{1E27}",
	"\x{1E28}" => "\x{1E29}",
	"\x{1E2A}" => "\x{1E2B}",
	"\x{1E2C}" => "\x{1E2D}",
	"\x{1E2E}" => "\x{1E2F}",
	"\x{1E30}" => "\x{1E31}",
	"\x{1E32}" => "\x{1E33}",
	"\x{1E34}" => "\x{1E35}",
	"\x{1E36}" => "\x{1E37}",
	"\x{1E38}" => "\x{1E39}",
	"\x{1E3A}" => "\x{1E3B}",
	"\x{1E3C}" => "\x{1E3D}",
	"\x{1E3E}" => "\x{1E3F}",
	"\x{1E40}" => "\x{1E41}",
	"\x{1E42}" => "\x{1E43}",
	"\x{1E44}" => "\x{1E45}",
	"\x{1E46}" => "\x{1E47}",
	"\x{1E48}" => "\x{1E49}",
	"\x{1E4A}" => "\x{1E4B}",
	"\x{1E4C}" => "\x{1E4D}",
	"\x{1E4E}" => "\x{1E4F}",
	"\x{1E50}" => "\x{1E51}",
	"\x{1E52}" => "\x{1E53}",
	"\x{1E54}" => "\x{1E55}",
	"\x{1E56}" => "\x{1E57}",
	"\x{1E58}" => "\x{1E59}",
	"\x{1E5A}" => "\x{1E5B}",
	"\x{1E5C}" => "\x{1E5D}",
	"\x{1E5E}" => "\x{1E5F}",
	"\x{1E60}" => "\x{1E61}",
	"\x{1E62}" => "\x{1E63}",
	"\x{1E64}" => "\x{1E65}",
	"\x{1E66}" => "\x{1E67}",
	"\x{1E68}" => "\x{1E69}",
	"\x{1E6A}" => "\x{1E6B}",
	"\x{1E6C}" => "\x{1E6D}",
	"\x{1E6E}" => "\x{1E6F}",
	"\x{1E70}" => "\x{1E71}",
	"\x{1E72}" => "\x{1E73}",
	"\x{1E74}" => "\x{1E75}",
	"\x{1E76}" => "\x{1E77}",
	"\x{1E78}" => "\x{1E79}",
	"\x{1E7A}" => "\x{1E7B}",
	"\x{1E7C}" => "\x{1E7D}",
	"\x{1E7E}" => "\x{1E7F}",
	"\x{1E80}" => "\x{1E81}",
	"\x{1E82}" => "\x{1E83}",
	"\x{1E84}" => "\x{1E85}",
	"\x{1E86}" => "\x{1E87}",
	"\x{1E88}" => "\x{1E89}",
	"\x{1E8A}" => "\x{1E8B}",
	"\x{1E8C}" => "\x{1E8D}",
	"\x{1E8E}" => "\x{1E8F}",
	"\x{1E90}" => "\x{1E91}",
	"\x{1E92}" => "\x{1E93}",
	"\x{1E94}" => "\x{1E95}",
	"\x{1E96}" => "\x{0068}\x{0331}",
	"\x{1E97}" => "\x{0074}\x{0308}",
	"\x{1E98}" => "\x{0077}\x{030A}",
	"\x{1E99}" => "\x{0079}\x{030A}",
	"\x{1E9A}" => "\x{0061}\x{02BE}",
	"\x{1E9B}" => "\x{1E61}",
	"\x{1EA0}" => "\x{1EA1}",
	"\x{1EA2}" => "\x{1EA3}",
	"\x{1EA4}" => "\x{1EA5}",
	"\x{1EA6}" => "\x{1EA7}",
	"\x{1EA8}" => "\x{1EA9}",
	"\x{1EAA}" => "\x{1EAB}",
	"\x{1EAC}" => "\x{1EAD}",
	"\x{1EAE}" => "\x{1EAF}",
	"\x{1EB0}" => "\x{1EB1}",
	"\x{1EB2}" => "\x{1EB3}",
	"\x{1EB4}" => "\x{1EB5}",
	"\x{1EB6}" => "\x{1EB7}",
	"\x{1EB8}" => "\x{1EB9}",
	"\x{1EBA}" => "\x{1EBB}",
	"\x{1EBC}" => "\x{1EBD}",
	"\x{1EBE}" => "\x{1EBF}",
	"\x{1EC0}" => "\x{1EC1}",
	"\x{1EC2}" => "\x{1EC3}",
	"\x{1EC4}" => "\x{1EC5}",
	"\x{1EC6}" => "\x{1EC7}",
	"\x{1EC8}" => "\x{1EC9}",
	"\x{1ECA}" => "\x{1ECB}",
	"\x{1ECC}" => "\x{1ECD}",
	"\x{1ECE}" => "\x{1ECF}",
	"\x{1ED0}" => "\x{1ED1}",
	"\x{1ED2}" => "\x{1ED3}",
	"\x{1ED4}" => "\x{1ED5}",
	"\x{1ED6}" => "\x{1ED7}",
	"\x{1ED8}" => "\x{1ED9}",
	"\x{1EDA}" => "\x{1EDB}",
	"\x{1EDC}" => "\x{1EDD}",
	"\x{1EDE}" => "\x{1EDF}",
	"\x{1EE0}" => "\x{1EE1}",
	"\x{1EE2}" => "\x{1EE3}",
	"\x{1EE4}" => "\x{1EE5}",
	"\x{1EE6}" => "\x{1EE7}",
	"\x{1EE8}" => "\x{1EE9}",
	"\x{1EEA}" => "\x{1EEB}",
	"\x{1EEC}" => "\x{1EED}",
	"\x{1EEE}" => "\x{1EEF}",
	"\x{1EF0}" => "\x{1EF1}",
	"\x{1EF2}" => "\x{1EF3}",
	"\x{1EF4}" => "\x{1EF5}",
	"\x{1EF6}" => "\x{1EF7}",
	"\x{1EF8}" => "\x{1EF9}",
	"\x{1F08}" => "\x{1F00}",
	"\x{1F09}" => "\x{1F01}",
	"\x{1F0A}" => "\x{1F02}",
	"\x{1F0B}" => "\x{1F03}",
	"\x{1F0C}" => "\x{1F04}",
	"\x{1F0D}" => "\x{1F05}",
	"\x{1F0E}" => "\x{1F06}",
	"\x{1F0F}" => "\x{1F07}",
	"\x{1F18}" => "\x{1F10}",
	"\x{1F19}" => "\x{1F11}",
	"\x{1F1A}" => "\x{1F12}",
	"\x{1F1B}" => "\x{1F13}",
	"\x{1F1C}" => "\x{1F14}",
	"\x{1F1D}" => "\x{1F15}",
	"\x{1F28}" => "\x{1F20}",
	"\x{1F29}" => "\x{1F21}",
	"\x{1F2A}" => "\x{1F22}",
	"\x{1F2B}" => "\x{1F23}",
	"\x{1F2C}" => "\x{1F24}",
	"\x{1F2D}" => "\x{1F25}",
	"\x{1F2E}" => "\x{1F26}",
	"\x{1F2F}" => "\x{1F27}",
	"\x{1F38}" => "\x{1F30}",
	"\x{1F39}" => "\x{1F31}",
	"\x{1F3A}" => "\x{1F32}",
	"\x{1F3B}" => "\x{1F33}",
	"\x{1F3C}" => "\x{1F34}",
	"\x{1F3D}" => "\x{1F35}",
	"\x{1F3E}" => "\x{1F36}",
	"\x{1F3F}" => "\x{1F37}",
	"\x{1F48}" => "\x{1F40}",
	"\x{1F49}" => "\x{1F41}",
	"\x{1F4A}" => "\x{1F42}",
	"\x{1F4B}" => "\x{1F43}",
	"\x{1F4C}" => "\x{1F44}",
	"\x{1F4D}" => "\x{1F45}",
	"\x{1F50}" => "\x{03C5}\x{0313}",
	"\x{1F52}" => "\x{03C5}\x{0313}\x{0300}",
	"\x{1F54}" => "\x{03C5}\x{0313}\x{0301}",
	"\x{1F56}" => "\x{03C5}\x{0313}\x{0342}",
	"\x{1F59}" => "\x{1F51}",
	"\x{1F5B}" => "\x{1F53}",
	"\x{1F5D}" => "\x{1F55}",
	"\x{1F5F}" => "\x{1F57}",
	"\x{1F68}" => "\x{1F60}",
	"\x{1F69}" => "\x{1F61}",
	"\x{1F6A}" => "\x{1F62}",
	"\x{1F6B}" => "\x{1F63}",
	"\x{1F6C}" => "\x{1F64}",
	"\x{1F6D}" => "\x{1F65}",
	"\x{1F6E}" => "\x{1F66}",
	"\x{1F6F}" => "\x{1F67}",
	"\x{1F80}" => "\x{1F00}\x{03B9}",
	"\x{1F81}" => "\x{1F01}\x{03B9}",
	"\x{1F82}" => "\x{1F02}\x{03B9}",
	"\x{1F83}" => "\x{1F03}\x{03B9}",
	"\x{1F84}" => "\x{1F04}\x{03B9}",
	"\x{1F85}" => "\x{1F05}\x{03B9}",
	"\x{1F86}" => "\x{1F06}\x{03B9}",
	"\x{1F87}" => "\x{1F07}\x{03B9}",
	"\x{1F88}" => "\x{1F00}\x{03B9}",
	"\x{1F89}" => "\x{1F01}\x{03B9}",
	"\x{1F8A}" => "\x{1F02}\x{03B9}",
	"\x{1F8B}" => "\x{1F03}\x{03B9}",
	"\x{1F8C}" => "\x{1F04}\x{03B9}",
	"\x{1F8D}" => "\x{1F05}\x{03B9}",
	"\x{1F8E}" => "\x{1F06}\x{03B9}",
	"\x{1F8F}" => "\x{1F07}\x{03B9}",
	"\x{1F90}" => "\x{1F20}\x{03B9}",
	"\x{1F91}" => "\x{1F21}\x{03B9}",
	"\x{1F92}" => "\x{1F22}\x{03B9}",
	"\x{1F93}" => "\x{1F23}\x{03B9}",
	"\x{1F94}" => "\x{1F24}\x{03B9}",
	"\x{1F95}" => "\x{1F25}\x{03B9}",
	"\x{1F96}" => "\x{1F26}\x{03B9}",
	"\x{1F97}" => "\x{1F27}\x{03B9}",
	"\x{1F98}" => "\x{1F20}\x{03B9}",
	"\x{1F99}" => "\x{1F21}\x{03B9}",
	"\x{1F9A}" => "\x{1F22}\x{03B9}",
	"\x{1F9B}" => "\x{1F23}\x{03B9}",
	"\x{1F9C}" => "\x{1F24}\x{03B9}",
	"\x{1F9D}" => "\x{1F25}\x{03B9}",
	"\x{1F9E}" => "\x{1F26}\x{03B9}",
	"\x{1F9F}" => "\x{1F27}\x{03B9}",
	"\x{1FA0}" => "\x{1F60}\x{03B9}",
	"\x{1FA1}" => "\x{1F61}\x{03B9}",
	"\x{1FA2}" => "\x{1F62}\x{03B9}",
	"\x{1FA3}" => "\x{1F63}\x{03B9}",
	"\x{1FA4}" => "\x{1F64}\x{03B9}",
	"\x{1FA5}" => "\x{1F65}\x{03B9}",
	"\x{1FA6}" => "\x{1F66}\x{03B9}",
	"\x{1FA7}" => "\x{1F67}\x{03B9}",
	"\x{1FA8}" => "\x{1F60}\x{03B9}",
	"\x{1FA9}" => "\x{1F61}\x{03B9}",
	"\x{1FAA}" => "\x{1F62}\x{03B9}",
	"\x{1FAB}" => "\x{1F63}\x{03B9}",
	"\x{1FAC}" => "\x{1F64}\x{03B9}",
	"\x{1FAD}" => "\x{1F65}\x{03B9}",
	"\x{1FAE}" => "\x{1F66}\x{03B9}",
	"\x{1FAF}" => "\x{1F67}\x{03B9}",
	"\x{1FB2}" => "\x{1F70}\x{03B9}",
	"\x{1FB3}" => "\x{03B1}\x{03B9}",
	"\x{1FB4}" => "\x{03AC}\x{03B9}",
	"\x{1FB6}" => "\x{03B1}\x{0342}",
	"\x{1FB7}" => "\x{03B1}\x{0342}\x{03B9}",
	"\x{1FB8}" => "\x{1FB0}",
	"\x{1FB9}" => "\x{1FB1}",
	"\x{1FBA}" => "\x{1F70}",
	"\x{1FBB}" => "\x{1F71}",
	"\x{1FBC}" => "\x{03B1}\x{03B9}",
	"\x{1FBE}" => "\x{03B9}",
	"\x{1FC2}" => "\x{1F74}\x{03B9}",
	"\x{1FC3}" => "\x{03B7}\x{03B9}",
	"\x{1FC4}" => "\x{03AE}\x{03B9}",
	"\x{1FC6}" => "\x{03B7}\x{0342}",
	"\x{1FC7}" => "\x{03B7}\x{0342}\x{03B9}",
	"\x{1FC8}" => "\x{1F72}",
	"\x{1FC9}" => "\x{1F73}",
	"\x{1FCA}" => "\x{1F74}",
	"\x{1FCB}" => "\x{1F75}",
	"\x{1FCC}" => "\x{03B7}\x{03B9}",
	"\x{1FD2}" => "\x{03B9}\x{0308}\x{0300}",
	"\x{1FD3}" => "\x{03B9}\x{0308}\x{0301}",
	"\x{1FD6}" => "\x{03B9}\x{0342}",
	"\x{1FD7}" => "\x{03B9}\x{0308}\x{0342}",
	"\x{1FD8}" => "\x{1FD0}",
	"\x{1FD9}" => "\x{1FD1}",
	"\x{1FDA}" => "\x{1F76}",
	"\x{1FDB}" => "\x{1F77}",
	"\x{1FE2}" => "\x{03C5}\x{0308}\x{0300}",
	"\x{1FE3}" => "\x{03C5}\x{0308}\x{0301}",
	"\x{1FE4}" => "\x{03C1}\x{0313}",
	"\x{1FE6}" => "\x{03C5}\x{0342}",
	"\x{1FE7}" => "\x{03C5}\x{0308}\x{0342}",
	"\x{1FE8}" => "\x{1FE0}",
	"\x{1FE9}" => "\x{1FE1}",
	"\x{1FEA}" => "\x{1F7A}",
	"\x{1FEB}" => "\x{1F7B}",
	"\x{1FEC}" => "\x{1FE5}",
	"\x{1FF2}" => "\x{1F7C}\x{03B9}",
	"\x{1FF3}" => "\x{03C9}\x{03B9}",
	"\x{1FF4}" => "\x{03CE}\x{03B9}",
	"\x{1FF6}" => "\x{03C9}\x{0342}",
	"\x{1FF7}" => "\x{03C9}\x{0342}\x{03B9}",
	"\x{1FF8}" => "\x{1F78}",
	"\x{1FF9}" => "\x{1F79}",
	"\x{1FFA}" => "\x{1F7C}",
	"\x{1FFB}" => "\x{1F7D}",
	"\x{1FFC}" => "\x{03C9}\x{03B9}",
	"\x{20A8}" => "\x{0072}\x{0073}",
	"\x{2102}" => "\x{0063}",
	"\x{2103}" => "\x{00B0}\x{0063}",
	"\x{2107}" => "\x{025B}",
	"\x{2109}" => "\x{00B0}\x{0066}",
	"\x{210B}" => "\x{0068}",
	"\x{210C}" => "\x{0068}",
	"\x{210D}" => "\x{0068}",
	"\x{2110}" => "\x{0069}",
	"\x{2111}" => "\x{0069}",
	"\x{2112}" => "\x{006C}",
	"\x{2115}" => "\x{006E}",
	"\x{2116}" => "\x{006E}\x{006F}",
	"\x{2119}" => "\x{0070}",
	"\x{211A}" => "\x{0071}",
	"\x{211B}" => "\x{0072}",
	"\x{211C}" => "\x{0072}",
	"\x{211D}" => "\x{0072}",
	"\x{2120}" => "\x{0073}\x{006D}",
	"\x{2121}" => "\x{0074}\x{0065}\x{006C}",
	"\x{2122}" => "\x{0074}\x{006D}",
	"\x{2124}" => "\x{007A}",
	"\x{2126}" => "\x{03C9}",
	"\x{2128}" => "\x{007A}",
	"\x{212A}" => "\x{006B}",
	"\x{212B}" => "\x{00E5}",
	"\x{212C}" => "\x{0062}",
	"\x{212D}" => "\x{0063}",
	"\x{2130}" => "\x{0065}",
	"\x{2131}" => "\x{0066}",
	"\x{2133}" => "\x{006D}",
	"\x{213E}" => "\x{03B3}",
	"\x{213F}" => "\x{03C0}",
	"\x{2145}" => "\x{0064}",
	"\x{2160}" => "\x{2170}",
	"\x{2161}" => "\x{2171}",
	"\x{2162}" => "\x{2172}",
	"\x{2163}" => "\x{2173}",
	"\x{2164}" => "\x{2174}",
	"\x{2165}" => "\x{2175}",
	"\x{2166}" => "\x{2176}",
	"\x{2167}" => "\x{2177}",
	"\x{2168}" => "\x{2178}",
	"\x{2169}" => "\x{2179}",
	"\x{216A}" => "\x{217A}",
	"\x{216B}" => "\x{217B}",
	"\x{216C}" => "\x{217C}",
	"\x{216D}" => "\x{217D}",
	"\x{216E}" => "\x{217E}",
	"\x{216F}" => "\x{217F}",
	"\x{24B6}" => "\x{24D0}",
	"\x{24B7}" => "\x{24D1}",
	"\x{24B8}" => "\x{24D2}",
	"\x{24B9}" => "\x{24D3}",
	"\x{24BA}" => "\x{24D4}",
	"\x{24BB}" => "\x{24D5}",
	"\x{24BC}" => "\x{24D6}",
	"\x{24BD}" => "\x{24D7}",
	"\x{24BE}" => "\x{24D8}",
	"\x{24BF}" => "\x{24D9}",
	"\x{24C0}" => "\x{24DA}",
	"\x{24C1}" => "\x{24DB}",
	"\x{24C2}" => "\x{24DC}",
	"\x{24C3}" => "\x{24DD}",
	"\x{24C4}" => "\x{24DE}",
	"\x{24C5}" => "\x{24DF}",
	"\x{24C6}" => "\x{24E0}",
	"\x{24C7}" => "\x{24E1}",
	"\x{24C8}" => "\x{24E2}",
	"\x{24C9}" => "\x{24E3}",
	"\x{24CA}" => "\x{24E4}",
	"\x{24CB}" => "\x{24E5}",
	"\x{24CC}" => "\x{24E6}",
	"\x{24CD}" => "\x{24E7}",
	"\x{24CE}" => "\x{24E8}",
	"\x{24CF}" => "\x{24E9}",
	"\x{3371}" => "\x{0068}\x{0070}\x{0061}",
	"\x{3373}" => "\x{0061}\x{0075}",
	"\x{3375}" => "\x{006F}\x{0076}",
	"\x{3380}" => "\x{0070}\x{0061}",
	"\x{3381}" => "\x{006E}\x{0061}",
	"\x{3382}" => "\x{03BC}\x{0061}",
	"\x{3383}" => "\x{006D}\x{0061}",
	"\x{3384}" => "\x{006B}\x{0061}",
	"\x{3385}" => "\x{006B}\x{0062}",
	"\x{3386}" => "\x{006D}\x{0062}",
	"\x{3387}" => "\x{0067}\x{0062}",
	"\x{338A}" => "\x{0070}\x{0066}",
	"\x{338B}" => "\x{006E}\x{0066}",
	"\x{338C}" => "\x{03BC}\x{0066}",
	"\x{3390}" => "\x{0068}\x{007A}",
	"\x{3391}" => "\x{006B}\x{0068}\x{007A}",
	"\x{3392}" => "\x{006D}\x{0068}\x{007A}",
	"\x{3393}" => "\x{0067}\x{0068}\x{007A}",
	"\x{3394}" => "\x{0074}\x{0068}\x{007A}",
	"\x{33A9}" => "\x{0070}\x{0061}",
	"\x{33AA}" => "\x{006B}\x{0070}\x{0061}",
	"\x{33AB}" => "\x{006D}\x{0070}\x{0061}",
	"\x{33AC}" => "\x{0067}\x{0070}\x{0061}",
	"\x{33B4}" => "\x{0070}\x{0076}",
	"\x{33B5}" => "\x{006E}\x{0076}",
	"\x{33B6}" => "\x{03BC}\x{0076}",
	"\x{33B7}" => "\x{006D}\x{0076}",
	"\x{33B8}" => "\x{006B}\x{0076}",
	"\x{33B9}" => "\x{006D}\x{0076}",
	"\x{33BA}" => "\x{0070}\x{0077}",
	"\x{33BB}" => "\x{006E}\x{0077}",
	"\x{33BC}" => "\x{03BC}\x{0077}",
	"\x{33BD}" => "\x{006D}\x{0077}",
	"\x{33BE}" => "\x{006B}\x{0077}",
	"\x{33BF}" => "\x{006D}\x{0077}",
	"\x{33C0}" => "\x{006B}\x{03C9}",
	"\x{33C1}" => "\x{006D}\x{03C9}",
	"\x{33C3}" => "\x{0062}\x{0071}",
	"\x{33C6}" => "\x{0063}\x{2215}\x{006B}\x{0067}",
	"\x{33C7}" => "\x{0063}\x{006F}\x{002E}",
	"\x{33C8}" => "\x{0064}\x{0062}",
	"\x{33C9}" => "\x{0067}\x{0079}",
	"\x{33CB}" => "\x{0068}\x{0070}",
	"\x{33CD}" => "\x{006B}\x{006B}",
	"\x{33CE}" => "\x{006B}\x{006D}",
	"\x{33D7}" => "\x{0070}\x{0068}",
	"\x{33D9}" => "\x{0070}\x{0070}\x{006D}",
	"\x{33DA}" => "\x{0070}\x{0072}",
	"\x{33DC}" => "\x{0073}\x{0076}",
	"\x{33DD}" => "\x{0077}\x{0062}",
	"\x{FB00}" => "\x{0066}\x{0066}",
	"\x{FB01}" => "\x{0066}\x{0069}",
	"\x{FB02}" => "\x{0066}\x{006C}",
	"\x{FB03}" => "\x{0066}\x{0066}\x{0069}",
	"\x{FB04}" => "\x{0066}\x{0066}\x{006C}",
	"\x{FB05}" => "\x{0073}\x{0074}",
	"\x{FB06}" => "\x{0073}\x{0074}",
	"\x{FB13}" => "\x{0574}\x{0576}",
	"\x{FB14}" => "\x{0574}\x{0565}",
	"\x{FB15}" => "\x{0574}\x{056B}",
	"\x{FB16}" => "\x{057E}\x{0576}",
	"\x{FB17}" => "\x{0574}\x{056D}",
	"\x{FF21}" => "\x{FF41}",
	"\x{FF22}" => "\x{FF42}",
	"\x{FF23}" => "\x{FF43}",
	"\x{FF24}" => "\x{FF44}",
	"\x{FF25}" => "\x{FF45}",
	"\x{FF26}" => "\x{FF46}",
	"\x{FF27}" => "\x{FF47}",
	"\x{FF28}" => "\x{FF48}",
	"\x{FF29}" => "\x{FF49}",
	"\x{FF2A}" => "\x{FF4A}",
	"\x{FF2B}" => "\x{FF4B}",
	"\x{FF2C}" => "\x{FF4C}",
	"\x{FF2D}" => "\x{FF4D}",
	"\x{FF2E}" => "\x{FF4E}",
	"\x{FF2F}" => "\x{FF4F}",
	"\x{FF30}" => "\x{FF50}",
	"\x{FF31}" => "\x{FF51}",
	"\x{FF32}" => "\x{FF52}",
	"\x{FF33}" => "\x{FF53}",
	"\x{FF34}" => "\x{FF54}",
	"\x{FF35}" => "\x{FF55}",
	"\x{FF36}" => "\x{FF56}",
	"\x{FF37}" => "\x{FF57}",
	"\x{FF38}" => "\x{FF58}",
	"\x{FF39}" => "\x{FF59}",
	"\x{FF3A}" => "\x{FF5A}",
	"\x{10400}" => "\x{10428}",
	"\x{10401}" => "\x{10429}",
	"\x{10402}" => "\x{1042A}",
	"\x{10403}" => "\x{1042B}",
	"\x{10404}" => "\x{1042C}",
	"\x{10405}" => "\x{1042D}",
	"\x{10406}" => "\x{1042E}",
	"\x{10407}" => "\x{1042F}",
	"\x{10408}" => "\x{10430}",
	"\x{10409}" => "\x{10431}",
	"\x{1040A}" => "\x{10432}",
	"\x{1040B}" => "\x{10433}",
	"\x{1040C}" => "\x{10434}",
	"\x{1040D}" => "\x{10435}",
	"\x{1040E}" => "\x{10436}",
	"\x{1040F}" => "\x{10437}",
	"\x{10410}" => "\x{10438}",
	"\x{10411}" => "\x{10439}",
	"\x{10412}" => "\x{1043A}",
	"\x{10413}" => "\x{1043B}",
	"\x{10414}" => "\x{1043C}",
	"\x{10415}" => "\x{1043D}",
	"\x{10416}" => "\x{1043E}",
	"\x{10417}" => "\x{1043F}",
	"\x{10418}" => "\x{10440}",
	"\x{10419}" => "\x{10441}",
	"\x{1041A}" => "\x{10442}",
	"\x{1041B}" => "\x{10443}",
	"\x{1041C}" => "\x{10444}",
	"\x{1041D}" => "\x{10445}",
	"\x{1041E}" => "\x{10446}",
	"\x{1041F}" => "\x{10447}",
	"\x{10420}" => "\x{10448}",
	"\x{10421}" => "\x{10449}",
	"\x{10422}" => "\x{1044A}",
	"\x{10423}" => "\x{1044B}",
	"\x{10424}" => "\x{1044C}",
	"\x{10425}" => "\x{1044D}",
	"\x{1D400}" => "\x{0061}",
	"\x{1D401}" => "\x{0062}",
	"\x{1D402}" => "\x{0063}",
	"\x{1D403}" => "\x{0064}",
	"\x{1D404}" => "\x{0065}",
	"\x{1D405}" => "\x{0066}",
	"\x{1D406}" => "\x{0067}",
	"\x{1D407}" => "\x{0068}",
	"\x{1D408}" => "\x{0069}",
	"\x{1D409}" => "\x{006A}",
	"\x{1D40A}" => "\x{006B}",
	"\x{1D40B}" => "\x{006C}",
	"\x{1D40C}" => "\x{006D}",
	"\x{1D40D}" => "\x{006E}",
	"\x{1D40E}" => "\x{006F}",
	"\x{1D40F}" => "\x{0070}",
	"\x{1D410}" => "\x{0071}",
	"\x{1D411}" => "\x{0072}",
	"\x{1D412}" => "\x{0073}",
	"\x{1D413}" => "\x{0074}",
	"\x{1D414}" => "\x{0075}",
	"\x{1D415}" => "\x{0076}",
	"\x{1D416}" => "\x{0077}",
	"\x{1D417}" => "\x{0078}",
	"\x{1D418}" => "\x{0079}",
	"\x{1D419}" => "\x{007A}",
	"\x{1D434}" => "\x{0061}",
	"\x{1D435}" => "\x{0062}",
	"\x{1D436}" => "\x{0063}",
	"\x{1D437}" => "\x{0064}",
	"\x{1D438}" => "\x{0065}",
	"\x{1D439}" => "\x{0066}",
	"\x{1D43A}" => "\x{0067}",
	"\x{1D43B}" => "\x{0068}",
	"\x{1D43C}" => "\x{0069}",
	"\x{1D43D}" => "\x{006A}",
	"\x{1D43E}" => "\x{006B}",
	"\x{1D43F}" => "\x{006C}",
	"\x{1D440}" => "\x{006D}",
	"\x{1D441}" => "\x{006E}",
	"\x{1D442}" => "\x{006F}",
	"\x{1D443}" => "\x{0070}",
	"\x{1D444}" => "\x{0071}",
	"\x{1D445}" => "\x{0072}",
	"\x{1D446}" => "\x{0073}",
	"\x{1D447}" => "\x{0074}",
	"\x{1D448}" => "\x{0075}",
	"\x{1D449}" => "\x{0076}",
	"\x{1D44A}" => "\x{0077}",
	"\x{1D44B}" => "\x{0078}",
	"\x{1D44C}" => "\x{0079}",
	"\x{1D44D}" => "\x{007A}",
	"\x{1D468}" => "\x{0061}",
	"\x{1D469}" => "\x{0062}",
	"\x{1D46A}" => "\x{0063}",
	"\x{1D46B}" => "\x{0064}",
	"\x{1D46C}" => "\x{0065}",
	"\x{1D46D}" => "\x{0066}",
	"\x{1D46E}" => "\x{0067}",
	"\x{1D46F}" => "\x{0068}",
	"\x{1D470}" => "\x{0069}",
	"\x{1D471}" => "\x{006A}",
	"\x{1D472}" => "\x{006B}",
	"\x{1D473}" => "\x{006C}",
	"\x{1D474}" => "\x{006D}",
	"\x{1D475}" => "\x{006E}",
	"\x{1D476}" => "\x{006F}",
	"\x{1D477}" => "\x{0070}",
	"\x{1D478}" => "\x{0071}",
	"\x{1D479}" => "\x{0072}",
	"\x{1D47A}" => "\x{0073}",
	"\x{1D47B}" => "\x{0074}",
	"\x{1D47C}" => "\x{0075}",
	"\x{1D47D}" => "\x{0076}",
	"\x{1D47E}" => "\x{0077}",
	"\x{1D47F}" => "\x{0078}",
	"\x{1D480}" => "\x{0079}",
	"\x{1D481}" => "\x{007A}",
	"\x{1D49C}" => "\x{0061}",
	"\x{1D49E}" => "\x{0063}",
	"\x{1D49F}" => "\x{0064}",
	"\x{1D4A2}" => "\x{0067}",
	"\x{1D4A5}" => "\x{006A}",
	"\x{1D4A6}" => "\x{006B}",
	"\x{1D4A9}" => "\x{006E}",
	"\x{1D4AA}" => "\x{006F}",
	"\x{1D4AB}" => "\x{0070}",
	"\x{1D4AC}" => "\x{0071}",
	"\x{1D4AE}" => "\x{0073}",
	"\x{1D4AF}" => "\x{0074}",
	"\x{1D4B0}" => "\x{0075}",
	"\x{1D4B1}" => "\x{0076}",
	"\x{1D4B2}" => "\x{0077}",
	"\x{1D4B3}" => "\x{0078}",
	"\x{1D4B4}" => "\x{0079}",
	"\x{1D4B5}" => "\x{007A}",
	"\x{1D4D0}" => "\x{0061}",
	"\x{1D4D1}" => "\x{0062}",
	"\x{1D4D2}" => "\x{0063}",
	"\x{1D4D3}" => "\x{0064}",
	"\x{1D4D4}" => "\x{0065}",
	"\x{1D4D5}" => "\x{0066}",
	"\x{1D4D6}" => "\x{0067}",
	"\x{1D4D7}" => "\x{0068}",
	"\x{1D4D8}" => "\x{0069}",
	"\x{1D4D9}" => "\x{006A}",
	"\x{1D4DA}" => "\x{006B}",
	"\x{1D4DB}" => "\x{006C}",
	"\x{1D4DC}" => "\x{006D}",
	"\x{1D4DD}" => "\x{006E}",
	"\x{1D4DE}" => "\x{006F}",
	"\x{1D4DF}" => "\x{0070}",
	"\x{1D4E0}" => "\x{0071}",
	"\x{1D4E1}" => "\x{0072}",
	"\x{1D4E2}" => "\x{0073}",
	"\x{1D4E3}" => "\x{0074}",
	"\x{1D4E4}" => "\x{0075}",
	"\x{1D4E5}" => "\x{0076}",
	"\x{1D4E6}" => "\x{0077}",
	"\x{1D4E7}" => "\x{0078}",
	"\x{1D4E8}" => "\x{0079}",
	"\x{1D4E9}" => "\x{007A}",
	"\x{1D504}" => "\x{0061}",
	"\x{1D505}" => "\x{0062}",
	"\x{1D507}" => "\x{0064}",
	"\x{1D508}" => "\x{0065}",
	"\x{1D509}" => "\x{0066}",
	"\x{1D50A}" => "\x{0067}",
	"\x{1D50D}" => "\x{006A}",
	"\x{1D50E}" => "\x{006B}",
	"\x{1D50F}" => "\x{006C}",
	"\x{1D510}" => "\x{006D}",
	"\x{1D511}" => "\x{006E}",
	"\x{1D512}" => "\x{006F}",
	"\x{1D513}" => "\x{0070}",
	"\x{1D514}" => "\x{0071}",
	"\x{1D516}" => "\x{0073}",
	"\x{1D517}" => "\x{0074}",
	"\x{1D518}" => "\x{0075}",
	"\x{1D519}" => "\x{0076}",
	"\x{1D51A}" => "\x{0077}",
	"\x{1D51B}" => "\x{0078}",
	"\x{1D51C}" => "\x{0079}",
	"\x{1D538}" => "\x{0061}",
	"\x{1D539}" => "\x{0062}",
	"\x{1D53B}" => "\x{0064}",
	"\x{1D53C}" => "\x{0065}",
	"\x{1D53D}" => "\x{0066}",
	"\x{1D53E}" => "\x{0067}",
	"\x{1D540}" => "\x{0069}",
	"\x{1D541}" => "\x{006A}",
	"\x{1D542}" => "\x{006B}",
	"\x{1D543}" => "\x{006C}",
	"\x{1D544}" => "\x{006D}",
	"\x{1D546}" => "\x{006F}",
	"\x{1D54A}" => "\x{0073}",
	"\x{1D54B}" => "\x{0074}",
	"\x{1D54C}" => "\x{0075}",
	"\x{1D54D}" => "\x{0076}",
	"\x{1D54E}" => "\x{0077}",
	"\x{1D54F}" => "\x{0078}",
	"\x{1D550}" => "\x{0079}",
	"\x{1D56C}" => "\x{0061}",
	"\x{1D56D}" => "\x{0062}",
	"\x{1D56E}" => "\x{0063}",
	"\x{1D56F}" => "\x{0064}",
	"\x{1D570}" => "\x{0065}",
	"\x{1D571}" => "\x{0066}",
	"\x{1D572}" => "\x{0067}",
	"\x{1D573}" => "\x{0068}",
	"\x{1D574}" => "\x{0069}",
	"\x{1D575}" => "\x{006A}",
	"\x{1D576}" => "\x{006B}",
	"\x{1D577}" => "\x{006C}",
	"\x{1D578}" => "\x{006D}",
	"\x{1D579}" => "\x{006E}",
	"\x{1D57A}" => "\x{006F}",
	"\x{1D57B}" => "\x{0070}",
	"\x{1D57C}" => "\x{0071}",
	"\x{1D57D}" => "\x{0072}",
	"\x{1D57E}" => "\x{0073}",
	"\x{1D57F}" => "\x{0074}",
	"\x{1D580}" => "\x{0075}",
	"\x{1D581}" => "\x{0076}",
	"\x{1D582}" => "\x{0077}",
	"\x{1D583}" => "\x{0078}",
	"\x{1D584}" => "\x{0079}",
	"\x{1D585}" => "\x{007A}",
	"\x{1D5A0}" => "\x{0061}",
	"\x{1D5A1}" => "\x{0062}",
	"\x{1D5A2}" => "\x{0063}",
	"\x{1D5A3}" => "\x{0064}",
	"\x{1D5A4}" => "\x{0065}",
	"\x{1D5A5}" => "\x{0066}",
	"\x{1D5A6}" => "\x{0067}",
	"\x{1D5A7}" => "\x{0068}",
	"\x{1D5A8}" => "\x{0069}",
	"\x{1D5A9}" => "\x{006A}",
	"\x{1D5AA}" => "\x{006B}",
	"\x{1D5AB}" => "\x{006C}",
	"\x{1D5AC}" => "\x{006D}",
	"\x{1D5AD}" => "\x{006E}",
	"\x{1D5AE}" => "\x{006F}",
	"\x{1D5AF}" => "\x{0070}",
	"\x{1D5B0}" => "\x{0071}",
	"\x{1D5B1}" => "\x{0072}",
	"\x{1D5B2}" => "\x{0073}",
	"\x{1D5B3}" => "\x{0074}",
	"\x{1D5B4}" => "\x{0075}",
	"\x{1D5B5}" => "\x{0076}",
	"\x{1D5B6}" => "\x{0077}",
	"\x{1D5B7}" => "\x{0078}",
	"\x{1D5B8}" => "\x{0079}",
	"\x{1D5B9}" => "\x{007A}",
	"\x{1D5D4}" => "\x{0061}",
	"\x{1D5D5}" => "\x{0062}",
	"\x{1D5D6}" => "\x{0063}",
	"\x{1D5D7}" => "\x{0064}",
	"\x{1D5D8}" => "\x{0065}",
	"\x{1D5D9}" => "\x{0066}",
	"\x{1D5DA}" => "\x{0067}",
	"\x{1D5DB}" => "\x{0068}",
	"\x{1D5DC}" => "\x{0069}",
	"\x{1D5DD}" => "\x{006A}",
	"\x{1D5DE}" => "\x{006B}",
	"\x{1D5DF}" => "\x{006C}",
	"\x{1D5E0}" => "\x{006D}",
	"\x{1D5E1}" => "\x{006E}",
	"\x{1D5E2}" => "\x{006F}",
	"\x{1D5E3}" => "\x{0070}",
	"\x{1D5E4}" => "\x{0071}",
	"\x{1D5E5}" => "\x{0072}",
	"\x{1D5E6}" => "\x{0073}",
	"\x{1D5E7}" => "\x{0074}",
	"\x{1D5E8}" => "\x{0075}",
	"\x{1D5E9}" => "\x{0076}",
	"\x{1D5EA}" => "\x{0077}",
	"\x{1D5EB}" => "\x{0078}",
	"\x{1D5EC}" => "\x{0079}",
	"\x{1D5ED}" => "\x{007A}",
	"\x{1D608}" => "\x{0061}",
	"\x{1D609}" => "\x{0062}",
	"\x{1D60A}" => "\x{0063}",
	"\x{1D60B}" => "\x{0064}",
	"\x{1D60C}" => "\x{0065}",
	"\x{1D60D}" => "\x{0066}",
	"\x{1D60E}" => "\x{0067}",
	"\x{1D60F}" => "\x{0068}",
	"\x{1D610}" => "\x{0069}",
	"\x{1D611}" => "\x{006A}",
	"\x{1D612}" => "\x{006B}",
	"\x{1D613}" => "\x{006C}",
	"\x{1D614}" => "\x{006D}",
	"\x{1D615}" => "\x{006E}",
	"\x{1D616}" => "\x{006F}",
	"\x{1D617}" => "\x{0070}",
	"\x{1D618}" => "\x{0071}",
	"\x{1D619}" => "\x{0072}",
	"\x{1D61A}" => "\x{0073}",
	"\x{1D61B}" => "\x{0074}",
	"\x{1D61C}" => "\x{0075}",
	"\x{1D61D}" => "\x{0076}",
	"\x{1D61E}" => "\x{0077}",
	"\x{1D61F}" => "\x{0078}",
	"\x{1D620}" => "\x{0079}",
	"\x{1D621}" => "\x{007A}",
	"\x{1D63C}" => "\x{0061}",
	"\x{1D63D}" => "\x{0062}",
	"\x{1D63E}" => "\x{0063}",
	"\x{1D63F}" => "\x{0064}",
	"\x{1D640}" => "\x{0065}",
	"\x{1D641}" => "\x{0066}",
	"\x{1D642}" => "\x{0067}",
	"\x{1D643}" => "\x{0068}",
	"\x{1D644}" => "\x{0069}",
	"\x{1D645}" => "\x{006A}",
	"\x{1D646}" => "\x{006B}",
	"\x{1D647}" => "\x{006C}",
	"\x{1D648}" => "\x{006D}",
	"\x{1D649}" => "\x{006E}",
	"\x{1D64A}" => "\x{006F}",
	"\x{1D64B}" => "\x{0070}",
	"\x{1D64C}" => "\x{0071}",
	"\x{1D64D}" => "\x{0072}",
	"\x{1D64E}" => "\x{0073}",
	"\x{1D64F}" => "\x{0074}",
	"\x{1D650}" => "\x{0075}",
	"\x{1D651}" => "\x{0076}",
	"\x{1D652}" => "\x{0077}",
	"\x{1D653}" => "\x{0078}",
	"\x{1D654}" => "\x{0079}",
	"\x{1D655}" => "\x{007A}",
	"\x{1D670}" => "\x{0061}",
	"\x{1D671}" => "\x{0062}",
	"\x{1D672}" => "\x{0063}",
	"\x{1D673}" => "\x{0064}",
	"\x{1D674}" => "\x{0065}",
	"\x{1D675}" => "\x{0066}",
	"\x{1D676}" => "\x{0067}",
	"\x{1D677}" => "\x{0068}",
	"\x{1D678}" => "\x{0069}",
	"\x{1D679}" => "\x{006A}",
	"\x{1D67A}" => "\x{006B}",
	"\x{1D67B}" => "\x{006C}",
	"\x{1D67C}" => "\x{006D}",
	"\x{1D67D}" => "\x{006E}",
	"\x{1D67E}" => "\x{006F}",
	"\x{1D67F}" => "\x{0070}",
	"\x{1D680}" => "\x{0071}",
	"\x{1D681}" => "\x{0072}",
	"\x{1D682}" => "\x{0073}",
	"\x{1D683}" => "\x{0074}",
	"\x{1D684}" => "\x{0075}",
	"\x{1D685}" => "\x{0076}",
	"\x{1D686}" => "\x{0077}",
	"\x{1D687}" => "\x{0078}",
	"\x{1D688}" => "\x{0079}",
	"\x{1D689}" => "\x{007A}",
	"\x{1D6A8}" => "\x{03B1}",
	"\x{1D6A9}" => "\x{03B2}",
	"\x{1D6AA}" => "\x{03B3}",
	"\x{1D6AB}" => "\x{03B4}",
	"\x{1D6AC}" => "\x{03B5}",
	"\x{1D6AD}" => "\x{03B6}",
	"\x{1D6AE}" => "\x{03B7}",
	"\x{1D6AF}" => "\x{03B8}",
	"\x{1D6B0}" => "\x{03B9}",
	"\x{1D6B1}" => "\x{03BA}",
	"\x{1D6B2}" => "\x{03BB}",
	"\x{1D6B3}" => "\x{03BC}",
	"\x{1D6B4}" => "\x{03BD}",
	"\x{1D6B5}" => "\x{03BE}",
	"\x{1D6B6}" => "\x{03BF}",
	"\x{1D6B7}" => "\x{03C0}",
	"\x{1D6B8}" => "\x{03C1}",
	"\x{1D6B9}" => "\x{03B8}",
	"\x{1D6BA}" => "\x{03C3}",
	"\x{1D6BB}" => "\x{03C4}",
	"\x{1D6BC}" => "\x{03C5}",
	"\x{1D6BD}" => "\x{03C6}",
	"\x{1D6BE}" => "\x{03C7}",
	"\x{1D6BF}" => "\x{03C8}",
	"\x{1D6C0}" => "\x{03C9}",
	"\x{1D6D3}" => "\x{03C3}",
	"\x{1D6E2}" => "\x{03B1}",
	"\x{1D6E3}" => "\x{03B2}",
	"\x{1D6E4}" => "\x{03B3}",
	"\x{1D6E5}" => "\x{03B4}",
	"\x{1D6E6}" => "\x{03B5}",
	"\x{1D6E7}" => "\x{03B6}",
	"\x{1D6E8}" => "\x{03B7}",
	"\x{1D6E9}" => "\x{03B8}",
	"\x{1D6EA}" => "\x{03B9}",
	"\x{1D6EB}" => "\x{03BA}",
	"\x{1D6EC}" => "\x{03BB}",
	"\x{1D6ED}" => "\x{03BC}",
	"\x{1D6EE}" => "\x{03BD}",
	"\x{1D6EF}" => "\x{03BE}",
	"\x{1D6F0}" => "\x{03BF}",
	"\x{1D6F1}" => "\x{03C0}",
	"\x{1D6F2}" => "\x{03C1}",
	"\x{1D6F3}" => "\x{03B8}",
	"\x{1D6F4}" => "\x{03C3}",
	"\x{1D6F5}" => "\x{03C4}",
	"\x{1D6F6}" => "\x{03C5}",
	"\x{1D6F7}" => "\x{03C6}",
	"\x{1D6F8}" => "\x{03C7}",
	"\x{1D6F9}" => "\x{03C8}",
	"\x{1D6FA}" => "\x{03C9}",
	"\x{1D70D}" => "\x{03C3}",
	"\x{1D71C}" => "\x{03B1}",
	"\x{1D71D}" => "\x{03B2}",
	"\x{1D71E}" => "\x{03B3}",
	"\x{1D71F}" => "\x{03B4}",
	"\x{1D720}" => "\x{03B5}",
	"\x{1D721}" => "\x{03B6}",
	"\x{1D722}" => "\x{03B7}",
	"\x{1D723}" => "\x{03B8}",
	"\x{1D724}" => "\x{03B9}",
	"\x{1D725}" => "\x{03BA}",
	"\x{1D726}" => "\x{03BB}",
	"\x{1D727}" => "\x{03BC}",
	"\x{1D728}" => "\x{03BD}",
	"\x{1D729}" => "\x{03BE}",
	"\x{1D72A}" => "\x{03BF}",
	"\x{1D72B}" => "\x{03C0}",
	"\x{1D72C}" => "\x{03C1}",
	"\x{1D72D}" => "\x{03B8}",
	"\x{1D72E}" => "\x{03C3}",
	"\x{1D72F}" => "\x{03C4}",
	"\x{1D730}" => "\x{03C5}",
	"\x{1D731}" => "\x{03C6}",
	"\x{1D732}" => "\x{03C7}",
	"\x{1D733}" => "\x{03C8}",
	"\x{1D734}" => "\x{03C9}",
	"\x{1D747}" => "\x{03C3}",
	"\x{1D756}" => "\x{03B1}",
	"\x{1D757}" => "\x{03B2}",
	"\x{1D758}" => "\x{03B3}",
	"\x{1D759}" => "\x{03B4}",
	"\x{1D75A}" => "\x{03B5}",
	"\x{1D75B}" => "\x{03B6}",
	"\x{1D75C}" => "\x{03B7}",
	"\x{1D75D}" => "\x{03B8}",
	"\x{1D75E}" => "\x{03B9}",
	"\x{1D75F}" => "\x{03BA}",
	"\x{1D760}" => "\x{03BB}",
	"\x{1D761}" => "\x{03BC}",
	"\x{1D762}" => "\x{03BD}",
	"\x{1D763}" => "\x{03BE}",
	"\x{1D764}" => "\x{03BF}",
	"\x{1D765}" => "\x{03C0}",
	"\x{1D766}" => "\x{03C1}",
	"\x{1D767}" => "\x{03B8}",
	"\x{1D768}" => "\x{03C3}",
	"\x{1D769}" => "\x{03C4}",
	"\x{1D76A}" => "\x{03C5}",
	"\x{1D76B}" => "\x{03C6}",
	"\x{1D76C}" => "\x{03C7}",
	"\x{1D76D}" => "\x{03C8}",
	"\x{1D76E}" => "\x{03C9}",
	"\x{1D781}" => "\x{03C3}",
	"\x{1D790}" => "\x{03B1}",
	"\x{1D791}" => "\x{03B2}",
	"\x{1D792}" => "\x{03B3}",
	"\x{1D793}" => "\x{03B4}",
	"\x{1D794}" => "\x{03B5}",
	"\x{1D795}" => "\x{03B6}",
	"\x{1D796}" => "\x{03B7}",
	"\x{1D797}" => "\x{03B8}",
	"\x{1D798}" => "\x{03B9}",
	"\x{1D799}" => "\x{03BA}",
	"\x{1D79A}" => "\x{03BB}",
	"\x{1D79B}" => "\x{03BC}",
	"\x{1D79C}" => "\x{03BD}",
	"\x{1D79D}" => "\x{03BE}",
	"\x{1D79E}" => "\x{03BF}",
	"\x{1D79F}" => "\x{03C0}",
	"\x{1D7A0}" => "\x{03C1}",
	"\x{1D7A1}" => "\x{03B8}",
	"\x{1D7A2}" => "\x{03C3}",
	"\x{1D7A3}" => "\x{03C4}",
	"\x{1D7A4}" => "\x{03C5}",
	"\x{1D7A5}" => "\x{03C6}",
	"\x{1D7A6}" => "\x{03C7}",
	"\x{1D7A7}" => "\x{03C8}",
	"\x{1D7A8}" => "\x{03C9}",
	"\x{1D7BB}" => "\x{03C3}",
);
my ($rfc3454_casemap) = map { qr{[$_]} } join('', keys %rfc3454_casemap);

sub canonicalize_value {
	my ($self, $name, $value) = @_;
	my $type = $rfc4519_typemap{$name}
		|| (ref $self && $self->matchingrule_for_attribute($name, 'equality'))
		|| $name;

    return $self->canonicalize_dn($value) 
        if $type eq 'distinguishedNameMatch';

	return $value
		unless $type =~ /^(?:case(?:Exact|Ignore)|numericStringMatch\z|integerMatch\z|telephoneNumberMatch\z)/a;

	# TODO:
	# bit string        https://tools.ietf.org/html/rfc4517#section-3.3.2
	# boolean           https://tools.ietf.org/html/rfc4517#section-3.3.3
	# fax numbers       https://tools.ietf.org/html/rfc4517#section-3.3.11
	# generalized time  https://tools.ietf.org/html/rfc4517#section-3.3.13

	# See RFC 4518 section 2 for a description of the steps below

	# 2.1.  Transcode
	utf8::decode($value) or die "invalid UTF-8 sequence in '$value'\n";

	# 2.2.  Map
	# map to nothing
	$value =~ s/[
		\x{00AD}
		\x{1806}
		\x{034F}
		\x{180B}-\x{180D}
		\x{FE00}-\x{FE0F}
		\x{FFFC}
		\x{200B}
		\x{0000}-\x{0008}
		\x{000E}-\x{001F}
		\x{007F}-\x{0084}
		\x{0086}-\x{009F}
		\x{06DD}
		\x{070F}
		\x{180E}
		\x{200C}-\x{200F}
		\x{202A}-\x{202E}
		\x{2060}-\x{2063}
		\x{206A}-\x{206F}
		\x{FEFF}
		\x{FFF9}-\x{FFFB}
		\x{1D173}-\x{1D17A}
		\x{E0001}
		\x{E0020}-\x{E007F}
	]+//agx;

	# map to space
	$value =~ s/[
		\x{0009}
		\x{000A}
		\x{000B}
		\x{000C}
		\x{000D}
		\x{0085}
		\x{0020}
		\x{00A0}
		\x{1680}
		\x{2000}-\x{200A}
		\x{2028}-\x{2029}
		\x{202F}
		\x{205F}
		\x{3000}
	]+/\x{0020}/agx;

	$value =~ s{($rfc3454_casemap)}{$rfc3454_casemap{$1}}ego
		if $type =~ /^caseIgnore/ || $type eq 'numericStringMatch';

	# 2.3.  Normalize
	$value = NFKC($value);

	# 2.4.  Prohibit
	# FIXME

	# 2.5.  Check bidi
	# (no-op)

	# 2.6.  Insignificant Character Handling
	if($type =~ /^case(?:Exact|Ignore)/) {
		$value =~ s/^ (?!\p{M})//g;
		$value =~ s/ (?!\p{M})\z//g;
#		$value = " $value ";
	} elsif($type eq 'numericStringMatch') {
		$value =~ s/\x{0020}(?!\p{M})//g;
	} elsif($type eq 'integerMatch') {
		$value =~ s/\x{0020}+//g;
		$value =~ s/^(-?)0*([1-9][0-9]*|0)\z/$1$2/a
			or die "invalid integer value '$value'\n";
	} elsif($type eq 'telephoneNumberMatch') {
		$value =~ s/[
			\x{0020}
			\x{002D}
			\x{058A}
			\x{2010}
			\x{2011}
			\x{2212}
			\x{FE63}
			\x{FF0D}
		](?!\p{M})//agx;
	}

	utf8::encode($value);
	return $value;
}

sub canonicalize_dn {
	my ($self, $dn) = @_;

	# A dn is a byte sequence, not a character sequence.
	# Be kind to users who may not be aware of this subtle distinction.
	utf8::encode($dn) if utf8::is_utf8($dn);

	# This is a little quirky: we split the string not on commas but
	# on everything *except* commas. This is necessary because we need
	# escape sequence parsing and can't just match any comma.
	my @rdns = split(/\ *(
		(?:
			[A-Za-z][A-Za-z0-9-]*
			|
			(?:oid\.|OID\.)?[0-9]+(?:\.[0-9]+)*
		)
		\ *=
		(?:\ *
			(?:
				\#(?:[0-9a-fA-F][0-9a-fA-F])+
				|
				(?:[^#,+"\\<>; ]|\\(?:[^0-9a-fA-F]|[0-9a-fA-F][0-9a-fA-F]))
				(?:
					(?:[^,+"\\<>;]|\\(?:[^0-9a-fA-F]|[0-9a-fA-F][0-9a-fA-F]))*
					(?:[^,+"\\<>; ]|\\(?:[^0-9a-fA-F]|[0-9a-fA-F][0-9a-fA-F]))
				)?
				|
				"(?:[^"\\]|\\.)*"
			)
		)?
	)\ */ax, $dn, -1);

	# Sanity checks. There should be empty elements before the first and after
	# the last RDN (because we used the RDNs as separators).
	die "failed to parse dn '$dn'\n" if @rdns < 2;
	die "failed to parse dn '$dn'\n" if shift @rdns ne '';
	die "failed to parse dn '$dn'\n" if pop @rdns ne '';

	foreach(@rdns) {
		if(/^[,+]\z/) {
			# skip the real separators
		} elsif(/^;\z/) {
			# accept LDAPv2 separators but convert them
			$_ = ',';
		} elsif(/=/) {
			my ($key, $val) = split(/ *= */, $_, 2);

			# accept LDAPv2 oid sequence indicators but remove them
			$key =~ s/^oid\.//i;

			$key = $self->canonicalize_name($key) // $key;

			# if this concerns a hash encoded value, there's not much we can do
			if($val =~ /^\#/) {
				$val = uc($val);
			} else {
				# remove quotes
				$val =~ s/^"((?:[^"\\]|\\.)*)"\z/$1/;
				# resolve escapes
				$val =~ s{\\(?:([^0-9a-fA-F])|([0-9a-fA-F][0-9a-fA-F]))}{$1 // chr(hex($2))}eg;

				$val = $self->canonicalize_value($key, $val);

				# encode special characters
				$val =~ s/([,+"\\<>;]|^[ \#]| \z)/\\$1/g;
				if(defined eval { Encode::decode('UTF-8', my $dummy = $val, Encode::FB_CROAK) }) {
					$val =~ s{([\x00-\x1F])}{sprintf('\\%02x', ord($1))}ge;
				} else {
					$val =~ s{([^\x20-\x7F])}{sprintf('\\%02x', ord($1))}ge;
				}
			}
			$_ = "$key=$val";
		} else {
			die "failed to parse dn '$dn'\n";
		}
	}

	# Reorder multivalued RDNs:
	my $start;
	# Fake an extra dummy entry at the end of the array, so we have a
	# chance to reorder a multivalued RDN that is at the end of the DN:
	push @rdns, '';
	for(my $i = 1; $i < @rdns; $i += 2) {
		if($rdns[$i] eq '+') {
			$start //= $i - 1;
		} elsif(defined $start) {
			# reorder all even-numbered entries between $start and $i
			my @indices = map { $start + $_ * 2 } 0..(($i - $start) / 2);
			@rdns[@indices] = sort @rdns[@indices];
			undef $start;
		}
	}

	return join('', @rdns);
}

1;

__END__

=encoding utf8

=head1 NAME

Net::LDAP::Schema::Canonical - transform LDAP values to a canonical representation

=head1 SYNOPSIS

 use Net::LDAP::Schema::Canonical;

 my $schema = $ldap_connection->schema;
 my $cn = $schema->canonicalize_value(cn => "Joe Sixpack");
 my $dn = $schema->canonicalize_dn("commonName=Joe Sixpack,dc=example,dc=com");

 # if you don't have a schema (note the package name!):
 Net::LDAP::Schema->canonicalize_value(telephoneNumber => "555-1234");

=head1 USAGE

Despite the name, Net::LDAP::Schema::Canonical registers its functions in
the Net::LDAP::Schema class.

It does this because schema information is necessary for proper
canonicalization: different LDAP attribute types have different
canonicalization rules. Some types are case sensitive, some are not, some
(such as integers) require further rewriting beyond simple insigificant
whitespace removal or case mapping.

By registering its functions in the Net::LDAP::Schema class, they are
immediately available for use when you have retrieve the LDAP schema.

However, a schema may not always be available. In such cases you can always
invoke the canonicalization methods as class methods, i.e.:

 Net::LDAP::Schema->canonicalize_value(...);

instead of:

 $schema->canonicalize_value(...);

The methods will then fall back to a small built-in list of common
attributes and their respective canonicalization rules (to be precise, the
ones specified in RFC 4519).

=head2 $schema->canonicalize_value($name, $value)

Canonicalizes a value based on the type of the attribute name. For example,
if $name is "cn", a case insensitive canonicalization is applied. Instead
of the attribute name you can also pass a matching type directly (for
example, caseIgnoreMatch).

=head2 $schema->canonicalize_dn($dn)

Canonicalize a distinguished name (DN) by disassembling the constituent
RDNs and normalizing each of them according to their respective
canonicalization rules.

The returned string is guaranteed to be valid UTF-8 but is not marked as
such (LDAP DNs are technically byte oriented, not character oriented).
Any values that would be invalid UTF-8 are encoded using numeric escapes
as necessary.

=head2 $schema->canonicalize_name($name)

Return the canonical name for an attribute. For example, "CN", "commonName"
and "2.5.4.3" all resolve to "cn". Any options (the bits after the
semicolon) are lowercased but otherwise left unmodified. For example,
"OU;lang-EN" becomes "ou;lang-en".

=head1 LIMITATIONS

Invalid values are sometimes flagged as such, but not always. This class is
not intended as a validator, so the "Garbage In = Garbage Out" (GIGO) rule
applies.

Some data types (bit string, boolean, fax numbers, time, lists) are not
handled at all, or not handled completely. It is the author's intention to
implement canonicalization for all datatypes specified in RFC 4517 which
have a Match rule.

Values that are represented using a sharp (#) sign followed by the
hexadecimal encoded BER representation of their ASN.1 datatype are not
canonicalized. This may change in the future (but don't hold your breath).

=head1 AUTHOR

Copyrigh (c) 2015 Wessel Dankers <wsl@uvt.nl>

=cut
