use strict;
use warnings  FATAL=> 'all';
package BaseLdapsearch;
use Carp qw(carp cluck);
use Baseobject;
use Exporter;
use Data::Dumper;
$Data::Dumper::Indent = 1;
use warnings FATAL=> 'all';

use XML::LibXML;
#use XML::LibXML::XPathContext;

use Net::LDAPS;
use Net::LDAP;
use MIME::Base64;

our @ISA=('Baseobject' );
our @EXPORT = qw (verbosity);
my $version ='$Id: Ldapsearch.pm 50026 2022-10-11 14:10:44Z anton $ ';

my $SELF;
my $hierarchy = {};
my $LDAP = {};

sub new {
	my $pkg = shift;
	my $self = bless (new Baseobject(@_),$pkg);
	$SELF=$self;

	$self->needs( [ qw (cfhash serverinfo) ] ) ;
	
	$self->allows ( { capath =>'/etc/ssl/certs' });
	$self->allows ( [ qw ( binddn bindpwd)]);
	
	$self->fields(@_);
	$self->checkneeds();

#	$self->{fields}->{cfhash}->{serverinfo} = $self->{fields}->{serverinfo}; 
	
	$self->ldapinit() unless $self->error();
	return $self;
}

sub ldapsearch {
	my ($self, $filter, $savestate) = @_;

	if ($filter =~ /^\(*blockradius/){
		my $radiusBlockedUvTAuthString = $self->{fields}->{cfhash}->{radiusBlockedUvTAuthString};
		$filter = "(uvt-auth=$radiusBlockedUvTAuthString*)";
	}

	warn "actual filter: $filter" 
		if $self->{fields}->{cfhash}->{dumpldapsearch} or 
		($self->{fields}->{cfhash}->{dumpldap}) ;

	my $msg = $self->{ldap}->search(base   => $self->{base},
									attrs => ['*', 'createTimestamp', 'modifyTimestamp', 'entryUUID'],
									filter => $filter);

	warn $self->inspect($msg) if $self->{fields}->{cfhash}->{dumpldap};

	if ($msg->code) {
		if ($msg->error() =~ /sizelimit exceeded/i) {
			$self->error('sizelimitexceeded');
		} else {
			warn ("ldapsearch failed: ", $msg->error(),
							"\nfilter => $filter\n",
							"base  => $self->{base}",
				);
			$self->friendlyerror('Search failed: '. $msg->error(). "\n");
		}
	}

	$self->{msg} = $msg if defined ($savestate) and $savestate;

#  warn "msg: $msg";
#  warn $self->inspect($msg);
	return $msg;
}

sub ldapinit {
	my $self = shift;
	my $host = $self->{fields}->{serverinfo};
	my $userinfo;
	my ($type, $port, $base);
	($type, $host, $port) = split(/:/,$host);
	($port, $base) = split(/\s*;\s*/, $port,2);
	$host =~ s{//}{};

	warn ("type: $type, port: $port, base: $base, host: $host") if $self->{fields}->{cfhash}->{dumpldap};
	$self->{base} = $base;

	unless (exists $LDAP->{$host}) {
		warn "Creating new ldap instance for host: $host"  if $self->{fields}->{cfhash}->{dumpldap};
		if ($type eq 'ldaps') {
			$self->{ldap}=Net::LDAPS->new(
				$host,
				'verify' => 'require',
				'capath' => $self->{fields}->{capath},
				'port' => $port,
				'raw' => qr/(?i:^jpegPhoto|;binary)/
				);

		} elsif ($type eq 'ldap') {
			$self->{ldap}= Net::LDAP->new(
				$host,
				'port'=>$port,
				'raw' => qr/(?i:^jpegPhoto|;binary)/
				);

		}else {
			warn "Invalid url in serverinfo: \"$type\", should be ldaps:// or ldap://";
			$self->error('invalid_serverinfo');
		}
		$LDAP->{$host} = $self->{ldap};
	} else {
		warn "No new ldapinstance for host: $host" if $self->{fields}->{cfhash}->{dumpldap};
		$self->{ldap}=$LDAP -> {$host};
	}

	unless ($self->{ldap}) {
		warn "Could not connect to $type//$host:$port $@";
		$self->error("connect_failed" );
	}

#  warn $self->inspect($self->{fields});

	my $msg;
	if (exists ($self->{fields}->{binddn})) {
		$msg = $self->{ldap}->bind(
			$self->{fields}->{binddn},
			password=>$self->{fields}->{bindpwd}
			);
		if ($msg->code) {
			warn "error binding $self->{fields}->{binddn} on ldapserver: $type://$host pw: \"$self->{fields}->{bindpwd}\" ", $msg->error();
			$self->error($msg->error());
		} else {
			warn "BIND SUCCESSFUL FOR $self->{fields}->{binddn}, ldapserver: $host" if $self->{fields}->{cfhash}->{dumpldap};
		}
	}
}


package Ldapsearch;
use Baseobject;
use Exporter;
use Displayentry;
use Carp qw(carp cluck);

use Data::Dumper;
$Data::Dumper::Indent=1;

use XML::LibXML;
use Net::LDAPS;
use Net::LDAP;
use MIME::Base64;

our @ISA=('BaseLdapsearch' );
our @EXPORT = qw (verbosity);

sub new {
	my $pkg = shift;
	my $self = bless (new Baseobject(@_),$pkg);
	$SELF = $self;

	$self->needs(
		[ qw ( filter serverinfo  ldapattribs babelfish cfhash configTree outTree orgTree cgiparams) ]
		) ;

	$self->allows (
		{ capath => '/etc/ssl/certs' ,
		  display => '',
		  language => 'nl',
		  singletry => 0,
		  show => 1,
		  limit => '',
		  attribAccess => {},
		}
		);

	$self->fields(@_);
	$self->checkneeds();
	return $self if ($self->error());

	$self->{displaytypes}=$self->attribs();

	$self->ldapinit();
	return $self if ($self->error());
	my $msg = $self->ldapsearch($self->{fields}->{filter}, 'savestate');

	# NB, the presence of a ':' in the errormessage is a signal
	#     for how the caller conveys the message to the user!
	#
	if ($msg->code) {
		$self->{fields}->{show} = 1;
	} elsif ($msg->entries == 0) {
		#$self->error('nothingfound');
		$self->verbose("nothingfound");
	}
	return $self;
}

sub attribs {
	my $self = shift;
	my $h;
	my $prevname;
	# ldapldattribs
	#          x = a
	#              b
	# means: x=a, x=b

	foreach my $assignment (@{$self->{fields}->{ldapattribs}}) {
		my ($name, $value) = split ( /\s*=\s*/, $assignment);
		unless (defined($value)) {
			# no "a=b", there is a value only, assign it to the previous name
			$value = $name;
			$name = $prevname ;
		}
		map {$h->{$_} = $name } split (/\s+/,$value);
		$prevname = $name;
	}

#  $self->debug( $self->inspect($h));
	return $h;
}

sub sortentries {
	my @list;
	#
	# if it has a sortname, use that to sort, otherwise use the sn or dn (i.e. orgID)
	# special accounts have no sortname
	#
	my $ae = $a->get_value('sortname') || $b->get_value('sn');;
	my $be = $b->get_value('sortname') || $b->get_value('sn');

	if (defined($ae) and defined($be)) {
		return  $ae cmp $be;
	}

	if ( defined ($ae) or defined ($be)) {
		# organisations first,
		return 1 if defined $ae;
		return -1;
	}
	
	# both organisations
	# 

	my $aname = $a->dn(); $aname =~s/,.*//; $aname =~s/^orgID=//;
	my $bname = $b->dn(); $bname =~s/,.*//; $bname =~s/^orgID=//;
	my ($ah, $bh);

	if ($a->get_value('orgID')) {
		$ah = $SELF->decodeOrgname($aname)->{name};
		$bh = $SELF->decodeOrgname($bname)->{name};
	} else {
		$ah = $SELF->translateorgname($aname)->{name};
		$bh = $SELF->translateorgname($bname)->{name};
	}

	$ah cmp $bh;
}

sub translateorgname {
	my ($self, $orgname) = @_;
	my $orgTree = $self->{fields}->{orgTree};
	my @node;
	my @res;
	my $search = "//*[name=\"$orgname\"]";
	my $en;

	return $orgname;


	my $value = $orgname;
	eval  {
		(@node)=$orgTree->{doc}->findnodes("$search");
	};
	if ($@) {
		warn "translateorgname: failed on findnodes($search): $@";
		die ;
	}
	unless (@node)  {
		warn "translateOrgname: organisation not found: $orgname";
		#warn "translateOrgname Orgtree:" . $orgTree->{doc}->toString(2);
#	 warn Dumper $orgTree->{doc}->toString(2);
	} elsif (@node >=1) {
		warn "translateOrgname MULTIPLE organisations with name: $orgname" if (@node > 1);
		$en = $node[0]->getAttribute('en');
	}

	if (defined ($en) and $en)  {
		$value = $en;
	}

	my $h;
	$h->{name} = $value;
	$h->{depth} = $node[0]->getAttribute('depth');
	$h;
}


sub decodeOrgname {
	my ($self, $orgID) = @_;
	my $orgTree = $self->{fields}->{orgTree};
	my @node;

	my $search = "//*[orgID=\"$orgID\"]";

	eval {
		(@node) = $orgTree->{doc}->findnodes("$search");
	};
	if ($@) {
		die "decodeOrgname: failed on findnodes($search): $@";
	}
	unless (@node)  {
		warn "decodeOrgname: organisation not found: $orgID, search: $search";
		return $orgID;
	}

	die "decodeOrgname: Multiple organisations with orgID: $orgID" if @node > 1;
	my $h;
	$h->{name} = $node[0]->findvalue('name');
	$h->{depth} = $node[0]->findvalue('depth');
	return $h;
}

sub gendisplay {
	#
	# Zou dit niet beter naar ldapbrowser.in verplaatst kunnen worden?
	#
	my $self = shift;
	my @entries = $self->{msg}->entries();
	my $multiple = @entries;
	my $multhash = {};

	foreach my $entry (@entries) {
		if ( grep /^organizationalunit$/i, $entry->get_value('objectclass')) {
			my ($orgID) = $entry->get_value('orgID');
			if ($orgID) {
				my $h = $self->decodeOrgname($orgID);

				if ($h->{depth} <= 1){
					$multhash->{'toporganisation'}++;
				}
				else {
					$multhash->{'suborganisation'}++;
				}
			} else {
				warn "FAKE downwardcompatibility, organisation should always have an orgID";
				$multhash->{'toporganisation'}++;
			}
		}
	}

	foreach my $entry (sort sortentries (@entries)) {
		my $di = new Displayentry(
			{
				multhash => $multhash,
				multiple => $multiple,
				entry => $entry,
				configTree => $self->{fields}->{configTree},
				orgTree => $self->{fields}->{orgTree},
				displaytypes => $self->{displaytypes},
				outTree  => $self->{fields}->{outTree},
				babelfish => $self->{fields}->{babelfish},
				hierarchy => $hierarchy,
				limit => $self->{fields}->{limit},
				ldapSearch => $self,
				attribAccess => $self->{fields}->{attribAccess},
			});

		if ($di->error()) {
			warn $di->error;
			$self->error( 'display_error');
		}
	}

	$self->multableheaders() if $self->{msg}->entries() > 1;
	$self->finalizehierarchy();
	$self->numberofhits() if $self->{fields}->{cgiparams}->{ldapfilter};

	$self->addSearchBox();
}

sub addSearchBox {
	my $self=shift;
	my $parent = $self->{fields}->{outTree}->{apptop};
	$parent = $self->output($parent, 'searchbox', '',
							{
								'class'=>'searchbox',
							});
}

sub numberofhits {
	my $self=shift;
	my $hits= $self->{msg}->entries();

	#better in printcss?
	return if ($hits == 1);

	my $parent=$self->{fields}->{outTree}->{apptop};
	$parent=$self->output($parent,'numberofhits','',
						  {
							  'title'=>$self->{fields}->{babelfish}->translate('numberofhits'),
						  });

	$self->output($parent,'value',$hits);
}

sub finalizehierarchy {
	my $self = shift;
	my @selected;
	my $parent = $self->{fields}->{outTree}->{apptop};

	unless (defined($hierarchy->{hierarchylist})) {
		if (exists($self->{fields}->{cgiparams}->{searchContext})
			 and
			 $self->{fields}->{cgiparams}->{searchContext}) 
		{
			$self->verbose("CONTEXT $self->{fields}->{cgiparams}->{searchContext}");
			$self->addhierarchy($self->{fields}->{cgiparams}->{searchContext});

		} else {
			my $child = $self->output($parent, 'hierarchy');
			$self->output($child, 'value',$self->{fields}->{babelfish}->translate('gotop'), { href=>"?top" }) ;
			return;
		}
	}

	unless (exists $hierarchy->{hierarchylist}) {
		warn "CONTEXT unknown for: $self->{fields}->{cgiparams}->{searchContext}";
		my $child = $self->output($parent, 'hierarchy');
		$self->output($child, 'value', $self->{fields}->{babelfish}->translate('gotop'), {href=>"?top" }) ;
		return;
	}

	my $joined;
	foreach my $ellist ( @{$hierarchy->{hierarchylist}}) {
		my $concat = '';
		foreach my $l (@$ellist) {
			$concat .= ">$l->{org}" if $l->{org};
		}
		$joined->{$concat} = $ellist;
		#warn "concat: $concat == ", $self->inspect($concat);
	}
	# de hierarchylist bevat iets als
	# Tilburg University > LIS: Library and IT Services > LIS: Information Technology > LIS: IT/Application and Services Management
	# Tilburg University > LIS: Library and IT Services > LIS: Information Technology
	# Tilburg University > LIS: Library and IT Services
	# Tilburg University
	#
	# Sorteer de lijst op lengte, langste eerst en voeg toe aan lijst
	# Ieder volgend element dat binnen de strings past die al in de lijst zitten wordt genegeerd.
	foreach my $key (sort{length($b) <=> length($a)} keys %$joined) {
		my $regexp = $key;
		$regexp  =~ s/(\W)/\\$1/g;
		push (@selected, $key) unless grep (/^$regexp/, @selected);
	}

	foreach my $key (sort @selected) {
		my $sel = $joined->{$key};
		my $child = $self->output($parent, 'hierarchy');

		$self->output($child, 'value',$self->{fields}->{babelfish}->translate('gotop'), { href => "?top" }) ;

		foreach my $el (@$sel) {
			$self->writehierarchy($child, $el);
		}
		# wordt niets mee gedaan, nog
		#push(@{$self->{hierarchy}}, $child->textContent());
	}
	$self->{hierarchy} = \@selected;
} 


sub addhierarchy {
	# hierin geplempt vanuit Displayentry
	my ($self, $value) = @_;
	#
	# remove leading parentstring
	#
#	$self->debug('start addhierarchy');
	#save the orginal value it contains a leading hierarchy string like: "ITS: xxx"
	$self->{fields}->{hierarchy}->{orgvalues}->{$value} = $value;

	my $orgTree = $self->{fields}->{orgTree};
	my @node;

	# source: Wessel!
	# werkt echter nog niet op etch!
	#  my $xc = new XML::LibXML::XPathContext;
	#  $xc->registerVarLookupFunc(sub {$_[0]->{$_[1]}}, {name => $value});
	my $search="//*[name=\"$value\"]";

	eval {
		#   (@node) = $xc->find('//*[@name=$search]', $orgTree->{doc});
		(@node)=$orgTree->{doc}->findnodes("$search");
	};
	if ($@) {
		warn "addhierarchy: failed on findnodes($value): $@";
	}

	unless (@node) {
		warn ("addhierarchy: ORGANISATION NOT FOUND: $value");
#		warn "Orgtree:" . $orgTree->{doc}->toString(2);
	}

	if (@node >1) {
		warn "MULTIPLE organisation in multiple trees: $value, taking the first ";
	}

	my @hierarchy = ();
	my $node = $node[0];
	if ($node) {
		my $parent = $node;
#		warn "node: " , $node->getAttribute('name');
		unshift (@hierarchy, $parent);

		while ($parent = $parent->parentNode and $parent = $parent->parentNode) {
			unshift(@hierarchy, $parent);
		}
	}

	my @hashlist = ();
	foreach my $el (@hierarchy)	{
		my $h;
		my $name = $el->findvalue('name');
		$h->{org} = $name;
		$h->{display} = $name;
		$h->{depth} = $el->findvalue('depth');
		push(@hashlist,$h);
	}
	# Eindresultaat:
 	push (@{$hierarchy->{hierarchylist}}, \@hashlist) if @hashlist;

}

sub writehierarchy {
	my ($self, $parent, $el) = @_;
	my $name = $el->{display};
	my $ldapencoded = ldapsearchencode($el->{org});
	my $ldapfilter;

	$ldapfilter = urlencode("(&(ou=$ldapencoded)(objectclass=organizationalunit))");

	my $searchContext = urlencode($el->{org});

	$self->output($parent, 'value','>', {
		'hierarchy' => $name,
				  }) if $name;

	$self->output($parent,'value', $name, {
		'href' => "?searchContext=$searchContext&ldapfilter=$ldapfilter",
		'hierarchy' => $name,
				  }) ;
}

sub multableheaders {
	my $self=shift;
	my @nodes;
	my $parent = $self->{fields}->{outTree}->{apptop};
	my $mth = $self->output($parent,'multableheaders');

	my $catnode;

	foreach my $category (qw (staff student pnil guest etranger)) {
		my $path = "/top/categories/$category/multiple/*";

		eval {
			@nodes=$self->{fields}->{configTree}->{doc}->findnodes("$path");
		};
		if ($@) {
			die "multableheaders: failed on findnodes($path): $@";
		}
        # warn "found nodes:" , scalar @nodes ,"for $category";

		$catnode = $self->output($mth, $category);
		foreach my $node  (@nodes) {
			my $nodeName = $node->nodeName();
			my $noprint = '';
			$noprint = '_noprint' if $node->getAttribute('noprint');

			if (exists ($self->{fields}->{attribAccess}->{$nodeName})) {
				next unless ($self->{fields}->{attribAccess}->{$nodeName});
			}
			my $string = $self->{fields}->{babelfish}->translate($nodeName);
			$self->output($catnode, 'multvalue', "\u$string", {class => "multhead$noprint"});
		}
		my $string = $self->{fields}->{babelfish}->translate('lookfor');
		$self->output($catnode, 'multvalue', "\u$string", { class => "lookfor"});
	}
}


sub output {
	my ($self, $parent, $el, $content, $attrhash) = @_;
	my $newel;

	$newel = $self->{fields}->{outTree}->{doc}->createElement($el);
	$parent->appendChild($newel);

	if (defined $content) {
		$newel->appendChild($self->{fields}->{outTree}->{doc}->createTextNode($content));
	}

	if (defined ($attrhash))  {
		foreach my $attr (keys(%$attrhash)) {
#      warn "setting atr: $attr,$attrhash->{$attr}";
			$newel->setAttribute($attr,$attrhash->{$attr}) if (defined ($attrhash->{$attr}));
		}
	}
	return $newel;
}

sub ldapsearchencode {

	#   http://www.cse.ohio-state.edu/cgi-bin/rfc/rfc2254.html
	#
	#   If a value should contain any of the following characters
	#
	#             Character       ASCII value
	#             ---------------------------
	#             *               0x2a
	#             (               0x28
	#             )               0x29
	#             \               0x5c
	#             NUL             0x00
	#
	#     the character must be encoded as the backslash '\' character (ASCII
	#     0x5c) followed by the two hexadecimal digits representing the ASCII
	#     value of the encoded character.
    # comfort emacs ;(
	#  map {$specialchars->{$_}=1 } split (//,'\()\*');

	# Het simpelst is gewoon ALLES coderen, maar dat vinden mensenogen niet fijn.
	# .. who cares?..
	#  my $encoded='\\'.join('\\', map {sprintf ("%x",$_)} unpack('U*',$str));

	#
	# alternatief: codeer alleen de noneword characters,
	# NB. blijf van de wildcard '*' af!
	#
    my $str = shift;
	my $res = '';
	if ($str) {
		foreach my $char (split (//,$str)) {
			if ($char =~ /\W/ and $char !~ /\*/) {
				$res .= '\\' . sprintf("%x",ord($char));
			} else {
				$res .= $char;
			}
		}
	}
	$res;
}


sub urlencode {
	my ($x) = @_ ;
	return "" if (! defined ($x));

	$x=$_[0] ;
	$x =~ s/([^ \w])/sprintf("%%%02X",ord($1))/ge;
	$x =~ s/ /\+/g ;
	$x ;
}




42;
