-- $Id: stored.sql 42198 2014-09-10 13:24:54Z wsl $
-- $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/doc/stored.sql $

\encoding UTF8

SET client_min_messages TO WARNING;

BEGIN;

SET search_path TO kiki;

CREATE OR REPLACE VIEW mail_addresses (mail_alias, address) AS
	SELECT m.mail_alias, m.name || '@' || n.name
	FROM mail_aliases m
	JOIN domains d USING (domain)
	JOIN domainnames n ON d.main = n.domainname;

CREATE OR REPLACE VIEW func_mail_addresses (mail_alias, address) AS
	SELECT m.mail_alias, m.name || '@' || n.name
	FROM mail_aliases m
	JOIN domains d USING (domain)
	JOIN domainnames n ON d.main = n.domainname
	WHERE m.person IS NULL;

CREATE OR REPLACE VIEW all_mail_addresses (mail_alias, address) AS
	SELECT m.mail_alias, m.name || '@' || n.name
	FROM mail_aliases m
	JOIN domainnames n USING (domain);

CREATE OR REPLACE FUNCTION internalize_domain(IN v TEXT) RETURNS VOID AS $$
	DECLARE
		d BIGINT;
		n BIGINT;
		a BIGINT;
		aa BIGINT;
		e BIGINT;
		p BIGINT;
		m TEXT;
		l TEXT;
	BEGIN
		SELECT INTO d domain FROM domainnames WHERE name = v;
		FOR e, a, m IN SELECT external_destination, mail_alias, mailaddress FROM external_destinations
				WHERE lower(mailaddress) LIKE '%@' || lower(v) LOOP
			l := split_part(m, '@', 1);
			SELECT INTO aa, p mail_alias, person FROM mail_aliases JOIN domains USING (domain)
				WHERE domains.domain = d AND lower(mail_aliases.name) = lower(l);
			IF NOT FOUND THEN
				INSERT INTO mail_aliases (name, domain, addressbook) VALUES (l, d, FALSE) RETURNING mail_alias INTO aa;
				p := NULL;
				RAISE WARNING 'While internalizing domain "%": "%" is mentioned as a destination for an alias and was created as a placeholder without any destinations. You MUST supply destinations yourself.', v, m;
			END IF;
			IF p IS NULL
			THEN
				INSERT INTO internal_destinations (internal_destination, mail_alias, destination)
					VALUES (e, a, aa);
			ELSE
				INSERT INTO personal_destinations (personal_destination, mail_alias, person)
					VALUES (e, a, p);
			END IF;
			DELETE FROM external_destinations WHERE external_destination = e;
		END LOOP;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION create_domain(IN v TEXT, OUT d BIGINT) AS $$
	DECLARE
		n BIGINT;
		a BIGINT;
		aa BIGINT;
		e BIGINT;
		m TEXT;
	BEGIN
		m := lower(v);
		SELECT INTO d domain FROM domainnames WHERE name = m;
		IF NOT FOUND THEN
			d := nextval('seq');
			INSERT INTO domainnames (domain, name) VALUES (d, m)
				RETURNING domainname INTO n;
			INSERT INTO domains (domain, main) VALUES (d, n);
		END IF;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION name2domain(n TEXT) RETURNS TEXT AS $$
	DECLARE
		d TEXT;
	BEGIN
		RETURN domain FROM domainnames WHERE name = lower(n);
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION alias2person(u TEXT, d BIGINT) RETURNS BIGINT AS $$
	BEGIN
		RETURN mail_aliases.person FROM mail_aliases
			WHERE mail_aliases.name = u AND mail_aliases.domain = d;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION alias2person(u BIGINT) RETURNS BIGINT AS $$
	BEGIN
		RETURN mail_aliases.person FROM mail_aliases
			WHERE mail_aliases.mail_alias = u;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION alias2person(u TEXT) RETURNS BIGINT AS $$
	BEGIN
		RETURN alias2person(split_part(u, '@', 1), name2domain(split_part(u, '@', 2)));
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION create_destination(ialias BIGINT, imail TEXT) RETURNS VOID AS $$
	DECLARE
		l TEXT;
		r TEXT;
		d BIGINT;
		a BIGINT;
		p BIGINT;
	BEGIN
		l = split_part(imail, '@', 1);
		r = lower(split_part(imail, '@', 2));
		d := name2domain(r);
		IF d IS NOT NULL THEN
			SELECT INTO a, p mail_alias, person FROM mail_aliases WHERE lower(name) = lower(l) AND domain = d;
			IF NOT FOUND THEN
				RAISE EXCEPTION 'Mail alias % not found', imail;
			END IF;
			IF p IS NULL
			THEN
				INSERT INTO internal_destinations (mail_alias, destination)
					VALUES (ialias, a);
			ELSE
				INSERT INTO personal_destinations (mail_alias, person)
					VALUES (ialias, p);
			END IF;
		ELSE
			INSERT INTO external_destinations (mail_alias, mailaddress)
				VALUES (ialias, imail);
		END IF;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION create_alias(ialias TEXT, iaddressbook BOOL) RETURNS BIGINT AS $$
	DECLARE
		l TEXT;
		r TEXT;
		d BIGINT;
		a BIGINT;
		p TEXT;
	BEGIN
		l = split_part(ialias, '@', 1);
		r = lower(split_part(ialias, '@', 2));
		d := create_domain(r);
		SELECT INTO a mail_alias FROM mail_aliases WHERE lower(name) = lower(l) AND domain = d;
		IF FOUND THEN
			UPDATE mail_aliases SET addressbook = iaddressbook WHERE mail_alias = a;
			DELETE FROM internal_destinations WHERE mail_alias = a;
			DELETE FROM external_destinations WHERE mail_alias = a;
			DELETE FROM personal_destinations WHERE mail_alias = a;
		ELSE
			INSERT INTO mail_aliases (name, domain, addressbook)
				VALUES (l, d, iaddressbook)
				RETURNING mail_alias INTO a;
		END IF;
		RETURN a;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION create_person(ianr BIGINT, idest TEXT) RETURNS VOID AS $$
	BEGIN
		INSERT INTO persons (person, canonical, mailaddress) VALUES (ianr, NULL, idest);
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION remove_person(ianr BIGINT) RETURNS VOID AS $$
	DECLARE
		victim TEXT;
	BEGIN
		FOR victim IN SELECT address FROM func_mail_addresses a JOIN personal_destinations d USING (mail_alias) WHERE d.person = ianr
		LOOP
			RAISE WARNING 'Because anr % is being removed, it is also deleted from %', ianr, victim;
		END LOOP;
		DELETE FROM persons WHERE person = ianr;
		IF NOT FOUND THEN
			RAISE WARNING 'Poging om % te verwijderen maar deze bestaat niet', ianr;
		END IF;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION upgrade_to_personal(ianr BIGINT, ialias BIGINT) RETURNS VOID AS $$
	DECLARE
		n TEXT;
		m TEXT;
		d TEXT;
		p BIGINT;
		e BIGINT;
	BEGIN
		n := address FROM mail_addresses WHERE mail_alias = ialias;
		IF EXISTS(SELECT * FROM internal_destinations WHERE mail_alias = ialias) THEN
			RAISE EXCEPTION 'Poging om % te registreren voor % maar deze bestaat al als functioneel alias', n, a;
		END IF;
		p := COUNT(*) FROM personal_destinations WHERE mail_alias = ialias;
		e := COUNT(*) FROM external_destinations WHERE mail_alias = ialias;
		IF p + e > 1 THEN
			RAISE EXCEPTION 'Poging om % te registreren voor % maar deze bestaat al als functioneel alias', n, a;
		ELSIF p = 1 THEN
			SELECT INTO p person FROM persons JOIN personal_destinations USING (person) WHERE mail_alias = ialias;
			IF p <> ianr THEN
				RAISE EXCEPTION 'Poging om % te registreren voor % maar deze bestaat al als functioneel alias (voor %)', n, ianr, p;
			END IF;
		ELSIF e = 1 THEN
			m := mailaddress FROM persons WHERE person = ianr;
			d := mailaddress FROM external_destinations WHERE mail_alias = ialias;
			IF lower(m) = lower(d) THEN
				DELETE FROM external_destinations WHERE mail_alias = ialias;
				INSERT INTO personal_destinations (mail_alias, person) VALUES (ialias, ianr);
			ELSE
				RAISE EXCEPTION 'Poging om % te registreren voor % (%) maar deze bestaat al als functioneel alias (voor %)', n, ianr, m, d;
			END IF;
		ELSE
			INSERT INTO personal_destinations (mail_alias, person) VALUES (ialias, ianr);
		END IF;
		UPDATE mail_aliases SET person = ianr WHERE mail_alias = ialias;

		-- internal_destinations mogen niet wijzen naar een mail_alias met ingevulde person
		FOR e IN DELETE FROM internal_destinations WHERE destination = ialias RETURNING mail_alias
		LOOP
			INSERT INTO personal_destinations (mail_alias, person) VALUES (a, ianr);
		END LOOP;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION force_to_personal(iperson BIGINT, ialias BIGINT) RETURNS VOID AS $$
	DECLARE
		n TEXT;
		m TEXT;
		d TEXT;
		p BIGINT;
		e BIGINT;
		a BIGINT;
	BEGIN
		UPDATE mail_aliases SET addressbook = FALSE WHERE mail_alias = ialias;
		DELETE FROM internal_destinations WHERE mail_alias = ialias;
		DELETE FROM external_destinations WHERE mail_alias = ialias;
		DELETE FROM personal_destinations WHERE mail_alias = ialias AND person <> iperson;
		IF NOT EXISTS(SELECT * FROM personal_destinations WHERE mail_alias = ialias AND person = iperson) THEN
			INSERT INTO personal_destinations (mail_alias, person) VALUES (ialias, iperson);
		END IF;
		UPDATE mail_aliases SET person = iperson WHERE mail_alias = ialias;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION create_personal_alias(ianr BIGINT, ialias TEXT) RETURNS BIGINT AS $$
	DECLARE
		l TEXT;
		r TEXT;
		d BIGINT;
		m BIGINT;
		a BIGINT;
	BEGIN
		IF NOT EXISTS(SELECT * FROM persons WHERE person = ianr) THEN
			RAISE EXCEPTION 'ANR % is niet bekend', ianr;
		END IF;

		l = split_part(ialias, '@', 1);
		r = lower(split_part(ialias, '@', 2));
		d := name2domain(r);
		SELECT INTO m, a mail_alias, person FROM mail_aliases WHERE lower(name) = lower(l) AND domain = d;
		IF FOUND THEN
			IF a IS NULL THEN
				PERFORM force_to_personal(ianr, m);
			ELSE
				UPDATE mail_aliases SET person = ianr WHERE mail_alias = m;
				UPDATE personal_destinations SET person = ianr WHERE mail_alias = m AND person = a;
			END IF;
		ELSE
			INSERT INTO mail_aliases (name, domain, addressbook, person) VALUES (l, d, FALSE, ianr)
				RETURNING mail_alias INTO m;
			INSERT INTO personal_destinations (mail_alias, person) VALUES (m, ianr);
		END IF;

		RETURN m;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION remove_personal_alias(ianr BIGINT, ialias TEXT) RETURNS BIGINT AS $$
	DECLARE
		l TEXT;
		r TEXT;
		d BIGINT;
		m BIGINT;
		a BIGINT;
	BEGIN
		IF NOT EXISTS(SELECT * FROM persons WHERE person = ianr) THEN
			RAISE EXCEPTION 'ANR % is niet bekend', ianr;
		END IF;

		l = split_part(ialias, '@', 1);
		r = lower(split_part(ialias, '@', 2));
		d := name2domain(r);
		SELECT INTO m, a mail_alias, person FROM mail_aliases WHERE lower(name) = lower(l) AND domain = d;
		IF FOUND THEN
			IF a IS NULL THEN
				RAISE EXCEPTION 'Poging om % te verwijderen van % maar deze is een functioneel alias', ialias, ianr;
			ELSIF a <> ianr THEN
				RAISE EXCEPTION 'Poging om % te verwijderen van % maar deze is een andere persoon', ialias, ianr;
			ELSE
				DELETE FROM mail_aliases WHERE mail_alias = m;
			END IF;
		ELSE
			RAISE 'Poging om % te verwijderen van % maar deze bestaat niet', ialias, ianr;
		END IF;

		RETURN m;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION tweak_alias(iold TEXT, inew TEXT) RETURNS BIGINT AS $$
	DECLARE
		l TEXT;
		r TEXT;
		n TEXT;
		m BIGINT;
		d BIGINT;
	BEGIN
		IF lower(iold) <> lower(inew) THEN
			RAISE EXCEPTION 'Poging om % te tweaken naar % maar deze zijn niet gelijk', iold, inew;
		END IF;

		n = split_part(inew, '@', 1);
		l = split_part(iold, '@', 1);
		r = lower(split_part(iold, '@', 2));
		d := name2domain(r);
		UPDATE mail_aliases SET name = n WHERE lower(name) = lower(l) AND domain = d RETURNING mail_alias INTO m;
		IF NOT FOUND THEN
			RAISE EXCEPTION 'Poging om % te tweaken naar % maar deze bestaat niet', ialias, ianr;
		END IF;

		RETURN m;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION set_canonical_alias(ianr BIGINT, ialias TEXT) RETURNS BIGINT AS $$
	DECLARE
		l TEXT;
		r TEXT;
		d BIGINT;
		m BIGINT;
		a BIGINT;
	BEGIN
		IF NOT EXISTS(SELECT * FROM persons WHERE person = ianr) THEN
			RAISE EXCEPTION 'ANR % is niet bekend', ianr;
		END IF;

		l = split_part(ialias, '@', 1);
		r = lower(split_part(ialias, '@', 2));
		d := name2domain(r);
		SELECT INTO m, a mail_alias, person FROM mail_aliases WHERE lower(name) = lower(l) AND domain = d;
		IF FOUND THEN
			IF a IS NULL THEN
				RAISE EXCEPTION 'Poging om % te activeren als canonical voor % maar deze is van een functioneel alias', ialias, ianr;
			ELSIF a <> ianr THEN
				RAISE EXCEPTION 'Poging om % te activeren als canonical voor % maar deze is van een andere user (%)', ialias, ianr, a;
			ELSE
				UPDATE persons SET canonical = m WHERE person = ianr;
			END IF;
		ELSE
			RAISE EXCEPTION 'Poging om % te activeren als canonical voor % maar deze bestaat niet', ialias, ianr;
		END IF;

		RETURN m;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION set_personal_destination(ianr BIGINT, idest TEXT) RETURNS VOID AS $$
	BEGIN
		UPDATE persons SET mailaddress = idest WHERE person = ianr;
		IF NOT FOUND THEN
			RAISE EXCEPTION 'ANR % is niet bekend', ianr;
		END IF;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION export_canonical() RETURNS TABLE (canon_from TEXT, canon_dest TEXT) AS $$
	BEGIN
		FOR canon_from, canon_dest IN SELECT am.address, cm.address
			FROM persons p
			JOIN mail_aliases aa ON aa.person = p.person
			JOIN mail_addresses cm ON cm.mail_alias = p.canonical
			JOIN mail_addresses am ON am.mail_alias = aa.mail_alias
			WHERE aa.mail_alias <> p.canonical AND NOT p.pita
		LOOP
			RETURN NEXT;
		END LOOP;

		FOR canon_from, canon_dest IN SELECT p.mailaddress, cm.address
			FROM persons p
			JOIN mail_aliases ca ON ca.mail_alias = p.canonical
			JOIN mail_addresses cm ON cm.mail_alias = ca.mail_alias
		LOOP
			RETURN NEXT;
		END LOOP;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION export_virtual() RETURNS TABLE (virt_from TEXT, virt_dest TEXT) AS $$
	DECLARE
		m BIGINT;
		d BIGINT;
	BEGIN
		CREATE TEMP TABLE virtual AS
				SELECT mail_alias, mail_alias AS derived FROM mail_aliases;
		CREATE INDEX virtual_mail_alias ON virtual (mail_alias);
		CREATE INDEX virtual_derived ON virtual (derived);
		CREATE UNIQUE INDEX virtual_tuples ON virtual (mail_alias, derived);

		LOOP
			FOR m, d IN
				SELECT DISTINCT v.mail_alias, i.destination
					FROM virtual v JOIN internal_destinations i ON v.derived = i.mail_alias
				EXCEPT ALL SELECT c.* FROM virtual c
			LOOP
				INSERT INTO virtual (mail_alias, derived) VALUES (m, d);
			END LOOP;

			EXIT WHEN NOT FOUND;
		END LOOP;

		FOR virt_from, virt_dest IN SELECT DISTINCT a.address, e.mailaddress
			FROM virtual v
			JOIN all_mail_addresses a ON a.mail_alias = v.mail_alias
			JOIN external_destinations e ON e.mail_alias = v.derived
		LOOP
			RETURN NEXT;
		END LOOP;

		FOR virt_from, virt_dest IN SELECT DISTINCT a.address, p.mailaddress
			FROM virtual v
			JOIN all_mail_addresses a ON a.mail_alias = v.mail_alias
			JOIN personal_destinations dst ON dst.mail_alias = v.derived
			JOIN persons p ON p.person = dst.person
		LOOP
			RETURN NEXT;
		END LOOP;

		DROP TABLE virtual;
	END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION fsck(ifix BOOL) RETURNS SETOF TEXT AS $$
	DECLARE
		ret BOOL;
		e_address TEXT;
		e_alias BIGINT;
		i_address TEXT;
		i_alias BIGINT;
		i_dest BIGINT;
		m_address TEXT;
		m_alias BIGINT;
		m_person BIGINT;
		p_address TEXT;
		p_person BIGINT;
	BEGIN
		-- external/person destinations mogen geen domeinen hebben die ook al in de domainnames-tabel staan
		FOR p_person, p_address IN SELECT p.person, p.mailaddress FROM persons p JOIN domainnames n
			ON lower(split_part(p.mailaddress, '@', 2)) = n.name
		LOOP
			-- niet fixbaar
			RETURN NEXT 'Destination ' || p_address || ' of ANR ' || p_person || ' refers to a domain that is also in the domain list';
		END LOOP;

		FOR e_alias, e_address IN SELECT e.mail_alias, e.mailaddress
			FROM external_destinations e JOIN domainnames n
			ON lower(split_part(e.mailaddress, '@', 2)) = n.name
		LOOP
			-- fixbaar
			RETURN NEXT 'Destination ' || e_address || ' of functional mail alias ' || e_alias || ' refers to a domain that is also in the domain list';
		END LOOP;

		-- internal_destinations mogen niet wijzen naar een mail_alias met ingevulde person
		FOR m_person, i_alias, i_dest IN SELECT m.person, i.mail_alias, i.internal_destination
			FROM internal_destinations i JOIN mail_aliases m
			ON i.destination = m.mail_alias WHERE m.person IS NOT NULL
		LOOP
			m_address := address FROM mail_addresses WHERE mail_alias = i_alias;
			RETURN NEXT 'Functional mail alias ' || m_address || ' refers to a person (' || m_person || ') but not through a personal alias';
			IF ifix THEN
				DELETE FROM internal_destinations WHERE internal_destination = i_dest;
				INSERT INTO personal_destinations (mail_alias, person) VALUES (i_alias, m_person);
			END IF;
		END LOOP;

		-- aliassen met non-null person mogen alleen personal destinations naar precies die person hebben
		FOR m_alias, m_person, p_person IN SELECT m.mail_alias, m.person, p.person
			FROM mail_aliases m JOIN personal_destinations p USING (mail_alias)
			WHERE m.person <> p.person
		LOOP
			m_address := address FROM mail_addresses WHERE mail_alias = m_alias;
			RETURN NEXT 'Alias ' || m_alias || ' for person ' || p_person || ' points to ' || m_person || ' instead';
		END LOOP;

		-- aliassen met non-null person mogen geen external destinations hebben
		FOR m_alias, m_person, e_address IN SELECT m.mail_alias, m.person, e.mailaddress
			FROM mail_aliases m JOIN external_destinations e USING (mail_alias)
			WHERE m.person IS NOT NULL
		LOOP
			m_address := address FROM mail_addresses WHERE mail_alias = m_alias;
			RETURN NEXT 'Alias ' || m_alias || ' for person ' || m_person || ' points to an external address (' || e_address || ')';
		END LOOP;

		-- aliassen met non-null person mogen geen internal destinations hebben
		FOR m_alias, m_person, i_alias IN SELECT m.mail_alias, m.person, i.mail_alias
			FROM mail_aliases m JOIN internal_destinations i USING (mail_alias)
			WHERE m.person IS NOT NULL
		LOOP
			i_address := address FROM mail_addresses WHERE mail_alias = i_alias;
			m_address := address FROM mail_addresses WHERE mail_alias = m_alias;
			RETURN NEXT 'Alias ' || m_alias || ' for person ' || m_person || ' points to an internal address (' || i_address || ')';
		END LOOP;

		-- canonical alias van elke persoon moet die person ingevuld hebben (en niet NULL zijn)
		FOR m_person, p_person IN SELECT m.person, p.person
			FROM mail_aliases m JOIN persons p ON m.mail_alias = p.canonical
			WHERE m.person <> p.person OR m.person IS NULL
		LOOP
			IF m_person IS NULL THEN
				RETURN NEXT 'Canonical alias for person ' || p_person || ' is not owned (by anyone)';
			ELSE
				RETURN NEXT 'Canonical alias for person ' || p_person || ' is owned by ' || m_person || ' instead';
			END IF;
		END LOOP;

		-- canonicalvelden moeten niet NULL zijn
		FOR p_person IN SELECT person FROM persons WHERE canonical IS NULL
		LOOP
			RETURN NEXT 'Person ' || p_person || ' does not have a canonical alias set';
		END LOOP;

		-- external destinations mogen niet verwijzen naar een persons.mailaddress
		FOR e_alias, p_person, p_address IN SELECT e.mail_alias, p.person, p.mailaddress
			FROM external_destinations e
				JOIN persons p ON lower(p.mailaddress) = lower(e.mailaddress)
		LOOP
			-- fixbaar
			m_address := address FROM mail_addresses WHERE mail_alias = e_alias;
			RETURN NEXT 'Alias ' || m_alias || ' points to the external address (' || p_address || ') of person ' || p_person;
		END LOOP;

		-- aliassen moeten minstens één destination hebben
		FOR m_alias IN SELECT m.mail_alias FROM mail_aliases m
			EXCEPT ALL SELECT mail_alias FROM internal_destinations
			EXCEPT ALL SELECT mail_alias FROM external_destinations
			EXCEPT ALL SELECT mail_alias FROM personal_destinations
		LOOP
			m_address := address FROM mail_addresses WHERE mail_alias = m_alias;
			IF ifix THEN
				DELETE FROM mail_aliases WHERE mail_alias = m_alias;
				RETURN NEXT 'Functional mail alias ' || m_address || ' deleted because it has no destinations';
			ELSE
				RETURN NEXT 'Functional mail alias ' || m_address || ' has no destinations';
			END IF;
		END LOOP;
	END;
$$ LANGUAGE plpgsql;

COMMIT;
