#! /usr/bin/env perl

# $Id: intreg2dhcp.pl 39802 2013-07-16 14:45:16Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/intreg2dhcp/intreg2dhcp.pl $

use strict;
use warnings FATAL => 'all';
use autodie;

use POSIX;
use IO::File;
use XML::LibXML;
use Errno qw(:POSIX);

my $statedir = "$ENV{HOME}/dhcp3";
my $dhcpdir = "$ENV{HOME}/dhcp3";

my $nodefile = "$statedir/nodes.txt"; # read, dynamic
my $vlanfile = "$statedir/vlans.xml"; # read, dynamic
my $vlanschema = "$statedir/vlans.xsd"; # read, static

my $dhcpfile = "$dhcpdir/dhcpd.conf"; # write

# available fields in the nodes.txt CSV file:
my @nheaders = qw(name mac ip);

umask 022;

my @commit;

sub macnormalize {
	my $mac = lc $_[0];
	$mac =~ tr/a-f0-9//cd;
	die "malformed MAC address '$_[0]'\n"
		unless length $mac == 12;
	return join ':', unpack('A2'x6, $mac);
}

sub ip2number {
	die 'wtf' unless defined $_[0];
	my @octets = reverse(split '\.', $_[0]);
	my $res = 0;
	my $shl = 24;
	while($#octets) {
		$res |= (pop @octets) << $shl;
		$shl -= 8
	}
	$res |= $octets[0]
}

sub number2ip {
	my $ip = $_[0];
	return join '.',
		$ip >> 24, ($ip >> 16) & 255, ($ip >> 8) & 255, $ip & 255;
}

sub number2net {
	my $ip = $_[0];
	return join '.',
		$ip >> 24, ($ip >> 16) & 255, ($ip >> 8) & 255;
}

sub number2rev {
	my $ip = $_[0];
	return join '.',
		($ip >> 8) & 255, ($ip >> 16) & 255, $ip >> 24;
}

sub bits2mask {
	return 0xFFFFFFFF - ((1 << (32 - $_[0])) - 1);
}

sub invert {
	return 0xFFFFFFFF ^ $_[0];
}

my %vlans;

do {
	my $parser = new XML::LibXML;
	$parser->pedantic_parser(1);
	$parser->no_network(1);
	$parser->keep_blanks(0);
	$parser->line_numbers(1);

	my $doc = $parser->parse_file($vlanfile);

	my $schema = new XML::LibXML::Schema(location => $vlanschema);

	$schema->validate($doc);

	my $xpath = new XML::LibXML::XPathContext;
	$xpath->registerNs(intreg => 'http://xml.uvt.nl/intreg/vlans');

	foreach my $vlan ($xpath->findnodes('/intreg:vlans/intreg:vlan', $doc)) {
		my %record = (nodes => []);

		$record{name} = $xpath->findvalue('intreg:name', $vlan);
		$record{network} = ip2number($xpath->findvalue('intreg:network', $vlan));
		$record{netmask} = bits2mask($xpath->findvalue('intreg:mask', $vlan));
		if(my $gw = $xpath->findvalue('intreg:gateway', $vlan)) {
			$record{gateway} = ip2number($gw);
		}

		if(my ($pool) = $xpath->findnodes('intreg:pool', $vlan)) {
			$record{pool}{start} = ip2number($xpath->findvalue('intreg:start', $pool));
			if(my $end = $xpath->findvalue('intreg:end', $pool)) {
				$record{pool}{end} = ip2number($end);
			} else {
				$record{pool}{end} = ($record{network} | invert($record{netmask})) - 1;
			}
		}

		if(my ($dns) = $xpath->findnodes('intreg:dns', $vlan)) {
			$record{resolvers} = [map { $_->textContent } $xpath->findnodes('intreg:server', $dns)];
			$record{domain} = $xpath->findvalue('intreg:search', $dns);
		}

		if(my ($wins) = $xpath->findnodes('intreg:wins', $vlan)) {
			$record{wins} = [map { $_->textContent } $xpath->findnodes('intreg:server', $wins)];
		}

		if(my ($boot) = $xpath->findnodes('intreg:boot', $vlan)) {
			$record{bootserver} = $xpath->findvalue('intreg:server', $boot);
			if(my $filename = $xpath->findvalue('intreg:file', $boot)) {
				$record{bootfile} = $filename;
			}
		}

		if(my ($voip) = $xpath->findnodes('intreg:voip', $vlan)) {
			my @drs = map { $_->textContent } $xpath->findnodes('intreg:drs', $voip);
			$record{drs} = \@drs if @drs;
			my @tservers = map { $_->textContent } $xpath->findnodes('intreg:tserver', $voip);
			$record{tservers} = \@tservers if @tservers;
		}

		die "Duplicate vlan $record{name}\n"
			if exists $vlans{$record{name}};
		$vlans{$record{name}} = \%record;
	}
};

sub findvlan {
	my $ip = shift;

	foreach(values %vlans) {
		my $network = $_->{network};
		return $_ if $ip >= $network && $ip <= ($network | invert($_->{netmask}));
	}
	die "unknown subnet for ", number2ip($ip), "\n";
}

open my $nodesfh, '<', $nodefile;

while(<$nodesfh>) {
	chomp;
	my @fields = split ',';
	my %record = ();

	die "Wrong number of fields in input line $.\n"
		unless @fields == @nheaders;

	for(my $i = 0; $i < @fields; $i++) {
		$record{$nheaders[$i]} = $fields[$i]
			if $fields[$i] ne '';
	}
	foreach(@nheaders) {
		die "Required field '$_' empty on line $.\n"
			unless exists $record{$_};
	}
	$record{name} = lc $record{name};
	$record{mac} = macnormalize($record{mac});
	$record{ip} = ip2number($record{ip});
	$record{vlan} = findvlan($record{ip});

	push @{$record{vlan}{nodes}}, \%record;
}

close $nodesfh;

open my $dhcpfh, '>', "$dhcpfile.new";

print $dhcpfh <<'EOH';
# Dynamically generated file, do not edit.
# Generated by $Id: intreg2dhcp.pl 39802 2013-07-16 14:45:16Z wsl $
# Program located at $URL: https://svn.uvt.nl/its-id/trunk/sources/intreg2dhcp/intreg2dhcp.pl $
EOH

foreach my $name (sort keys %vlans) {
	my $v = $vlans{$name};
	print $dhcpfh "\nshared-network uvt-$name {\n",
			"\tsubnet ", number2ip($v->{network}),
			' netmask ', number2ip($v->{netmask}), " {}\n",
			"\toption subnet-mask ", number2ip($v->{netmask}), ";\n",
			"\toption broadcast-address ", number2ip($v->{network} | invert($v->{netmask})), ";\n";

	if(my $gw = $v->{gateway}) {
		print $dhcpfh "\toption routers ", number2ip($v->{gateway}), ";\n";
	}

	if(my $pool = $v->{pool}) {
		my $start = number2ip($pool->{start});
		my $end = number2ip($pool->{end});
		if($start eq $end) {
			print $dhcpfh "\trange $start;\n";
		} else {
			print $dhcpfh "\trange $start $end;\n";
		}
	}

	if(my $dns = $v->{resolvers}) {
		print $dhcpfh
				"\toption domain-name-servers ", join(', ', @$dns), ";\n",
				"\toption domain-name \"$v->{domain}\";\n";
	}

	if(my $boot = $v->{bootserver}) {
		print $dhcpfh "\tnext-server $boot;\n";
	}

	if(my $file = $v->{bootfile}) {
		$file =~ s/["\\]/\\$&/g;
		print $dhcpfh "\tfilename \"$file\";\n";
	}

	if(my $drs = $v->{drs}) {
		print $dhcpfh "\toption voip-drs ", join(', ', @$drs), ";\n";
	}

	if(my $tservers = $v->{tservers}) {
		print $dhcpfh "\toption voip-tservers ", join(', ', @$tservers), ";\n";
	}

	if(my $wins = $v->{wins}) {
		print $dhcpfh "\toption netbios-name-servers ", join(', ', @$wins), ";\n";
	}

	print $dhcpfh "}\n";

	foreach(@{$v->{nodes}}) {
		print $dhcpfh "\nhost $_->{name} {\n",
				"\thardware ethernet $_->{mac};\n",
				"\tfixed-address ", number2ip($_->{ip}), ";\n",
				"}\n";
	}
}

close $dhcpfh;

push @commit, $dhcpfile;

exit if exists $ENV{DEBUG};

# Commit all files

foreach(@commit) {
	eval { unlink "$_.old" };
	die $@ if $@ && $@->errno != ENOENT;
	eval { link $_, "$_.old" };
	die $@ if $@ && $@->errno != ENOENT;
	rename "$_.new", $_;
	eval { unlink "$_.old" };
	die $@ if $@ && $@->errno != ENOENT;
}
