package UvT::Brebo;

# $Id: Brebo.pm 41894 2014-07-08 13:52:35Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/brebo/lib/UvT/Brebo.pm $

use Clarity -self;

use IO::File;
use Text::CSV_XS;
use POSIX qw(strftime ENOENT);
use Encode qw(decode encode_utf8 decode_utf8);

field debug => undef;
field now => sub { time };
field commits => {};
field csv => sub { Text::CSV_XS->new({ binary => 1 }) };

field records => [];
field nics => [];
field connections => {};
field allips => {};
field vlans => {};
field subnets => [];
field wmbnames => {};
field state => {};
field tpgids => {};

field ensure_consistent => sub { shift->analyze; 1 };
field assume_inconsistent => sub { shift->ensure_consistent_reset; 1 };

use constant state_headers => [qw/id mac ip atime mtime/];
use constant required_headers => [
	'naam onderwijsinstelling',
	'studentnummer onderwijsinstelling',
];

sub makehostname() {
	return sprintf('dyn-%08x', ip2number($_[0]->{ip}))
}

sub make_nicid() {
	my $rec = shift;
	my $required = required_headers;
	return join $;, (map $rec->{$_}, @$required), @_
}

sub macnormalize() {
	my $mac = lc $_[0];
	$mac =~ tr/a-f0-9//cd;
	die "geen correct MAC-adres: '$_[0]'\n"
		unless length $mac == 12;
	return join ':', unpack('A2'x6, $mac);
}

sub mac2wim() {
	my $mac = uc $_[0];
	$mac =~ tr/A-F0-9//cd;
	die "geen correct MAC-adres: '$_[0]'\n"
		unless length $mac == 12;
	return join '-', unpack('A2'x6, $mac);
}

sub mac2cisco() {
	my $mac = lc $_[0];
	$mac =~ tr/a-f0-9//cd;
	die "geen correct MAC-adres: '$_[0]'\n"
		unless length $mac == 12;
	return join '.', unpack('A4'x3, $mac);
}

sub mac2radius() {
	my $mac = lc $_[0];
	$mac =~ tr/a-f0-9//cd;
	die "geen correct MAC-adres: '$_[0]'\n"
		unless length $mac == 12;
	return $mac;
}

sub macpretty() {
	local $_ = $_[0];
	return '' unless defined && length;
	return eval { macnormalize(uc) } // '(ongeldig)';
}

sub ip2number() {
	confess('undefined input') unless defined $_[0];
	my @octets = reverse(split '\.', $_[0]);
	my $res = 0;
	my $shl = 24;
	while($#octets) {
		$res |= (pop @octets) << $shl;
		$shl -= 8
	}
	$res |= $octets[0]
}

sub number2ip() {
	my $ip = $_[0];
	return join '.',
		($ip >> 24) & 255, ($ip >> 16) & 255, ($ip >> 8) & 255, $ip & 255;
}

sub number2net() {
	my $ip = $_[0];
	return join '.',
		($ip >> 24) & 255, ($ip >> 16) & 255, ($ip >> 8) & 255;
}

sub number2rev() {
	my $ip = $_[0];
	return join '.',
		($ip >> 8) & 255, ($ip >> 16) & 255, ($ip >> 24) & 255;
}

sub cidr2range() {
	my ($ip, $bits) = split '/', $_[0];
	my $mask = $bits ? ~((1 << (32-$bits)) - 1) : 0;
	$ip = ip2number($ip);
	return bless {
		from => $ip & $mask,
		to => $ip | ~$mask,
		mask => $mask,
		ip => $ip,
		bits => $bits
	}, "range";
}

sub barescape() {
	my $x = $_[0];
	return '' unless defined $x;
	$x =~ s/[\\|]/\\$&/g;
	return $x
}

my %htmlescapechars = (
	'&' => '&amp;',
	'<' => '&lt;',
	'>' => '&gt;',
	'"' => '&quot;'
);

my $htmlescapechars = quotemeta(join('', keys %htmlescapechars));

sub htmlescape() {
	my $x = $_[0];
	return '' unless defined $x;
	return $x =~ s/[$htmlescapechars]/$htmlescapechars{$&}/egor;
}

sub addfile {
	my $file = shift;

	$self->assume_inconsistent;

	my $records = $self->records;
	my $required = required_headers;
	my $csv = $self->csv;

	my $fh = new IO::File($file, '<')
		or die "$file: $!\n";

	$csv->parse($fh->getline)
		or die "kan eerste regel niet doorgronden -- is dit wel het juiste bestand?\n";

	my @headers;
	@headers = map { decode('windows-1252', $_) } $csv->fields;
	my %headers;
	@headers{@headers} = ();

	foreach(@$required) {
		die "vereiste header '$_' niet gevonden -- is dit wel het juiste bestand?\n"
			unless exists $headers{$_};
	}

	for(;;) {
		local $_ = $fh->getline;
		last unless defined;
		eval {
			$csv->parse($_) or die "CSV opmaak klopt niet -- is dit wel het juiste bestand?\n";

			my @fields = map { decode('windows-1252', $_) } $csv->fields;
			my %record = (file => $file);

			die "aantal velden klopt niet -- is het bestand wel volledig?\n"
				unless @fields == @headers;

			for(my $i = 0; $i < @fields; $i++) {
				$record{$headers[$i]} = $fields[$i]
					if $fields[$i] ne '';
			}
			foreach(@$required) {
				die "vereist veld '$_' is leeg -- zijn alle gegevens wel gecontroleerd?\n"
					unless exists $record{$_};
			}
			push @$records, (bless \%record, "rec");
		};
		if(my $err = $@) {
			my $line = $fh->input_line_number;
			die "line $line: $err\n";
		}
	}
	$fh->eof or die "$file: read(): $!\n";
	$fh->close;

	return;
}

sub findsubnet {
	my $ip = ip2number($_[0]);
	my $subnets = $self->subnets;
	foreach(@$subnets) {
		return $_ if $ip > $_->{from} && $ip < $_->{to}
	}

	return undef;
}

sub allocate {
	my $rec = $_[0];
	my $vlan = $rec->{vlan};
	my ($from, $to) = (0, 0);

	my $allips = $self->allips;
	my $state = $self->state;

	foreach(@{$vlan->{cidr}}) {
		($from, $to) = ($_->{from}, $_->{to});
		$from++; # skip network address
		$from++; # skip router address
		while($from < $to) {
			last unless exists $allips->{$from}
				|| ($from & 0xFF) == 0 # skip .0
				|| ($from & 0xFF) == 255; # skip .255
			$from++
		}
		last unless $from == $to
	}
	if($from == $to) {
		return undef
			unless scalar @{$vlan->{recycle}};
		my $donor = pop @{$vlan->{recycle}};
		delete $state->{$donor->{id}};
		$from = ip2number($donor->{ip})
	}
	$allips->{$from} = $rec;
	return number2ip($from)
}

sub analyze {
	$self->nics_reset;
	$self->connections_reset;
	$self->commits_reset;

	my $nics = $self->nics;
	my $vlans = $self->vlans;
	my $tpgids = $self->tpgids;
	my $connections = $self->connections;
	my $records = $self->records;
	my $state = $self->state;
	my $sheaders = state_headers;
	my $allips = $self->allips;
	my $now = $self->now;

	foreach(@$records) {
		my @macs;

		push @macs, macnormalize($_->{'Mac-adres1'})
			if exists $_->{'Mac-adres1'};
		push @macs, macnormalize($_->{'Mac-adres2'})
			if exists $_->{'Mac-adres2'};

		for(my $i = 0; $i < @macs; $i++) {
			push @$nics, {
				id => make_nicid($_, $i),
				rec => $_,
				mac => $macs[$i]
			}
		}
		$connections->{$_->{file}} += @macs;

		my $instelling = $_->{'naam onderwijsinstelling'};
		my $tpgid = $tpgids->{$instelling} // $tpgids->{default};
		die "onbekende onderwijsinstelling '$instelling' gevonden voor MAC-adres $macs[0]\n"
			unless defined $tpgid;
	}

	# controleer of MACs uniek zijn
	my %nicids;
	my %nicmacs;

	foreach(@$nics) {
		my $id = join ', ', split($;, $_->{id});
		die "studentnummer / instelling niet uniek ($id)\n"
			if exists $nicids{$_->{id}};
		$nicids{$_->{id}} = undef;
		my $mac = $_->{mac};
		die "MAC-adres niet uniek ($mac)\n"
			if exists $nicmacs{$mac};
		$nicmacs{$mac} = undef
	}

	die "no default vlan defined\n"
		unless exists $vlans->{default};

	my $wmbnames = $self->wmbnames;
	foreach(@$records) {
		my $vlan = lc $_->{'naam onderwijsinstelling'};
		$vlan = '' unless exists $wmbnames->{$vlan};
		$_->{vlan} = $wmbnames->{$vlan};
	}

	foreach(@$nics) {
		if(my $st = $state->{$_->{id}}) {
			$st->{atime} = $now;
		}
	}

	foreach(values %$state) {
		my $subnet = $self->findsubnet($_->{ip});
		confess('unknown subnet')
			unless defined $subnet;
		$_->{subnet} = $subnet;
		$_->{vlan} = $subnet->{vlan};
		push @{$_->{vlan}{recycle}}, $_
			if $_->{atime} < $now;
		$allips->{ip2number($_->{ip})} = $_
	}

	foreach(values %$vlans) {
		$_->{recycle} = [sort { $b->{atime} <=> $a->{atime} } @{$_->{recycle}}]
	}

	foreach(@$nics) {
		if(my $st = $state->{$_->{id}}) {
			$_->{state} = $st;
		} else {
			my %new = (
				id => $_->{id},
				mac => $_->{mac},
				vlan => $_->{rec}{vlan},
				atime => $now,
				ctime => $now,
				mtime => $now,
			);
			$new{ip} = $self->allocate(\%new);
			die "kon geen vrij adres vinden in VLAN '$new{vlan}{id}' -- neem contact op met UvT LIS Unix\n"
				unless defined $new{ip};
			$new{subnet} = $self->findsubnet($new{ip});
			foreach(@$sheaders) {
				confess("required field '$_' empty on new entry")
					unless exists $new{$_};
			}
			my $rec = bless \%new, "newstate";
			$state->{$_->{id}} = $rec;
			$_->{state} = $rec;
		}
		push @{$_->{state}{subnet}{nics}}, $_;
	}

	return;
}

sub read_file {
	my $type = shift;
	my $file = shift;
	my $sub = shift;

	eval {
		my $fh = new IO::File($file, '<')
			or die "open(): $!\n";

		for(;;) {
			local $_ = $fh->getline;
			last unless defined;
			eval { $sub->($_) };
			if(my $err = $@) {
				my $line = $fh->input_line_number;
				die "line $line: $err\n";
			}
		}
	};
	die "$type file $file: $@" if $@;
	return;
}

sub read_csv_file {
	my $type = shift;
	my $file = shift;
	my $headers = shift;
	my $sub = shift;

	my $csv = $self->csv;

	return $self->read_file($type, $file, sub {
		my $line = shift;
		$csv->parse($line)
			or die "bad CSV syntax\n";

		my @fields = map { decode_utf8($_, Encode::FB_CROAK) } $csv->fields;

		die "wrong number of fields\n"
			unless @fields == @$headers;

		$sub->(@fields);
	});
}

sub read_vlansfile {
	my $file = shift;

	$self->assume_inconsistent;

	my $vlans = $self->vlans;
	my $wmbnames = $self->wmbnames;

	my @headers = qw/id wmbname name comment/;
	my %lowercasefields = (wmbname => undef);

	return $self->read_csv_file('vlans', $file, \@headers, sub {
		my %record;
		for(my $i = 0; $i < @_; $i++) {
			if(exists $lowercasefields{$headers[$i]}) {
				$record{$headers[$i]} = lc $_[$i]
			} elsif($_[$i] ne '') {
				$record{$headers[$i]} = $_[$i]
			}
		}

		foreach(@headers) {
			die "required field '$_' empty\n"
				unless exists $record{$_};
		}

		die "VLAN '$record{id}' multiply defined\n"
			if exists $vlans->{$record{id}};

		$record{recycle} = [];

		my $rec = bless \%record, "vlan";
		$vlans->{$record{id}} = $rec;
		$wmbnames->{$record{wmbname}} = $rec;
	});
}

sub read_subnetfile {
	my $file = shift;

	$self->assume_inconsistent;

	my $vlans = $self->vlans;
	my $subnets = $self->subnets;
	my @headers = qw/id cidr/;

	return $self->read_csv_file('subnets', $file, \@headers, sub {
		my %record;

		for(my $i = 0; $i < @_; $i++) {
			$record{$headers[$i]} = $_[$i]
				if !$i || $_[$i] ne '';
		}

		foreach(@headers) {
			die "required field '$_' empty\n"
				unless exists $record{$_};
		}

		my $cidr = cidr2range($record{cidr});
		my $vlan = $vlans->{$record{id}};
		$cidr->{vlan} = $vlan;
		$cidr->{nics} = [];
		push @{$vlan->{cidr}}, $cidr;
		push @$subnets, $cidr;
	});
}

sub read_tpgidfile {
	my $file = shift;

	$self->assume_inconsistent;

	my $tpgids = $self->tpgids;
	my @headers = qw/id name/;

	return $self->read_csv_file('tpgids', $file, \@headers, sub {
		die "duplicate name '$_[1]'\n"
			if exists $tpgids->{$_[1]};
		$tpgids->{$_[1]} = $_[0];
	});
}

sub read_statefile {
	my $file = shift;

	$self->assume_inconsistent;

	my $state = $self->state;
	my $csv = $self->csv;
	my $sheaders = state_headers;

	return $self->read_csv_file('state', $file, $sheaders, sub {
		my %record;

		for(my $i = 0; $i < @_; $i++) {
			$record{$sheaders->[$i]} = $_[$i]
				if $_[$i] ne '';
		}

		foreach(@$sheaders) {
			die "required field '$_' empty\n"
				unless exists $record{$_};
		}

		$state->{$record{id}} = bless \%record, "state";
	});
}

sub write_file {
	my $type = shift;
	my $file = shift;

	my $contents = shift;

	eval {
		my $fh = new IO::File("$file,new", '>')
			or die "open(): $!\n";

		foreach(@$contents) {
			$fh->write($_)
				or die "write(): $!\n";
		}

		$fh->flush or die "write(): $!\n";
		$fh->sync or die "fsync(): $!\n";
		$fh->close or die "write(): $!\n";;

		undef $self->commits->{$file};
	};
	die "$type file $file,new: $@" if $@;
	return;
}

sub write_csv_file {
	my $type = shift;
	my $file = shift;

	my $contents = shift;

	my $csv = $self->csv;

	my @contents = map {
		$csv->combine(map { encode_utf8($_ // '') } @$_)
			or die "couldn't combine record\n";
		$csv->string."\015\012"
	} @$contents;

	return $self->write_file($type, $file, \@contents);
}

sub write_statsfile {
	$self->ensure_consistent;
	my $connections = $self->connections;
	my @contents = [qw(bestand aantal)];
	keys %$connections;
	while(my @e = each %$connections) {
		push @contents, \@e;
	}
	return $self->write_csv_file('stats', shift, \@contents);
}

sub write_statefile {
	$self->ensure_consistent;
	my $sheaders = state_headers;
	my $state = $self->state;
	my @contents = map { [@$_{@$sheaders}] } values %$state;
	return $self->write_csv_file('state', shift, \@contents);
}

sub write_certfile {
	$self->ensure_consistent;

	my $records = $self->records;
	my $state = $self->state;

	my @fields = (
		'voornaam', 'voorvoegsels', 'achternaam',
		'Mac-adres1', 'Mac-adres2',
		'naam onderwijsinstelling', 'studentnummer onderwijsinstelling',
		'emailadres onderwijsinstelling'
	);

	my @contents = map {
		my $rec = $_;
		[ @$_{@fields}, map { my $n = make_nicid($rec, $_); exists $state->{$n} ? $state->{$n}{ip} : '' } (0, 1)]
	} @$records;

	return $self->write_csv_file('cert', shift, \@contents);
}

sub write_radusersfile {
	$self->ensure_consistent;
	my $tpgids = $self->tpgids;
	my $nics = $self->nics;

	my @contents = ("# File dynamically generated by $0, do not edit.\n");

	foreach my $n (@$nics) {
		my $mac = mac2radius($n->{mac});
		my $rec = $n->{rec};
		my $instelling = $rec->{'naam onderwijsinstelling'};
		my $tpgid = $tpgids->{$instelling} // $tpgids->{default};

		push @contents, "\n$mac\n\tTunnel-Private-Group-ID = \"$tpgid\",\n\tSession-Timeout = \"43200\"\n";
	}
	return $self->write_file('radusers', shift, \@contents);
}

sub write_pcshopfile {
	$self->ensure_consistent;
	my $headerfile = shift;
	my $file = shift;
	my $footerfile = shift;
	my @content;

	$self->read_file('header', $headerfile, sub { push @content, decode_utf8(shift, Encode::FB_CROAK) });

	push @content, "<table class=\"wit\">\n",
		"<tr><th>Naam</th><th>Email</th><th>Studentnummer</th><th>MAC 1</th><th>MAC 2</th><th>Adres</th><th>IP 1</th><th>IP 2</th></tr>\n";

	my $state = $self->state;
	my $records = $self->records;

	foreach my $l (@$records) {
		my @ips = map { make_nicid($l, $_) } (0, 1);

		my @list = map { htmlescape($_) } (
			join(' ', grep { defined } @$l{qw/voornaam voorvoegsels achternaam/}),
			$l->{'emailadres onderwijsinstelling'},
			join(' ', grep { defined } @$l{('naam onderwijsinstelling', 'studentnummer onderwijsinstelling')}),
			(map { macpretty $_ } @$l{('Mac-adres1', 'Mac-adres2')}),
			join(' ', grep { defined } @$l{qw/straat huisnummer/}),
			(map { exists $state->{$_} ? $state->{$_}->{ip} : '' } @ips)
		);
		push @content, "<tr><td>", join('</td><td>', @list), "</td></tr>\n";
	}
	push @content, "</table>\n",
		"<p>Laatst bijgewerkt: ", strftime('%Y-%m-%d %H:%M:%S', localtime), ".</p>\n",

	$self->read_file('footer', $footerfile, sub { push @content, decode_utf8(shift, Encode::FB_CROAK) });

	foreach(@content) {
		utf8::encode($_);
	}

	return $self->write_file('pcshop', $file, \@content);
}

sub write_dhcpfile {
	$self->ensure_consistent;

	my $vlans = $self->vlans;

	my @content;

	push @content, "# Dynamically generated file, do not edit.\n";

	foreach my $vlan (values %$vlans) {
		push @content, "\nshared-network ", $vlan->{id}, " {\n";
		my $lf = '';
		foreach(@{$vlan->{cidr}}) {
			push @content, $lf, "\tsubnet ", number2ip($_->{from}),
					' netmask ', number2ip($_->{mask}), " {\n",
					"\t\toption subnet-mask ", number2ip($_->{mask}), ";\n",
					"\t\toption broadcast-address ",
					number2ip($_->{from} | ~$_->{mask}), ";\n",
					"\t\toption routers ", number2ip($_->{from} + 1), ";\n",
					"\t}\n";
			$lf = "\n";
		}
		push @content, "}\n";

		foreach(@{$vlan->{cidr}}) {
			foreach(@{$_->{nics}}) {
				push @content, "\nhost ", makehostname($_->{state}), " {\n",
						"\thardware ethernet $_->{mac};\n",
						"\tfixed-address $_->{state}{ip};\n",
						"}\n";
			}
		}
	}

	return $self->write_file('dhcp', shift, \@content);
}

sub write_accessfile {
	$self->ensure_consistent;

	my $subnets = $self->subnets;

	my %content;

	foreach(@$subnets) {
		for(my $ip = $_->{from}; $ip < $_->{to}; $ip += 256) {
			my $f = $ip & 0xFFFFFF00;
			my $t = $f + 255;
			$f = $_->{from} + 1 if $f <= $_->{from};
			my $net = number2net $f;
			undef $content{"$net OK\n"};
		}
	}
	return $self->write_file('access', shift, [keys %content]);
}

sub write_uvtnetfile {
	my @uvtnetheaders = (
		'achternaam',
		'ingangsdatum internet', 'einddatum internet',
		'Mac-adres1', 'Mac-adres2',
		'straat', 'huisnummer', 'vlan'
	);

	my $records = $self->records;

	my @content;

	foreach(@$records) {
		my %list = %{$_};
		$list{'Mac-adres1'} = mac2wim($list{'Mac-adres1'})
			if exists $list{'Mac-adres1'} && defined $list{'Mac-adres1'};
		$list{'Mac-adres2'} = mac2wim($list{'Mac-adres2'})
			if exists $list{'Mac-adres2'} && defined $list{'Mac-adres2'};
		$list{vlan} = $list{vlan}->{name};
		push @content, encode_utf8(join('|', map { if(defined){s/[|\\]/\\$&/g}else{$_ = ''}; $_ }
				@list{@uvtnetheaders})), "\n";
	}

	return $self->write_file('uvtnet', shift, \@content);
}

sub write_vmpsfile {
	$self->ensure_consistent;

	my $nics = $self->nics;
	my @content;
	foreach(@$nics) {
		my $mac = mac2cisco $_->{mac};
		my $vlan = $_->{state}{vlan}{name};
		push @content, "address $mac vlan-name $vlan\n";
	}
	return $self->write_file('vmps', shift, \@content);
}

sub commit {
	$self->ensure_consistent;

	foreach(keys %{$self->commits}) {
		unlink "$_,old"
			or $! == ENOENT
			or die "unlink('$_,old'): $!\n";
		link $_, "$_,old"
			or $! == ENOENT
			or die "link('$_', '$_,old'): $!\n";
		rename "$_,new", $_
			or die "rename('$_,new', '$_'): $!\n";
		unlink "$_,old"
	}

	$self->commits_reset;
}
