#! /usr/bin/perl

# $Id: xyzzy.t 42589 2014-11-21 09:44:55Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/t/xyzzy.t $

use strict;
use warnings FATAL => 'all';

package Dummy::LDAP::Result;

use Clarity -self;

field _values;

sub dn { return 'uid='.$self->get_value('uid') }
sub get_value { return $self->_values->{$_[0]} }

package Dummy::LDAP::Schema;

use Clarity -base;

sub attribute {
	return {name => $_[1]};
}

package Net::LDAP;

use Clarity -base;
use Data::Dumper;

sub bind {
	my ($self, $uid, undef, $pwd) = @_;
	die "bad password. go sit in corner.\n" if defined $uid && $pwd ne 'hunter2';
	return;
}

sub search {
	my ($self, %args) = @_;
	my $filter = $args{filter} // confess("unable to reverse engineer filter in test suite");
	my $equalityMatch = $filter->{equalityMatch} // confess("unable to reverse engineer filter in test suite");
	my $assertionValue = $equalityMatch->{assertionValue} // confess("unable to reverse engineer filter in test suite");
	my $values = $assertionValue eq 'bob' ? {uid => $assertionValue}
		: $assertionValue eq 'jane' ? {uid => $assertionValue, expiry => 1}
		: die "internal error in test suite: unknown uid '$assertionValue'";
	return new Clarity(entries => new Dummy::LDAP::Result(_values => $values), count => 1);
}

sub schema {
	return new Dummy::LDAP::Schema;
}

sub AUTOLOAD { warn our $AUTOLOAD }

package GSSAPI;

use Clarity;
use Exporter qw(import);

our @EXPORT_OK = qw(GSS_C_INDEFINITE GSS_C_NO_OID_SET GSS_C_ACCEPT GSS_C_NO_CHANNEL_BINDINGS);
our @EXPORT = @EXPORT_OK;

use constant GSS_C_INDEFINITE => undef;
use constant GSS_C_NO_OID_SET => undef;
use constant GSS_C_ACCEPT => undef;
use constant GSS_C_NO_CHANNEL_BINDINGS => undef;

package GSSAPI::Context;

use Clarity -self;

sub accept {
	$_[3] = new GSSAPI::Cred;
	return undef;
}

package GSSAPI::OID;

use constant gss_nt_krb5_name => undef;

package GSSAPI::Name;

sub import {
	$_[1] = 1;
	return undef;
}

package GSSAPI::Cred;

use Clarity -base;

sub acquire_cred {
	return undef;
}

sub display {
	our $_username;
	$_[1] = "$_username\@example.org";
	return undef;
}

package main;

use Test::More;
use Test::Xyzzy qw(xyzzy_from_hash xyzzy_request xyzzy_request_as_cgi);
use XML::LibXML;
use Aselect::Util qw(aselect_split);
use Data::Dumper;

my $app = xyzzy_from_hash(
	Application => 'Aselect',
	AselectSecret => 'secret',
	AselectServerID => 'sso.example.com',
	AselectRequestor => 'frobber',
	LDAPBase => 'foo',
	LDAPExpiryAttributes => [qw(expiry expiry uid)],
	KerberosPrincipal => 'HTTP/sso.example.org@DOMAIN.EXAMPLE.ORG',
	StylesheetDir => 'etc/stylesheets',
	AuthFailTimeout => 0,
);

my $handler = $app->handler;

isa_ok($handler, 'Xyzzy::Handler');

sub parse_xml {
	my $parser = new XML::LibXML;
	$parser->validation(0);
	$parser->load_ext_dtd(0);
	$parser->expand_entities(0);
	$parser->pedantic_parser(1);
	return $parser->parse_string(shift);
}

sub bake_cookies {
	my %cookies = @_;
	my @cookies;
	while(my ($name, $value) = each %cookies) {
		push @cookies, "$name=$value";
	}
	return join('; ', @cookies);
}

sub parse_cookies {
	return map { /^Set-Cookie:\s+([^;=]+)=([^;]+)/ ? ($1, $2) : () } @_;
}

sub parse_cgi {
	my ($headers, $body) = split(/\015\012\015\012/, shift, 2);
	my @headers = split(/\015\012/, $headers);
	return $body, @headers;
}

my $bigblob = 'YEIGBisGAQUFAqA4MDagMDAuBgkqhkiC9xIBAgIGCSqGSIb3EgECAgYKKwYBBAGCNwICHgYKKwYBBAGCNwICCqICBAA=';

do {
	my $res = xyzzy_request($handler, '/status');
	like($res, qr{Status overview}, "output contains stuff from the templates");
};

do {
	my $res = xyzzy_request_as_cgi($handler, '/login', 'raw');
	my ($body, @headers) = parse_cgi($res);
	my %cookies = parse_cookies(@headers);
	my $xml = parse_xml($body);
	my $lt = $xml->findvalue('/password/lt');

	foreach my $remember ('true', 'false') {
		foreach my $setting (undef, 'true', 'false') {
			foreach my $service (undef, 'http://example.com') {
				local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /bad password/ };
				my %c = %cookies;
				$c{remember} = $setting if defined $setting;
				my $param = $remember eq 'true' ? '&remember=$remember' : '';
				$param .= "&service=$service" if defined $service;
				my $effective_remember = defined $service ? $remember : 'true';
				my $effective_setting = $setting // 'true';
				my $effective_service = $service // 'http://localhost/status';
				$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=*******&lt=$lt$param", HTTP_COOKIE => bake_cookies(%c));
				($body, @headers) = parse_cgi($res);
				$xml = parse_xml($body);
				is($xml->findvalue('/*/login/remember'), $effective_remember, "remember is $effective_remember");
				is($xml->findvalue('/*/settings/remember'), $effective_setting, "remember setting is $effective_setting");
				ok(!(grep { m{^Location: \Q$effective_service} } @headers), "headers do not contain location to $effective_service");
			}
		}
	}

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'bob' using password$/ };
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt", HTTP_COOKIE => bake_cookies(%cookies, warn => 'true'));
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { $_ eq 'Location: http://localhost/status' } @headers), "headers contain location to status url");
		is($xml->findvalue('/login-success/login/url'), 'http://localhost/status', "result contains status url");
	};

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&gateway=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { $_ eq 'Location: http://example.com/foo' } @headers), "headers contain location to service url");
	is($xml->findvalue('/giveup/login/url'), 'http://example.com/foo', "result contains service url");

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'bob' using password$/ };
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt&service=http://example.com", HTTP_COOKIE => bake_cookies(%cookies));
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { m{^Location: http://example.com} } @headers), "headers contain location to service url");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com', "result contains service url");
	};

	my %noremember_cookies = (%cookies, parse_cookies(@headers));

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%noremember_cookies));
	($body, @headers) = parse_cgi($res);
	ok(!(grep { m{^Location: http://example.com/foo} } @headers), "headers do not contain location to status url after no-remember login");

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'bob' using password$/ };
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt", HTTP_COOKIE => bake_cookies(%cookies));
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { $_ eq 'Location: http://localhost/status' } @headers), "headers contain location to status url");
		is($xml->findvalue('/login-success/login/url'), 'http://localhost/status', "result contains status url");
	};

	%cookies = (%cookies, parse_cookies(@headers));

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	my $ticket = '';
	foreach(@headers) {
		if(m{^Location: http://example.com/foo\?ticket=(ST-[^&]+)}) {
			$ticket = $1;
			last;
		}
	}

	$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket");
	($body, @headers) = parse_cgi($res);
	is($body, "yes\nbob\n", "result contains uid");

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /CAS validation request requested checking/ };
		$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket&renew=true");
		($body, @headers) = parse_cgi($res);
		is($body, "no\n\n", "result does not contain uid");
	};

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&gateway=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&renew=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	is($xml->findvalue('/password/session/uid'), 'bob', "result is a password page");
	$lt = $xml->findvalue('/password/lt');

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'bob' using password$/ };
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt&service=http://example.com/foo&renew=true", HTTP_COOKIE => bake_cookies(%cookies));
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
	};

	foreach(@headers) {
		if(m{^Location: http://example.com/foo\?ticket=(ST-[^&]+)}) {
			$ticket = $1;
			last;
		}
	}

	$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket");
	($body, @headers) = parse_cgi($res);
	is($body, "yes\nbob\n", "result contains uid");

	$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket&renew=true");
	($body, @headers) = parse_cgi($res);
	is($body, "yes\nbob\n", "result contains uid");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies, warn => 'true'));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	my $nonce = $xml->findvalue('/login-success/nonce');

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&nonce=$nonce", HTTP_COOKIE => bake_cookies(%cookies, warn => 'true'));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	$res = xyzzy_request_as_cgi($handler, '/server', "request=authenticate&app_id=frobber&app_url=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	my %res = aselect_split($body);

	is($res{result_code}, '0000', "request=authenticate returned code 0000");
	my $as_url = $res{as_url};
	like($as_url, qr{^http://localhost/login}, "request=authenticate returned an url");
	$as_url =~ s{^https?://[^/]+}{};

	my $args = "raw&rid=$res{rid}&a-select-server=$res{'a-select-server'}";
	$res = xyzzy_request_as_cgi($handler, $as_url, $args, HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?.*aselect_credentials=} } @headers), "headers contain location to service url");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	do {
		$res = xyzzy_request_as_cgi($handler, '/login', "raw");
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		$lt = $xml->findvalue('/password/lt');
		%cookies = parse_cookies(@headers);
	};

	do {
		my $checked;
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'jane' using password$/; $checked = 1 };
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&remember&username=jane&password=hunter2&lt=$lt&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies));
		ok($checked, "login credentials were checked");
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { m{^Set-Cookie: session=} } @headers), "headers contain new session cookie");
		ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
	};

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^session: Short token$/i };
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&remember&username=jane&password=hunter2&lt=$lt&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies, session => 'invalid'));
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { m{^Set-Cookie: session=} } @headers), "headers contain new session cookie");
		ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	};

	do {
		my $checked;
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'jane' using SPNEGO$/; $checked = 1 };
		local $GSSAPI::Cred::_username = 'jane';
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&spnego&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies), HTTP_AUTHORIZATION => "Negotiate $bigblob");
		ok($checked, "login credentials were checked");
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
#		print STDERR "\n\n", Dumper(\%cookies, \@headers, $xml->toString(1)), "\n\n";
		ok((grep { m{^Set-Cookie: session=} } @headers), "headers contain new session cookie");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
		ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	};

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^(?:Successfully authenticated as 'jane' using SPNEGO|session: Short token)$/ };
		local $GSSAPI::Cred::_username = 'jane';
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&spnego&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies, session => 'invalid'), HTTP_AUTHORIZATION => "Negotiate $bigblob");
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
#		print STDERR "\n\n", Dumper(\%cookies, \@headers, $xml->toString(1)), "\n\n";
		ok((grep { m{^Set-Cookie: session=} } @headers), "headers contain new session cookie");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
		ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	};

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^(?:Successfully authenticated as 'jane' using SPNEGO|session: Short token)$/ };
		local $GSSAPI::Cred::_username = 'jane';
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&spnego&service=http://example.com/foo", HTTP_AUTHORIZATION => "Negotiate $bigblob");
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
#		print STDERR "\n\n", Dumper(\%cookies, \@headers, $xml->toString(1)), "\n\n";
		ok((grep { m{^Set-Cookie: session=} } @headers), "headers contain new session cookie");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
		ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
		%cookies = parse_cookies(@headers);
	};

	do {
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&spnego&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies));
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
#		print STDERR "\n\n", Dumper(\%cookies, \@headers, $xml->toString(1)), "\n\n";
		ok(!(grep { m{^Set-Cookie: session=} } @headers), "headers do not contain new session cookie");
		ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
	};

	do {
		my $checked;
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /^Successfully authenticated as 'bob' using SPNEGO$/; $checked = 1 };
		local $GSSAPI::Cred::_username = 'bob';
		$res = xyzzy_request_as_cgi($handler, '/login', "raw&spnego&service=http://example.com/foo", HTTP_AUTHORIZATION => "Negotiate $bigblob");
		ok($checked, "login credentials were checked");
		($body, @headers) = parse_cgi($res);
		$xml = parse_xml($body);
		ok((grep { m{^Set-Cookie: session=} } @headers), "headers contain new session cookie");
		ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
		is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
	};

};

done_testing();
