# $Id: LDAP.pm 42262 2014-09-18 15:18:19Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/LDAP.pm $

package Aselect::LDAP;

use Net::LDAP;
use Net::LDAP::Filter;

use Xyzzy::LDAP -self;

field expiry_conditions => sub { shift->cfg->ldap_expiry_conditions };
field expiry_conditions_canonicalized => sub {
	my $self = shift;
	my $conds = $self->expiry_conditions;
	my %canonicalized;
	@canonicalized{map { $self->canonicalize($_) // $_ } @$conds} = ();
	return [keys %canonicalized];
};

field expiry_attributes => sub { shift->cfg->ldap_expiry_attributes };
field expiry_attributes_canonicalized => sub {
	my $self = shift;
	my $conds = $self->expiry_conditions;
	my $attrs = $self->expiry_attributes;
	my %canonicalized;
	@canonicalized{map { $self->canonicalize($_) // $_ } keys %$attrs} = values %$attrs;
	return \%canonicalized;
};

# [[a b] [x y] [1 2]] ->
#   [[a x 1] [b x 1] [a y 1] [b y 1] [a x 2] [b x 2] [a y 2] [b y 2]]
sub cart() {
	my $a = [[]];
	my $b = [];
	foreach my $list (@_) {
		my @x = @$list;
		return [] unless @x;
		my $last = pop @x;
		foreach my $x (@x) {
			foreach my $e (@$a) {
				my @e = @$e;
				push @e, $x;
				push @$b, \@e;
			}
		}
		foreach my $e (@$a) {
			push @$e, $last;
			push @$b, $e;
		}
		$a = $b;
		$b = [];
	}
	return $a;
}

sub replace() {
	my ($fmt, $values, $lookup) = @_;
	@$lookup{keys %$values} = keys %$values
		unless defined $lookup;
	my $cart = cart(values %$values);
	return map {
		my %keys;
		@keys{keys %$values} = @$_;
		my $out = $fmt;
		$out =~ s/\$(?:(\$)|\{([^}]*)\}|([a-z_]\w*))/
				my $replacement = '';
				if(defined $1) {
					$replacement = '$';
				} elsif(defined $2) {
					$replacement = $keys{$lookup->{$2}};
				} elsif(defined $3) {
					$replacement = $keys{$lookup->{$3}};
				}
				$replacement
			/eig;
		$out
	} @$cart;
}

sub scan() {
	my $fmt = shift;

	my %keys;

	$fmt =~ s/\$(?:(\$)|\{([^}]*)\}|([a-z_]\w*))/
			if(defined $2) {
				undef $keys{$2};
			} elsif(defined $3) {
				undef $keys{$3};
			}
			''
		/eig;

	return [keys %keys];
}

const canon => {};

sub canonicalize {
	my $fullname = shift;

	my $canon = $self->canon;
	return $canon->{$fullname} if exists $canon->{$fullname};

	my $name = $fullname;
	$name =~ s/(;.*)//;
	my $options = $1 // '';

	if(exists $canon->{$name}) {
		my $n = $canon->{$name};
		return $canon->{$fullname} = defined $n ? $n.$options : undef;
	}

	my $schema = $self->schema;
	my $c = $schema->attribute($name);
	if(defined $c) {
		my $n = $c->{name};
		my $f = $n.$options;
		$canon->{$fullname} = $f;
		$canon->{$name} = $n;
		return $f;
	} else {
		$canon->{$name} = undef;
		return undef;
	}
}

sub attributes {
	my ($uid, $app) = @_;

	return [] unless defined $app;

	# convention: "attributes" refers to A-Select attributes,
	# while "attrs" refers to LDAP attributes
	my $attributes = $app->policy;
	return [] unless defined $attributes;
	return [] unless %$attributes;

	my $ldap = $self->connection;

	my $schema = $self->schema;
	my $canon = $self->canon;

	my %req;
	my %attrs;
	my %substs;
	ATTR: while(my ($key, $val) = each(%$attributes)) {
		my %s;
		my $attrs = scan($val);
		foreach my $a (@$attrs) {
			my $c = $self->canonicalize($a);
			next ATTR unless defined $c;
			undef $s{$c};
			undef $attrs{$c};
			$c =~ s/;.*//;
			undef $req{$c};
		}
		$substs{$key} = \%s;
	}

	my $search = $self->search($uid, [keys %req]);
	return [] unless $search->count;

	# should be only one
	foreach my $entry ($search->entries) {
		foreach my $a (keys %attrs) {
			$attrs{$a} = $entry->get_value($a, asref => 1) || [];
		}
	}

	my @res;
	while(my ($key, $val) = each(%substs)) {
		my @a = keys %$val;
		my %a; @a{@a} = @attrs{@a};
		push @res, map { $key, $_ } replace($attributes->{$key}, \%a, $canon);
	}
	return \@res;
}

field cached_uid;
field cached_result;

# Wrap the search method so we can piggy-back fetching the expiry attributes
sub search {
	my $uid = shift;
	my $attrs = shift;

	$self->cached_uid_reset;

	return undef unless defined $uid;
	$attrs = [] unless defined $attrs;
	$attrs = [$attrs] unless ref $attrs;

	# only piggy-back on searches that look like authentication
	return super unless @$attrs == 1;
	return super unless $attrs->[0] eq $self->attribute;

	# add all expiry attributes that aren't in the list yet
	my $expiry_attrs = $self->expiry_attributes_canonicalized;
	my @expiry_attrs = keys %$expiry_attrs;
	my $expiry_conds = $self->expiry_conditions_canonicalized;
	my @expiry_conds = @$expiry_conds;
	my %expiry;
	@expiry{map { lc } (@expiry_conds, @expiry_attrs)} = (@expiry_conds, @expiry_attrs);
	delete @expiry{map { lc } @$attrs};
	push @$attrs, values %expiry;

	my $res = super($uid, $attrs, @_);
	$self->cached_uid($uid);
	$self->cached_result($res);
	return $res;
}

sub expiration {
	my $uid = shift;

	my $expiry_conds = $self->expiry_conditions_canonicalized;
	return undef unless @$expiry_conds;
	my $res = $self->cached_uid_isset && lc($uid) eq lc($self->cached_uid)
		? $self->cached_result
		: $self->search($uid);

	unless($res->count == 1) {
		warn "invalid username '$uid'\n";
		return undef;
	}

	my %res;
	my $expiry_attrs = $self->expiry_attributes;
	foreach my $entry ($res->entries) {
		return undef unless grep { defined $entry->get_value($_) } @$expiry_conds;
		while(my ($attrname, $elname) = each %$expiry_attrs) {
			my @values = $entry->get_value($attrname)
				or next;
			push @{$res{$elname}}, @values;
		}
	}

	return \%res;
}
