# $Id: Transaction.pm 48567 2019-12-12 16:22:55Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki/Database/Transaction.pm $

use re '/aa';

package UvT::Kiki::Database::Transaction;

use UvT::Kiki::Database::Alias;
use UvT::Kiki::Database::Person;
use UvT::Kiki::Database::External;
use UvT::Kiki::Database::Domain;

use Clarity -self;

field db;
field cfg;

field hostresolvable_cache => {};

sub hostresolvable {
	my $host = shift;
	$host =~ s/.*\@//;
	$host = lc $host;
	return 0 unless $host =~ /^(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*(?:\.(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*)+\z/;
	my $cache = $self->hostresolvable_cache;
	return $cache->{$host}
		if exists $cache->{$host};
	my $resolver = $self->cfg->dns;
	foreach my $type (qw(MX AAAA A)) {
		my $q = $resolver->send($host, $type)
			or next;
		foreach my $rr ($q->answer) {
			return $cache->{$host} = 1
				if $rr->type eq $type;
		}
	}
	return $cache->{$host} = 0;
}

field domains_by_id => sub {
	my $self = shift;
	my %domains;
	my $res = $self->query('SELECT n.domain, n.name
		FROM domainnames n JOIN domains d USING (domain)
		ORDER BY n.domainname = d.main DESC, n.name');
	foreach my $d (@$res) {
		my ($domain, $name) = @$d;
		push @{$domains{$domain}}, $name;
	}
	my %objects;
	my %names;
	while(my ($key, $val) = each %domains) {
		my $domain = new UvT::Kiki::Database::Domain(db => $self, id => $key, names => $val);
		$objects{$key} = $domain;
		foreach my $name (@$val) {
			$names{$name} = $domain;
		}
	}
	$self->domains_by_name(\%names);
	return \%objects;
};

field domains_by_name => sub { shift->domains_by_id; return };

sub domain_by_id {
	my $id = shift;
	return $self->domains_by_id->{$id};
}

sub domain_by_name {
	my $name = shift;
	return $self->domains_by_name->{lc($name)};
}

sub record2alias {
	my $domains = $self->domains_by_id;
	my @res;
	foreach my $record (@_) {
		my ($id, $name, $domain, $addressbook, $anr, $mailbox) = @$record;
		if(defined $anr) {
			push @res, new UvT::Kiki::Database::Person(db => $self, anr => $anr, mailbox => $mailbox);
		} elsif(defined $domain) {
			$domain = $domains->{$domain} // confess("Unknown domain '$domain'");
			push @res, new UvT::Kiki::Database::Alias(db => $self, id => $id, localpart => $name, domain => $domain, addressbook => $addressbook ? 1 : 0);
		} else {
			push @res, new UvT::Kiki::Database::External(db => $self, id => $id, address => $name);
		}
	}
	return \@res;
}

sub new_alias {
	return new UvT::Kiki::Database::Alias(db => $self, id => undef, original => undef);
}

sub new_person {
	return new UvT::Kiki::Database::Person(db => $self, anr => shift, original => undef);
}

sub perform {
	my $q = $self->db->prepare_cached(shift);
	my $res = $q->execute(@_);
	$q->finish;
	return $res;
}

sub query {
	my $sql = shift;
	my $dst;
	if(ref $sql) {
		$dst = $sql;
		$sql = shift;
	}
	my $q = $self->db->prepare_cached($sql);
	$q->execute(@_);
	my $res = $q->fetchall_arrayref;
	$q->finish;
	if($dst) {
		push @$dst, @$res;
		return;
	} else {
		return $res;
	}
}

sub create_alias {
	my $alias = shift;

	my $q = $self->db->prepare_cached('INSERT INTO mail_aliases (name, domain, addressbook) VALUES (?, ?, ?) RETURNING mail_alias');
	$q->execute($alias->localpart, $alias->domain->id, $alias->addressbook ? 't' : 'f');
	my ($id) = $q->fetchrow_array;
	$q->finish;
	$alias->id($id);

	$self->add_destinations($alias);
}

sub exists_alias {
	my ($local, $domain) = @_;
	
	my $res = $self->query('SELECT EXISTS(SELECT * FROM mail_aliases WHERE lower(mail_aliases.name) = lower(?) AND mail_aliases.domain = ?)', $local, $domain->id);
	return $res->[0][0];
}

sub update_alias {
	my $alias = shift;

	my $id = $alias->id;

	$self->perform('UPDATE mail_aliases SET name = ?, domain = ?, addressbook = ? WHERE mail_alias = ?',
		$alias->localpart, $alias->domain->id, $alias->addressbook ? 't' : 'f', $id);
	$self->perform('DELETE FROM personal_destinations WHERE mail_alias = ?', $id);
	$self->perform('DELETE FROM internal_destinations WHERE mail_alias = ?', $id);
	$self->perform('DELETE FROM external_destinations WHERE mail_alias = ?', $id);

	$self->add_destinations($alias);
}

sub remove_alias {
	my $alias = shift;
	$self->perform('DELETE FROM mail_aliases WHERE mail_alias = ?', $alias->id);
}

sub remove_person {
	my $person = shift;
	$self->perform('DELETE FROM persons WHERE person = ?', $person->anr);
}

sub add_destinations {
	my $alias = shift;
	my $id = $alias->id;
	my $destinations = $alias->destinations;
	for my $d (@$destinations) {
		my $type = $d->type;
		if($type eq 'alias') {
			$self->perform('INSERT INTO internal_destinations (mail_alias, destination) VALUES (?, ?)', $id, $d->id);
		} elsif($type eq 'person') {
			$self->perform('INSERT INTO personal_destinations (mail_alias, person) VALUES (?, ?)', $id, $d->anr);
		} elsif($type eq 'external') {
			$self->perform('INSERT INTO external_destinations (mail_alias, mailaddress) VALUES (?, ?)', $id, $d->address);
		} else {
			confess("internal error: unknown type '$type'");
		}
	}
}

sub tentative_label {
	my ($local, $domain) = @_;
	return new UvT::Kiki::Database::Alias(db => $self, localpart => $local, domain => $domain, addressbook => 0);
}

sub upsert_person {
	my $person = shift;
	my $anr = $person->anr;
	my $labels = $person->labels;
	my %existing =
		map { $_->id => $_ }
		grep { $_->id_isset }
		@$labels;
	if(my $original = $person->original) {
		foreach my $label (@{$original->labels}) {
			my $id = $label->id;
			$self->perform('DELETE FROM mail_aliases WHERE mail_alias = ?', $id)
				unless exists $existing{$id};
		}

		my $mailbox = $person->mailbox;
		$self->perform('UPDATE persons SET mailaddress = ? WHERE person = ?', $mailbox, $anr)
			unless $original->mailbox eq $mailbox;
	} else {
		$self->perform('INSERT INTO persons (person, mailaddress) VALUES (?, ?)', $anr, $person->mailbox);
	}
	foreach my $label (@$labels) {
		if($label->id_isset) {
			if(my $original = $label->original) {
				my $localpart = $label->localpart;
				my $domain_id = $label->domain->id;
				$self->perform("UPDATE mail_aliases SET name = ?, domain = ? WHERE mail_alias = ?",
						$localpart, $domain_id, $label->id)
					unless $original->localpart eq $localpart && $original->domain->id == $domain_id;
			}
		} else {
			my $res = $self->query("INSERT INTO mail_aliases (name, domain, addressbook, person) VALUES (?, ?, 'f', ?)
				RETURNING mail_alias", $label->localpart, $label->domain->id, $anr);
			my $id = $res->[0][0];
			confess("internal error") unless defined $id;
			$label->id($id);
			$self->perform('INSERT INTO personal_destinations (mail_alias, person) VALUES (?, ?)', $id, $anr);
		}
	}
	$self->perform('UPDATE persons SET canonical = ? WHERE person = ?', $labels->[0]->id, $anr);
}

sub mailbox_for_anr {
	my $anr = shift;

	my $res = $self->query('SELECT p.mailaddress FROM persons p WHERE p.person = ?', $anr);

	return $res->[0][0];
}

sub labels_for_anr {
	my $anr = shift;

	my $res = $self->query('SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a
		JOIN persons p ON p.person = a.person
		JOIN mail_aliases c ON c.mail_alias = p.canonical
		JOIN domains d ON d.domain = a.domain
		JOIN domainnames n ON n.domainname = d.main
		WHERE p.person = ?
		ORDER BY
			p.canonical = a.mail_alias DESC,
			a.domain = c.domain DESC,
			n.name ASC,
			lower(a.name) ASC', $anr);

	return $self->record2alias(@$res);
}

sub person_for_anr {
	my $anr = shift;

	my $res = $self->query('SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
		FROM persons p WHERE p.person = ?', $anr);

	return $self->record2alias(@$res)->[0];
}

sub person_for_mailbox {
	my $mailbox = shift;

	my $res = $self->query('SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
		FROM persons p WHERE lower(p.mailaddress) = lower(?)', $mailbox);

	return $self->record2alias(@$res)->[0];
}

sub destinations_for {
	my $id = shift;

	my @res;

	$self->query(\@res, 'SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
		FROM personal_destinations d
			JOIN persons p ON d.person = p.person
		WHERE d.mail_alias = ?', $id);

	$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a
			JOIN internal_destinations i ON i.destination = a.mail_alias
		WHERE i.mail_alias = ?', $id);

	$self->query(\@res, 'SELECT e.external_destination, e.mailaddress
		FROM external_destinations e WHERE e.mail_alias = ?', $id);

	return $self->record2alias(@res);
}

sub referrers_for {
	my $id = shift;

	my $res = $self->query('SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a
		JOIN internal_destinations i USING (mail_alias)
		WHERE i.destination = ?', $id);

	return $self->record2alias(@$res);
}

sub has_referrers {
	my $alias = shift;

	my $res = $self->query('SELECT EXISTS(SELECT * FROM internal_destinations WHERE destination = ?)', $alias->id);

	return $res->[0][0];
}

sub export {
	my $db = $self->db;

	my %res;

	my $q1 = $db->prepare_cached('SELECT s.address, m.address
		FROM internal_destinations d
			JOIN mail_addresses s ON s.mail_alias = d.mail_alias
			JOIN mail_addresses m ON m.mail_alias = d.destination');
	$q1->execute;
	while(my ($from, $to) = $q1->fetchrow_array) {
		push @{$res{$from}}, $to;
	}
	$q1->finish;

	my $q2 = $db->prepare_cached('SELECT s.address, m.address
		FROM personal_destinations d
			JOIN func_mail_addresses s ON s.mail_alias = d.mail_alias
			JOIN persons p ON p.person = d.person
			JOIN mail_addresses m ON m.mail_alias = p.canonical');
	$q2->execute;
	while(my ($from, $to) = $q2->fetchrow_array) {
		push @{$res{$from}}, $to;
	}
	$q2->finish;

	my $q3 = $db->prepare_cached('SELECT s.address, d.mailaddress
		FROM external_destinations d
			JOIN mail_addresses s ON s.mail_alias = d.mail_alias');
	$q3->execute;
	while(my ($from, $to) = $q3->fetchrow_array) {
		push @{$res{$from}}, $to;
	}
	$q3->finish;

	return \%res;
}

sub blacklist {
	my @res;

	$self->query(\@res, "SELECT a.name || '\@' || n.name
			FROM mail_aliases a
			JOIN domains d ON d.domain = a.domain
			JOIN domainnames n ON n.domainname = d.main
			ORDER BY lower(a.name), n.name");

	return [map { $_->[0] } @res];
}

sub all_persons {
	my $res = $self->query("SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress FROM persons p");
	$res = $self->record2alias(@$res);
	my %res = map { $_->labels([]); ($_->anr, $_) } @$res;

	my $labels = $self->query("SELECT p.person, a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a
		JOIN persons p ON p.person = a.person
		JOIN mail_aliases c ON c.mail_alias = p.canonical
		JOIN domains d ON d.domain = a.domain
		JOIN domainnames n ON n.domainname = d.main
		ORDER BY
			p.canonical = a.mail_alias DESC,
			a.domain = c.domain DESC,
			n.name ASC,
			lower(a.name) ASC");

	foreach my $label (@$labels) {
		my $anr = shift @$label;
		my $person = $res{$anr};
		push @{$person->labels}, @{$self->record2alias($label)};
	}

	return $res;
}

sub fetch_alias_by_localpart {
	my $query = shift;

	confess("fetch_address(): \$query undefined") unless defined $query;

	confess("\@ sign in localpart") if $query =~ /\@/;

	my @res;

	$self->query(\@res, 'SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
		FROM mail_aliases a
			JOIN persons p ON p.person = a.person
			JOIN mail_aliases c ON c.mail_alias = p.canonical
		WHERE lower(a.name) = lower(?)', $query);

	$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a WHERE lower(a.name) = lower(?) AND a.person IS NULL', $query);

	return $self->record2alias(@res);
}

sub fetch_alias {
	my $query = shift;

	confess("fetch_address(): \$query undefined") unless defined $query;

	my ($local, $domain) = $query =~ /^(.*)@([^@]*)\z/;

	confess("fetch_address(): \$query does not contain an \@ sign") unless defined $domain;

	my $res = $self->query('SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
		FROM mail_aliases a
			JOIN domainnames d ON d.domain = a.domain
			JOIN persons p ON p.person = a.person
		WHERE lower(a.name) = lower(?)
			AND d.name = lower(?)', $local, $domain);

	unless(@$res) {
		$res = $self->query('SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
			FROM persons p
			WHERE lower(p.mailaddress) = lower(?)', $query);
	}

	unless(@$res) {
		$res = $self->query('SELECT a.mail_alias, a.name, a.domain, a.addressbook
			FROM mail_aliases a
				JOIN domainnames d ON d.domain = a.domain
			WHERE lower(a.name) = lower(?)
				AND d.name = lower(?)
				AND a.person IS NULL', $local, $domain);
	}

	return $self->record2alias(@$res)->[0];
}

sub search_alias {
	my $query = shift;

	return [] unless defined $query;

	#my ($local, $domain) = $query =~ /^(.*)@([^@]*)\z/;
	#return $self->search_exact($query) if defined $domain;

	my $escaped = $query;
	$escaped =~ s/([%_])/\\$1/g;

	my %parts; 
	@parts{split(' ', lc($query))} = ();
	return [] unless %parts;

	my %domain_parts;
	my @exact_parts;
	my @substring_where;
	my @substring_parts;

	foreach(keys %parts) {
		if(s/^\@//) {
			my $dom = $self->domain_by_name($_);
			undef $domain_parts{$dom->id} if defined $dom;
		} elsif(/\@/) {
			push @exact_parts, $_;
		} else {
			push @substring_parts, $_;
			s/([%_\\])/\\$1/g;
			push @substring_where, "%$_%";
		}	
	}

	my @domain_where = "a.domain IN (".join(', ', sort keys %domain_parts).")" if %domain_parts;

	# (substring AND substring AND (domain OR domain OR domain)) OR exact OR exact

	# 1) zoek exacte overeenkomsten

	my @res;

	foreach my $exact (@exact_parts) {
		$exact =~ /^(.+)\@([^@]+)\z/;
		my $local = $1;
		my $domain = $self->domain_by_name($2);

		if($domain) {
			$domain = $domain->id;
			$self->query(\@res, 'SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
				FROM mail_aliases a JOIN persons p ON p.person = a.person
				WHERE lower(a.name) = lower(?) AND a.domain = ?', $local, $domain);

			$self->query(\@res, 'SELECT a.mail_alias, a.name, n.name, a.addressbook
				FROM mail_aliases a
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
				WHERE lower(a.name) = lower(?) AND a.domain = ? AND a.person IS NULL', $local, $domain);
		} else {
			$self->query(\@res, 'SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
				FROM persons p WHERE lower(p.mailaddress) = lower(?)', $exact);

			$self->query(\@res, 'SELECT a.mail_alias, a.name, n.name, a.addressbook
				FROM mail_aliases a
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
					JOIN external_destinations e USING (mail_alias)
				WHERE lower(e.mailaddress) = lower(?)', $exact);
		}
	}

	# exacte overeenkomst met ANRs
	foreach my $substring (@substring_parts) {
		if($substring =~ /^\d{6}\z/) {
			$self->query(\@res, 'SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
				FROM persons p WHERE p.person = ?', int($substring));
		}
	}

	# 2) zorg dat de meest obvious match bovenaan komt

	if(@substring_parts == 1) {
		my $substring_part = $substring_parts[0];

		my $where = join(' AND ', @domain_where, 'lower(a.name) = lower(?)');

		$self->query(\@res, "SELECT a.mail_alias, a.name, n.name, a.addressbook
			FROM mail_aliases a
				JOIN domains d ON d.domain = a.domain
				JOIN domainnames n ON n.domainname = d.main
			WHERE $where AND a.person IS NULL ORDER BY a.mail_alias", $substring_part);

		$self->query(\@res, "SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
			FROM mail_aliases a JOIN persons p ON p.person = a.person
			WHERE $where ORDER BY a.mail_alias", $substring_part);
	}

	# 3) zoek fuzzy overeenkomsten

	my $where = join(' AND ', @domain_where, map { "lower(a.name) LIKE ?" } @substring_parts);
	if($where) {
		$self->query(\@res, "SELECT a.mail_alias, a.name, n.name, a.addressbook
			FROM mail_aliases a
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
			WHERE $where AND a.person IS NULL
			ORDER BY a.mail_alias LIMIT 50", @substring_where);

		$self->query(\@res, "SELECT NULL, NULL, NULL, NULL, p.person, p.mailaddress
			FROM mail_aliases a
				JOIN persons p ON p.person = a.person
			WHERE $where ORDER BY a.mail_alias LIMIT 50", @substring_where);
	}

	# ontdubbel

	do {
		my (%aliases, %persons);
		@res = grep { defined $_->[0] ? !$aliases{$_->[0]}++ : !$persons{$_->[4]}++ } @res;
	};

	# 4) zoek verwijzingen

	if(@res == 1) {
		my $res = $res[0];
		my ($a, undef, undef, undef, $p) = @$res;

		if($p) {
			$self->query(\@res, 'SELECT a.mail_alias, a.name, n.name, a.addressbook
				FROM mail_aliases a
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
					JOIN personal_destinations r ON r.mail_alias = a.mail_alias
				WHERE r.person = ? AND a.person IS NULL', $p);
		} else {
			$self->query(\@res, 'SELECT a.mail_alias, a.name, n.name, a.addressbook
				FROM mail_aliases a
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
					JOIN internal_destinations r ON r.mail_alias = a.mail_alias
				WHERE r.destination = ?', $a);
		}
	}

	# ontdubbel weer
	my (%aliases, %persons);
	@res = grep { defined $_->[0] ? !$aliases{$_->[0]}++ : !$persons{$_->[4]}++ } @res;

	my (%destinations, %labels);

	if(%aliases) {
		my $in = join(', ', map { '?' } keys %aliases);
		my @dst;
		$self->query(\@dst, "SELECT r.mail_alias, 'alias', a.name || '\@' || n.name
				FROM internal_destinations r
					JOIN mail_aliases a ON a.mail_alias = r.destination
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
				WHERE r.mail_alias IN ($in) ORDER BY lower(a.name), n.name", keys %aliases);
		$self->query(\@dst, "SELECT r.mail_alias, 'person', a.name || '\@' || n.name
				FROM personal_destinations r
					JOIN persons p ON p.person = r.person
					JOIN mail_aliases a ON a.mail_alias = p.canonical
					JOIN domains d ON d.domain = a.domain
					JOIN domainnames n ON n.domainname = d.main
				WHERE r.mail_alias IN ($in) ORDER BY lower(a.name), n.name", keys %aliases);
		$self->query(\@dst, "SELECT r.mail_alias, 'external', r.mailaddress
				FROM external_destinations r
				WHERE r.mail_alias IN ($in) ORDER BY lower(r.mailaddress)", keys %aliases);
		@destinations{keys %aliases} = map { {} } keys %aliases;
		foreach my $dst (@dst) {
			my ($a, $t, $e) = @$dst;
			push @{$destinations{$a}{$t}}, $e;
		}
	}
	if(%persons) {
		my $in = join(', ', map { '?' } keys %persons);
		my $labels = $self->query("SELECT p.person, a.name || '\@' || n.name
			FROM mail_aliases a
			JOIN persons p ON p.person = a.person
			JOIN mail_aliases c ON c.mail_alias = p.canonical
			JOIN domains d ON d.domain = a.domain
			JOIN domainnames n ON n.domainname = d.main
			WHERE p.person IN ($in)
			ORDER BY
				p.canonical = a.mail_alias DESC,
				a.domain = c.domain DESC,
				n.name ASC,
				lower(a.name) ASC", keys %persons);
		foreach my $row (@$labels) {
			my ($anr, $alias) = @$row;
			push @{$labels{$anr}}, $alias;
		}
	}

	return [map {
		defined $_->[0]
			? {address => $_->[1] . '@' . $_->[2], destinations => $destinations{$_->[0]}, addressbook => $_->[3]}
			: {labels => $labels{$_->[4]}, anr => $_->[4], mailbox => $_->[5]}
	} @res];
}
