use strict;
use warnings FATAL => 'all';
use utf8;

package Test::Xyzzy;

use Exporter qw(import);

our @EXPORT_OK = qw(xyzzy_from_hash xyzzy_from_config xyzzy_from_file xyzzy_request_response xyzzy_request xyzzy_request_as_cgi);

use Carp qw(confess);
use Scalar::Util qw(blessed reftype);
use Xyzzy;
use Xyzzy::Util qw(uri_escape_plus);
use Xyzzy::Request::Root;

sub xyzzy_from_hash {
	my $xyzzy = new Xyzzy;
	while(@_) {
		my $key = shift;
		my $val = shift;
		my $method = lc "set_$key";
		$xyzzy->$method(ref $val ? @$val : $val);
	}
	return $xyzzy;
}

sub xyzzy_from_config {
	my $xyzzy = new Xyzzy;
	my $cfg = shift;
	open(my $fh, '<', \$cfg) or die $!;
	$xyzzy->include_fh($fh, 'here document');
	return $xyzzy;
}

sub xyzzy_from_file {
	return xyzzy_from_config(include => shift);
}

my $default_env = {
	DOCUMENT_ROOT => '',
	FCGI_ROLE => 'RESPONDER',
	GATEWAY_INTERFACE => 'CGI/1.1',
	HTTP_ACCEPT => 'application/xhtml+xml,application/xml,text/html;q=0.9,*/*;q=0.8',
	HTTP_ACCEPT_CHARSET => 'UTF-8,*',
	HTTP_ACCEPT_ENCODING => 'gzip, deflate',
	HTTP_ACCEPT_LANGUAGE => 'nl,en-us;q=0.7,en;q=0.3',
	HTTP_CONNECTION => 'keep-alive',
	HTTP_HOST => 'localhost',
	HTTP_USER_AGENT => 'Mozilla/5.0 (Foo; Bar) Quux/1.0',
	PATH_INFO => '/',
	PATH_TRANSLATED => '/',
	QUERY_STRING => '',
	REDIRECT_STATUS => '200',
	REMOTE_ADDR => '127.0.0.1',
	REMOTE_PORT => '12345',
	REQUEST_METHOD => 'GET',
	REQUEST_URI => '/',
	SCRIPT_FILENAME => '',
	SCRIPT_NAME => '',
	SERVER_ADDR => '127.0.0.1',
	SERVER_NAME => 'localhost',
	SERVER_PORT => '80',
	SERVER_PROTOCOL => 'HTTP/1.1',
	SERVER_SOFTWARE => 'httpd/1.0',
};

sub xyzzy_query {
	my $ref = $_[0];
	my $reftype = reftype $ref;
	if(defined $reftype) {
		if($reftype eq 'HASH') {
			@_ = %$ref;
		} else {
			@_ = @$ref;
		}
	}
	my @q;
	while(@_) {
		my $key = uri_escape_plus(shift);
		my $val = shift;
		if(defined $val) {
			$val = uri_escape_plus($val);
			push @q, "$key=$val";
		} else {
			push @q, $key;
		}
	}
	return join('&', @q);
}

sub xyzzy_request_response {
	my ($handler, $path, $query, %options) = @_;

	$path //= '/';
	$query = xyzzy_query($query) if ref $query;

	$options{REQUEST_URI} //= $path.(defined $query ? "?$query" : '');
	$options{PATH_INFO} //= $path;
	$options{QUERY_STRING} //= $query // '';

	$handler = xyzzy_from_file($handler)
		unless ref $handler;

	$handler = xyzzy_from_config(@$handler)
		unless blessed $handler;

	$handler = $handler->handler
		if UNIVERSAL::isa($handler, 'Xyzzy');

	my ($inbuf, $outbuf, $errbuf) = ($options{REQUEST_BODY} // '', '', '');
	open(my $in, '<', \$inbuf) or die;
	open(my $out, '>', \$outbuf) or die;
	open(my $err, '>', \$errbuf) or die;
	my $req = new Xyzzy::Request::Root(in => $in, out => $out, err => $err, env => {%$default_env, %options});
	my $res = eval { $handler->handle($req) };
	unless($res) {
		my $err = $@;
		if(UNIVERSAL::isa($err, 'Xyzzy::Response')) {
			$res = $err;
		} else {
			die $err;
		}
	}
	die $errbuf if $errbuf ne '';
	return $res;
}

sub xyzzy_request {
	my $res = xyzzy_request_response(@_);
	my $content = $res->content;
	confess("Xyzzy::Response content must be binary! Convert logical characters explicitly using Encode::encode or Encode::encode_utf8")
		if utf8::is_utf8($content);
	return $content;
}

sub xyzzy_request_as_cgi {
	my $res = xyzzy_request_response(@_);
	my $as_cgi = $res->as_cgi;
	$as_cgi = join('', @$as_cgi);
	confess("Xyzzy::Response content and headers must be binary! Convert logical characters explicitly using Encode::encode or Encode::encode_utf8")
		if utf8::is_utf8($as_cgi);
	return $as_cgi;
}

1;
