use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Crypto;

use MIME::Base64;
use Encode;
use Digest::SHA qw(sha256);
use Crypt::OpenSSL::Random;
use Crypt::OpenSSL::Bignum;
use Time::HiRes qw(time);
use Xyzzy::Util qw(utf8_testandset_inplace);

use Clarity -self;

field cfg;
field secret => sub { shift->cfg->cryptosecret };
field jitter => sub { shift->cfg->cryptoclockjitter };

sub random_bytes {
	die "Insufficient entropy\n"
		unless Crypt::OpenSSL::Random::random_status;
	return Crypt::OpenSSL::Random::random_bytes(@_);
}

sub random_hex {
	my $hex = unpack('H*', $self->random_bytes($@));
	$hex =~ tr/a-z/A-Z/;
	return $hex;
}

sub hmac {
	my $token = join('', map { my $str = $_; Encode::_utf8_off($str); $str } @_);
	foreach my $key (@{$self->secret}) {
		$token = sha256($key . $token);
	}
	return $token;
}

sub create_token_data {
	my ($hidden, @keys) = @_;

	foreach(@keys) {
		Encode::_utf8_off($_);
		die "Bad characters in request\n"
			unless index($_, "\0") < 0;
	}

	my $time = int(time * 1e6);
	my $salt = $self->random_bytes(16);
	my $token = pack('Q>a*a*', $time, $salt, join("\0", @keys));

	my $hash = $self->hmac($hidden, "\0", $token);

	return $time, $salt, $hash.$token;
}

sub create_token {
	my ($time, $salt, $raw) = $self->create_token_data(@_);

	my $encoded = encode_base64($raw, '');
	$encoded =~ s/=+$//;
	$encoded =~ tr|/+|._|;

	return $time, $salt, $encoded;
}

sub check_token_data {
	my ($hidden, $token, $expiry) = @_;

	my $hash = substr($token, 0, 32, '');
	die "Short token\n"
		unless length($hash) == 32;

	die "Bad signature on token\n"
		unless $self->hmac($hidden, "\0", $token) eq $hash;

	my ($time, $salt, @keys) = unpack('Q>a16(Z*)*', $token);

	$time /= 1e6;

	my $now = time;

	die "Future token\n"
		if $time > $now + $self->jitter;

	die "Expired token\n"
		if $expiry && $time + $expiry < $now;

	utf8_testandset_inplace(@keys);

	if(wantarray) {
		return $time, $salt, @keys;
	} else {
		return (shift @keys) // 1;
	}
}

sub check_token {
	my ($hidden, $token, $expiry) = @_;

	die "Undefined token\n"
		unless defined $token;

	die "Invalid token\n"
		unless $token =~ /^[a-zA-Z0-9_.]+$/;

	$token =~ tr|._|/+|;
	$token .= '=' x -(length($token) % -4);
	$token = decode_base64($token);

	return $self->check_token_data($hidden, $token, $expiry);
}
