use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Document::Config;

use Clarity -self;

field templatedir => sub { die "TemplateDir not configured\n" };
field fragmentdir => sub { die "FragmentDir not configured\n" };
field contentdir => sub { die "ContentDir not configured\n" };
field stylesheetdir => sub { die "StylesheetDir not configured\n" };
field cache_templates => 1;

sub set_templatedir {
	my $dir = shift;
	die "Missing argument to TemplateDir\n" unless defined $dir;
	stat $dir;
	warn "TemplateDir $dir doesn't exist\n" unless -e _;
	warn "TemplateDir $dir is not a directory\n" unless -d _;
	$self->templatedir($dir);
}

sub set_fragmentdir {
	my $dir = shift;
	die "Missing argument to FragmentDir\n" unless defined $dir;
	stat $dir;
	warn "FragmentDir $dir doesn't exist\n" unless -e _;
	warn "FragmentDir $dir is not a directory\n" unless -d _;
	$self->fragmentdir($dir);
}

sub set_contentdir {
	my $dir = shift;
	die "Missing argument to ContentDir\n" unless defined $dir;
	stat $dir;
	warn "ContentDir $dir doesn't exist\n" unless -e _;
	warn "ContentDir $dir is not a directory\n" unless -d _;
	$self->contentdir($dir);
}

sub set_stylesheetdir {
	my $dir = shift;
	die "Missing argument to StylesheetDir\n" unless defined $dir;
	stat $dir;
	warn "StylesheetDir $dir doesn't exist\n" unless -e _;
	warn "StylesheetDir $dir is not a directory\n" unless -d _;
	$self->stylesheetdir($dir);
}

sub set_cachetemplates {
	$self->cache_templates(Xyzzy::parse_bool(shift));
}

package Xyzzy::Document;

use XML::LibXML;
use XML::LibXML::XPathContext;
use XML::LibXSLT;
use Scalar::Util qw(reftype);

use Clarity -self;
use Xyzzy::Hook -mixin;

field xml => sub { new XML::LibXML::Document("1.0", "UTF-8") };
field cfg => sub { shift->req->cfg };
field req;

sub _gethash() {
	return $_[0]->{$_[1]}
}

sub getElementById {
	my $id = shift;

	our $expr ||= new XML::LibXML::XPathExpression('//*[@id=$id]');

	our $ctx ||= new XML::LibXML::XPathContext;

	our $prev;
	unless(defined $prev && $prev eq $id) {
		$ctx->registerVarLookupFunc(\&_gethash, {id => $id});
		$prev = $id;
	}

	my $nodes = $ctx->find($expr, $self->xml);

	confess "Document ID '$id' not unique"
		if @$nodes > 1;

	return $nodes->[0]
}

sub xmlParser {
	our $xmlparser ||= new XML::LibXML;
	$xmlparser->set_options({
		expand_entities => 0,
		expand_xinclude => 0,
		load_ext_dtd => 0,
		no_cdata => 1,
		no_network => 1,
		pedantic_parser => 1,
		validation => 0,
	});

	return $xmlparser;
}

sub xsltParser {
	return our $xsltparser ||= new XML::LibXSLT;
}

sub getStylesheet {
	my $name = shift;
	my $cfg = $self->cfg;
	my $dir = $cfg->stylesheetdir;
	my $file = "$dir/$name.xsl";

	our %stylesheetcache;
	return $stylesheetcache{$file}
		if exists $stylesheetcache{$file};

	my $xmlparser = $self->xmlParser;
	my $xml = $xmlparser->parse_file($file)
		or die "can't parse $file\n";

	my $xsltparser = $self->xsltParser;
	my $xslt = $xsltparser->parse_stylesheet($xml)
		or die "can't parse $file as XSLT\n";

	$stylesheetcache{$file} = $xslt
		if $cfg->cache_templates;;
	return $xslt;
}

field transform_params => sub {
	my $req = shift->req;
	return [
		XML::LibXSLT::xpath_to_string(map {
			my $val = eval { $req->$_() };
			defined $val ? ($_, $val) : ()
		} qw(
				user_agent path_info server_name server_software server_port server_protocol
				remote_ident remote_user auth_type request_uri request_method query_string
				script_name script_filename path_translated remote_host remote_addr referer
				user_name virtual_host virtual_port param_string protocol base_url url_path
				path_url url self_path self_url script_path script_url)
		),
		map {$_, (eval { $req->$_() } ? 'true()' : 'false()')} qw(msie secure),
	];
};

sub transform {
	my $xslt = shift;
	$xslt = $self->getStylesheet($xslt)
		unless ref $xslt;
	my $xml = $xslt->transform($self->xml, @{$self->transform_params}, @_)
		or die "XSLT transform failed\n";
	$self->xml($xml);
	return;
}

sub construct {
	my $name = shift;
	utf8::upgrade($name);
	my $node = $self->createElement($name);
	foreach my $child (@_) {
		if(ref $child) {
			my $type = reftype($child);
			if($type eq 'ARRAY') {
				$node->appendChild($self->construct(@$child));
			} elsif($type eq 'HASH') {
				while(my ($key, $val) = each(%$child)) {
					utf8::upgrade($key);
					utf8::upgrade($val);
					$node->setAttribute($key, $val);
				}
			} else {
				$node->appendChild($child);
			}
		} else {
			utf8::upgrade($child);
			$node->appendText($child);
		}
	}
	return $node;
}

sub serialize {
	return new Xyzzy::Response(mimetype => 'application/xml', content => $self->toString);
}

sub response {
	$self->hook_call('build');
	my $res = $self->serialize;
	delete $self->{xml};
	return $res;
}

sub bakecookie {
	my $name = shift;
	my $value = shift;
	my $req = $self->req;
	my %defaults = (Secure => $req->secure);
	my $path = $req->script_path;
	$defaults{Path} = $path || '/' if defined $path;

	return new Xyzzy::Cookie($name => $value, %defaults, @_);
}

sub DESTROY() {} # don't try to autoload this

sub AUTOLOAD {
	my $off = rindex(our $AUTOLOAD, '::');
	confess "no package name in '$AUTOLOAD'"
		if $off == -1;
	my $sub = substr($AUTOLOAD, $off + 2);
	my $xml = $self->xml;
	my $code = "sub $sub { my \$self = shift; return \$self->xml->$sub(\@_) }";
	my $err = do { local $@; eval $code; $@ };
	confess $err if $err;
	return $xml->$sub(@_);
}
