use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy;

use Xyzzy::Status;
use Xyzzy::Static;
use Xyzzy::Directory;
use Xyzzy::Document;

use Clarity -self;
use Xyzzy::Document::Config -mixin;
use Xyzzy::Document::XHTML::Config -mixin;
use Xyzzy::Static::Config -mixin;

use IO::File;

our @EXPORT_BASE = qw(parse_time parse_size parse_bool);

sub set_application {
	my $class = $_[0];
	die "Missing argument to Application\n" unless defined $class;
	eval "use $class";
	die $@ if $@;
	bless $self, $class;
	$self->new;
}

field benchmark => undef;

sub set_benchmark {
	$self->benchmark(!!$_[0]);
}

{
	my %units = (
		'' => 1,
		s => 1,
		m => 60,
		h => 3600,
		d => 86400,
		w => 604800,
		m => 31 * 86400,
		q => 92 * 86400,
		y => int(365.2425 * 86400),
	);

	sub parse_time {
		my ($time) = @_;
		$time =~ /^(\d+)\s*([a-z]?)$/i;
		my ($seconds, $unit) = ($1, $2);
		die "can't parse time value '$time'\n"
			unless defined $seconds;
		my $multiplier = $units{lc $unit}
			or die "unknown unit $unit\n";
		return $seconds * $multiplier;
	}
}

{
	my %units = (
		'' => 1,
		k => 1024,
		m => 1048576,
		g => 1073741824,
		t => 1099511627776,
		p => 1125899906842624,
		e => 1152921504606846976,
		z => 1180591620717411303424,
		y => 1208925819614629174706176,
	);

	sub parse_size {
		my ($size) = @_;
		$size =~ /^(\d+)\s*([a-z]?)$/i
			or die "can't parse size value '$size'\n";
		my ($bytes, $unit) = ($1, $2);
		my $multiplier = $units{lc $unit}
			or die "unknown unit $unit\n";
		return $bytes * $multiplier;
	}
}

{
	sub parse_bool {
		local $_ = shift;
		return 1 if /^(?:1|yes|true|on|enabled?)$/i;
		return 0 if /^(?:0|no|false|off|disabled?)$/i;
		die "unknown boolean value '$_'\n";
	}
}

sub handle_configuration_item {
	my ($key, $values, $filename, $lineno) = @_;
	return unless defined $key;
	eval {
		my $lkey = lc("set_$key");
		die "unknown key\n"
			unless $self->can($lkey);

		$self->$lkey(@$values);
	};
	die "$filename:$lineno: $key: $@" if $@;
}

sub include {
	foreach my $filename (@_) {
		my $fh = new IO::File($filename, '<:utf8')
			or die "Can't open configuration file '$filename': $!\n";;

		my ($ws, $lineno);
		my ($key, @values);

		while(my $line = $fh->getline) {
			next if ord($line) == 35; #
			$line =~ s/\s+$//;
			if($line eq '') {
				$ws = 1;
				next;
			}
			if($line =~ s/^\s//) {
				unless(defined $key) {
					my $lineno = $fh->input_line_number;
					warn "Line continuation before key at $filename:$lineno\n";
					next;
				}
				if($ws) {
					push @values, '';
					undef $ws;
				}
				push @values, $line;
			} else {
				$self->handle_configuration_item($key, \@values, $filename, $lineno);
				@values = ();
				$lineno = $fh->input_line_number;
				($key, my $val) = split(/\s+/, $line, 2);
				push @values, $val;
				undef $ws;
			}	
		}

		die "Can't read from configuration file '$filename': $!\n"
			unless $fh->eof;

		$self->handle_configuration_item($key, \@values, $filename, $lineno);

		$fh->close;
	}
}

*set_include = *include;

sub error {
	my $status = new Xyzzy::Status(@_);
	return $status->response;
}

sub handler {
	my $static = new Xyzzy::Static(cfg => $self);

	return new Xyzzy::Directory(cfg => $self,
		handler => $static,
		fallback => $static,
	);
}

sub new {
	return $self if ref $self;
	return bless {}, $self;
}
