#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;

sub survive(&) {
	my $func = shift;
	if(eval { $func->(); 1 }) {
		return 1;
	} else {
		return fail($@);
	}
}

sub suicide(&$) {
	my ($func, $message) = @_;
	my $res = eval { $func->() // 'undef' };
	if(defined $res) {
		return fail("$message: should have died but returned '$res'");
	} else {
		return ok($@);
	}
}

sub carefully(&) {
	my $func = shift;
	my $res = eval { $func->() };
	return $res unless $@;
	warn $@;
	return;
}

BEGIN { use_ok('Xyzzy::Request') or BAIL_OUT('need Xyzzy::Request to run') }

new_ok('Xyzzy::Request') or BAIL_OUT('need a Xyzzy::Request object to run');

foreach my $r (
	['/foo', '/bar', undef],
	['/', '/', ''],
	['//', '/', ''],
	['/foo', '/', undef],
	['/foo/', '/', '/foo'],
	['//foo', '/', undef],
	['//foo/', '/', '//foo'],
	['//foo//', '/', '//foo'],
	['/foo', '/foo', ''],
	['//foo', '/foo', ''],
	['/foo/', '/foo/', ''],
	['//foo/', '/foo/', ''],
	['//foo//', '/foo/', ''],
	['/foo/bar', '/bar', '/foo'],
	['//foo/bar', '/bar', '//foo'],
	['//foo//bar', '/bar', '//foo'],
	['/foo/bar/', '/bar/', '/foo'],
	['//foo/bar/', '/bar/', '//foo'],
	['//foo//bar/', '/bar/', '//foo'],
	['/foo/bar//', '/bar/', '/foo'],
	['//foo/bar//', '/bar/', '//foo'],
	['//foo//bar//', '/bar/', '//foo'],
) {
	my ($uri, $info, $path) = @$r;
	my $req = new_ok('Xyzzy::Request::Root', [env => {REQUEST_URI => $uri, PATH_INFO => $info}])
		or next;
	is(carefully { $req->script_path }, $path, "script_path() failed for uri=$uri path_info=$info");
}

foreach my $r (
	['x=y', 'y'],
	['x=', ''],
	['y=', undef],
	['x', ''],
) {
	my ($q, $expected) = @$r;
	foreach my $sep('&', ';') {
		foreach my $pfx (
			[$q],
			['a=b', $q],
			[$q, 'p=q'],
			['a=b', $q, 'p=q'],
			['a', $q, 'p=q'],
		) {
			my $q = join($sep, @$pfx);
			{
				my $req = new_ok('Xyzzy::Request::Root', [env => {QUERY_STRING => $q}]);
				survive { is($req->param('x'), $expected, "\$req->param('x') (GET /?$q)") };
			}

			{
				open(my $in, '<', \$q);
				my $req = new_ok('Xyzzy::Request::Root', [env => {HTTP_CONTENT_TYPE => 'application/x-www-form-urlencoded'}, in => $in]);
				survive { is($req->param('x'), $expected, "\$req->param('x') (POST /?$q)") };
			}
		}
	}
}

foreach my $r (
	['x=a', '', 'a', 'a', undef],
	['', 'x=a', 'a', undef, 'a'],
	['x=a', 'x=b', 'b', 'a', 'b'],
) {
	my ($get, $post, $param, $url_param, $body_param) = @$r;
	{
		open(my $in, '<', \$post);
		my $req = new_ok('Xyzzy::Request::Root', [env => {QUERY_STRING => $get, HTTP_CONTENT_TYPE => 'application/x-www-form-urlencoded'}, in => $in]);
		survive { is($req->param('x'), $param, "\$req->param('x') (/?$get with body='$post')") };
		survive { is($req->url_param('x'), $url_param, "\$req->url_param('x') (/?$get with body='$post')") };
		survive { is($req->body_param('x'), $body_param, "\$req->body_param('x') (/?$get with body='$post')") };
	}
}

package Dummy::Request;

use Xyzzy::Request -self;

my $trema = "\x{00EB}";
my $u8trema = pack('C*', 0xC3, 0xAB);
my $l1trema = pack('C*', 0xEB);

param p1;
param p2 => sub { Encode::_utf8_off($_) };
param p3 => sub { die unless defined };
param p4 => sub { die unless $_ };
param p5 => sub { die if defined && /\W/ };
param p6 => sub { $_ ||= "none" };
param p7 => sub { $_ //= "none" };
param p8 => sub { $_ = lc };
param p9 => sub { s/bob/robert/g if defined };
param p10 => 'none';

package main;

new_ok('Dummy::Request')
	or BAIL_OUT('need Dummy::Request for further tests');

foreach my $r (
	['p1=x', 'p1', 'x'],
	['p1=', 'p1', ''],
	['foo', 'p1', undef],
	['p1=%C3%AB', 'p1', $trema],
	['p1=%EB', 'p1', $l1trema],
	['p2=%C3%AB', 'p2', $u8trema],
	['p2=%EB', 'p2', $l1trema],
	['p3=x', 'p3', 'x'],
	['p3=', 'p3', ''],
	['p3', 'p3', ''],
	['foo', 'p3', '_death'],
	['p4=x', 'p4', 'x'],
	['p4=0', 'p4', '_death'],
	['p4=', 'p4', '_death'],
	['p4', 'p4', '_death'],
	['foo', 'p4', '_death'],
	['p5=x', 'p5', 'x'],
	['p5=0', 'p5', '0'],
	['p5=*', 'p5', '_death'],
	['p5=', 'p5', ''],
	['p5', 'p5', ''],
	['foo', 'p5', undef],
	['p6=x', 'p6', 'x'],
	['p6=0', 'p6', 'none'],
	['p6=', 'p6', 'none'],
	['p6', 'p6', 'none'],
	['foo', 'p6', 'none'],
	['p7=x', 'p7', 'x'],
	['p7=0', 'p7', '0'],
	['p7=', 'p7', ''],
	['p7', 'p7', ''],
	['foo', 'p7', 'none'],
	['p8=x', 'p8', 'x'],
	['p8=X', 'p8', 'x'],
	['p8=0', 'p8', '0'],
	['p8=', 'p8', ''],
	['p8', 'p8', ''],
	['foo', 'p8', ''],
	['p9=x', 'p9', 'x'],
	['p9=bob', 'p9', 'robert'],
	['p9=0', 'p9', '0'],
	['p9=', 'p9', ''],
	['p9', 'p9', ''],
	['foo', 'p9', undef],
	['p10=x', 'p10', 'x'],
	['p10=0', 'p10', '0'],
	['p10=', 'p10', ''],
	['p10', 'p10', ''],
	['foo', 'p10', 'none'],
) {
	my ($q, $method, $expected) = @$r;
	my $ctx = new_ok('Xyzzy::Request::Root', [env => {QUERY_STRING => $q}]);
	my $req = new_ok('Dummy::Request', [ctx => $ctx]) or next;
	if(defined $expected && $expected eq '_death') {
		suicide(sub { $req->$method() }, "\$req->$method (GET /?$q)");
	} else {
		survive { is($req->$method(), $expected, "\$req->$method (GET /?$q)") };
	}
}

{
	my $ctx = new_ok('Xyzzy::Request::Root', [env => {}]);
	my $req = new_ok('Dummy::Request', [ctx => $ctx]) or next;
	survive { is($req->_get_safe_property('server_port'), 80, "\$req->_get_safe_property('server_port')") };
	suicide(sub { $req->_get_safe_property('cfg') }, "\$req->_get_safe_property('cfg')");
}

{
	my $body = <<'EOT';
-----------------------------1985796259607567365678475
Content-Disposition: form-data; name="file"; filename="test.txt"
Content-Type: text/plain

plaintest

-----------------------------1985796259607567365678475
Content-Disposition: form-data; name="switch"

on
-----------------------------1985796259607567365678475
Content-Disposition: form-data; name="submit"


-----------------------------1985796259607567365678475--
EOT

	{
		open(my $in, '<', \$body) or BAIL_OUT("Can't create in-memory filehandle");
		my $ctx = new_ok('Xyzzy::Request::Root', [env => {CONTENT_TYPE => 'multipart/form-data; boundary=---------------------------1985796259607567365678475', HTTP_CONTENT_LENGTH => length($body)}, in => $in]);
		my $req = new_ok('Dummy::Request', [ctx => $ctx]);
		survive {
			is($req->param('switch'), 'on', "\$req->param('switch')");
			my $file = $req->upload('file');
			my $line = $file->getline;
			is($line, "plaintest\n", "uploaded file");
		};
	}

	{
		local $Xyzzy::Request::Root::body_max = 10;
		open(my $in, '<', \$body) or BAIL_OUT("Can't create in-memory filehandle");
		my $ctx = new_ok('Xyzzy::Request::Root', [env => {CONTENT_TYPE => 'multipart/form-data; boundary=---------------------------1985796259607567365678475', HTTP_CONTENT_LENGTH => length($body)}, in => $in]);
		my $req = new_ok('Dummy::Request', [ctx => $ctx]);
		survive {
			is($req->param('switch'), 'on', "\$req->param('switch')");
			my $file = $req->upload('file');
			my $line = $file->getline;
			is($line, "plaintest\n", "uploaded file");
		};
	}
}

foreach my $r (
	['a=x; b=y; c=z', 'y'],
	['a="x"; b="y"; c="z"', 'y'],
	['a="x;x"; b="y"; c="z"', 'y'],
	['a="x;x; b="y"; c="z"', 'y'],
	['a="x;x; b="y; c="z"', '"y'],
	['a="x;x; b="y"', 'y'],
	['a="x;x; b="y', '"y'],
	['a="x;x; b="y%C3%AB; c="z"', "\"y$trema"],
	['a="x;x; b="y%EB; c="z"', "\"y$l1trema"],
) {
	my ($c, $expected) = @$r;
	my $ctx = new_ok('Xyzzy::Request::Root', [env => {HTTP_COOKIE => $c}]);
	my $req = new_ok('Dummy::Request', [ctx => $ctx]) or next;
	survive { is($req->cookie('b'), $expected, "\$req->cookie('b') (Cookie: $c)") };
}

{
	my $ctx = new_ok('Xyzzy::Request::Root', [env => {HTTP_USER_AGENT => 'Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US) AppleWebKit/525.13 (KHTML, like Gecko) Chrome/0.2.149.27 Safari/525.13'}]);
	my $req = new_ok('Dummy::Request', [ctx => $ctx]) or next;
	survive { ok(!$req->msie, "\$req->msie (User-Agent: Chrome)") };
}

{
	my $ctx = new_ok('Xyzzy::Request::Root', [env => {HTTP_USER_AGENT => 'Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; en-GB)'}]);
	my $req = new_ok('Dummy::Request', [ctx => $ctx]) or next;
	survive { ok($req->msie, "\$req->msie (User-Agent: MSIE)") };
}

# test mime_to_file/mime_to_encoding

done_testing();
