<?php
/*
Plugin Name: CAS Authentication
Version: 1.4
Plugin URI: http://non-gnu.uvt.nl/
Description: Authenticate users using CAS. User info retrieved from LDAP.
Author: Wessel Dankers
Author URI: mailto:wsl@uvt.nl
*/

/*
 * $Id: cas-authentication.php 43417 2015-06-05 13:22:57Z wsl $
 * $URL: https://svn.uvt.nl/its-id/trunk/sources/wordpress-cas/cas-authentication.php $
 */

require_once ABSPATH.WPINC.'/registration-functions.php';

/*
 * Retrieve info from LDAP for newly created users
 */
function cas_ldapuserinfo($uid) {
	if(!eregi('^[a-z_][a-z_0-9-]*$', $uid))
		return null;

	$server = get_option('cas_ldap_server');
	$base = get_option('cas_ldap_base');
	$filter = get_option('cas_ldap_filter');

	$attrs = array('cn', 'givenName', 'sn', 'mail');

	$ldap = ldap_connect($server);

	if(!$ldap)
		die("Can't contact LDAP-server");

	if(empty($filter)) {
		$filter = "(uid=$uid)";
	} else {
		if(!preg_match('/^\(.*\)$/', $filter))
			die("Malformed LDAP filter, contact the site administrator");
		$filter = '(&'.$filter."(uid=$uid))";
	}

	$search = ldap_search($ldap, $base, $filter, $attrs, 0, 2);

	if(!$search)
		die("Can't search LDAP-server");

	$entry = ldap_first_entry($ldap, $search);
	if(!$entry)
		return null;

	if(ldap_next_entry($ldap, $entry))
		die("Too many results");

	$all = ldap_get_attributes($ldap, $entry);

	$ret = array();

	foreach($attrs as $attr) {
		$values = ldap_get_values($ldap, $entry, $attr);
		$ret[$attr] = $values[0];
	}

	return $ret;
}

/*
 * Cleaned up version of the current URL
 */
function cas_service() {
	$scheme = is_ssl() || force_ssl_login() || force_ssl_admin() ? 'https' : 'http';
	return $scheme.'://'.$_SERVER['HTTP_HOST'].remove_query_arg('ticket');
}

/*
 * Ignore ticket parameters that do not look like CAS tickets
 */
function cas_ticket() {
	$ticket = $_REQUEST['ticket'];
	if(!is_null($ticket) && preg_match('/^ST-/', $ticket))
		return $ticket;
}

/*
 * Check tickets with the CAS server
 */
function cas_validate($ticket, $service) {
	$q = array('service' => $service, 'ticket' => $ticket);
	$url = add_query_arg($q, get_option('cas_validate_uri'));

	$curl = curl_init($url);
	if($curl === false)
		throw new ErrorException("Unable to create CURL handle");

	$curl_options = array(
		CURLOPT_BINARYTRANSFER => true,
		CURLOPT_FAILONERROR => true,
		CURLOPT_RETURNTRANSFER => true,
		CURLOPT_CONNECTTIMEOUT => 1,
		CURLOPT_TIMEOUT => 5,
		CURLOPT_USERAGENT => "simpleSAMLphp",
	);

	if(!curl_setopt_array($curl, $curl_options))
		throw new ErrorException("Unable to set CURL options: ".curl_error($curl));

	$str = curl_exec($curl);
	$err = curl_error($curl);

	curl_close($curl);

	if($str === false)
		throw new ErrorException("Unable to retrieve URL: $error");

	if(preg_match('{^yes\n([a-z0-9_-]+)\n$}i', $str, $parts))
		return $parts[1];

	return null;
}

/*
 * Fetch user data and handle non-existent users
 */
function cas_userdatabylogin($uid) {
	if(is_null($uid))
		return null;

	if(!function_exists('get_userdatabylogin'))
		die("Could not load user data");

	$user = get_userdatabylogin($uid);
	if(!$user) {
		if(get_option('cas_auto_create') != 'on')
			die("User ID '$uid' doesn't exist yet; contact the admin to have it created.");
		$ldap = cas_ldapuserinfo($uid);
		if(is_null($ldap))
			die("User ID '$uid' can't be created automatically; contact the admin.");
		$new_user = wp_create_user($uid, $ldap['cn'], $ldap['mail']);
		update_usermeta($new_user, 'first_name', $ldap['givenName']);
		update_usermeta($new_user, 'last_name', $ldap['sn']);
		$user = get_userdatabylogin($uid);
		if(!$user)
			die("Can't create account for '$uid': contact the admin.");
	}

	if($user->user_login != $uid)
		die("Can't locate account for '$uid': contact the admin.");

	return $user;
}

/*
 * This is how WP seems to want to do redirects
 */
function cas_redirect($url) {
	wp_redirect($url);
	exit("<!DOCTYPE html>
<html><head>
<title>Redirection</title>
</head><body>
<h1>Redirection</h1>
<p>Please stand by.</p>
</body></html>
");
}

/*
 * Initiate a CAS login sequence
 */
function cas_login() {
	cas_redirect(add_query_arg('service', cas_service(), get_option('cas_login_uri')));
}

/*
 * This seems to be the only place we can intercept anonymous logins.
 * Also catch requests with a ticket
 */
if(!function_exists('wp_get_current_user')):
function wp_get_current_user() {
	global $current_user;

	if(cas_ticket())
		$current_user = wp_signon();

	get_currentuserinfo();

	if($current_user->ID || get_option('cas_allow_anon') == 'on')
		return $current_user;

	$current_user = wp_signon();

	/* Make sure we don't let any anonymous user through */
	if($current_user->ID)
		return $current_user;

	/* This shouldn't happen */
	cas_login();
}
endif;

if(!class_exists('CASAuthenticationPlugin')) {
	class CASAuthenticationPlugin {
		/*
		 * Add hooks for this plugin to the database.
		 */
		function CASAuthenticationPlugin() {
			if(isset($_GET['activate']) and $_GET['activate'] == 'true')
				add_action('init', array(&$this, 'init'));
			add_action('admin_menu', array(&$this, 'admin_menu'));
			add_action('wp_login', array(&$this, 'login'), 100, 2);
			add_action('wp_logout', array(&$this, 'logout'));
			add_action('lost_password', array(&$this, 'disable_function'));
			add_action('retrieve_password', array(&$this, 'disable_function'));
			add_action('password_reset', array(&$this, 'disable_function'));
			add_action('check_passwords', array(&$this, 'check_passwords'), 10, 3);
			add_action('user_new_form', array(&$this, 'user_new_form'));
			add_filter('show_password_fields', array(&$this, 'show_password_fields'));
			add_filter('loginout', array(&$this, 'loginout'));

			remove_all_actions('authenticate');
			add_action('authenticate', array(&$this, 'authenticate'), 100, 3);
		}


		/*************************************************************
		 * Plugin hooks
		 *************************************************************/

		/*
		 * Add configuration options for this plugin to the database.
		 */
		function init() {
			if(current_user_can('manage_options')) {
				add_option('cas_login_uri', 'https://sso.example.com/login', 'The URL to the CAS login page');
				add_option('cas_validate_uri', 'https://sso.example.com/validate', 'The URL to the CAS validation service');
				add_option('cas_logout_uri', get_option('home'), 'The URI to which the user is redirected when she chooses "Logout".');
				add_option('cas_allow_anon', 'no', 'Allow anonymous users.');
				add_option('cas_auto_create', 'no', 'Automatically create accounts.');
				add_option('cas_ldap_server', 'ldaps://ldap.example.com', 'The URL to the LDAP server');
				add_option('cas_ldap_base', 'dc=example,dc=com', 'The base DN of the LDAP server');
				add_option('cas_ldap_filter', '(organizationalStatus=staff)', 'Additional filter for new users');
			}
		}

		/*
		 * Add an options pane for this plugin.
		 */
		function admin_menu() {
			if(function_exists('add_options_page'))
				add_options_page('CAS Authentication', 'CAS', 'manage_options', 'wp-content/plugins/cas-authentication.php', array(&$this, 'display_options_page'));
		}

		/*
		 * Main authentication handler.
		 */
		function authenticate($user, $uid, $passwd) {
			if(is_a($user, 'WP_User'))
				return $user;

			$service = cas_service();
			$ticket = cas_ticket();
			if(!$ticket)
				cas_login();

			$uid = cas_validate($ticket, $service);
			if(is_null($uid))
				cas_login();

			$user = cas_userdatabylogin($uid);
			return new WP_User($user->ID);
		}

		/*
		 * This handler is called after the authentication completes
		 * and the session is started.
		 */
		function login($uid, $user) {
			/* Clean the URL up (tickets are sort of sensitive data). */
			if(cas_ticket() && !isset($_REQUEST['redirect_to']))
				cas_redirect(cas_service());
		}

		/*
		 * Logout the user by redirecting them to the logout URI.
		 */
		function logout() {
			$location = get_option('cas_logout_uri');
			if(empty($location))
				$location = get_option('home');
			cas_redirect($location);
		}

		/*
		 * Generate a password for the user. This plugin does not
		 * require the user to enter this value, but we want to set it
		 * to something nonobvious.
		 */
		function check_passwords($username, $password1, $password2) {
			$password1 = $password2 = md5(wp_salt('secure_auth'));
		}

		/*
		 * Used to disable certain display elements, e.g. password
		 * fields on profile screen.
		 */
		function show_password_fields($show_password_fields) {
			return false;
		}

		/*
		** Add dummy password fields to appease WP
		*/
		function user_new_form() {
			$pass = base64_encode(openssl_random_pseudo_bytes(15));
			echo "<input type='hidden' name='pass1' value='$pass' />\n";
			echo "<input type='hidden' name='pass2' value='$pass' />\n";
		}

		/*
		 * Turn the login link into something usable
		 */
		function loginout($html) {
			if(strpos($html, '?') === FALSE)
				return str_replace('/wp-login.php', '/wp-admin/', $html);
			return $html;
		}

		/*
		 * Used to disable certain login functions, e.g. retrieving a
		 * user's password.
		 */
		function disable_function() {
			die('Disabled');
		}


		/*************************************************************
		 * Functions
		 *************************************************************/

		/*
		 * Display the options for this plugin.
		 */
		function display_options_page() {
			$login_uri = get_option('cas_login_uri');
			$validate_uri = get_option('cas_validate_uri');
			$logout_uri = get_option('cas_logout_uri');
			$auto_create = get_option('cas_auto_create');
			$allow_anon = get_option('cas_allow_anon');
			$ldap_server = get_option('cas_ldap_server');
			$ldap_base = get_option('cas_ldap_base');
			$ldap_filter = get_option('cas_ldap_filter');
?>
<div class="wrap">
  <h2>CAS Authentication Options</h2>
  <form action="options.php" method="post">
	<input type="hidden" name="action" value="update" />
	<input type="hidden" name="page_options" value="cas_login_uri,cas_validate_uri,cas_logout_uri,cas_auto_create,cas_allow_anon,cas_ldap_server,cas_ldap_base,cas_ldap_filter" />
	<?php if(function_exists('wp_nonce_field')): wp_nonce_field('update-options'); endif; ?>

	<fieldset class="options">
	  <table class="editform optiontable">

		<tr valign="top">
		  <th scope="row"><label for="cas_login_uri">CAS Login URI</label></th>
		  <td>
			<input type="text" name="cas_login_uri" id="cas_logout_uri" value="<?php echo htmlspecialchars($login_uri); ?>" size="50" /><br />
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_validate_uri">CAS Validate URI</label></th>
		  <td>
			<input type="text" name="cas_validate_uri" id="cas_validate_uri" value="<?php echo htmlspecialchars($validate_uri); ?>" size="50" /><br />
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_logout_uri">Logout URI</label></th>
		  <td>
			<input type="text" name="cas_logout_uri" id="cas_logout_uri" value="<?php echo htmlspecialchars($logout_uri); ?>" size="50" /><br />
			Default is <code><?php echo htmlspecialchars(get_settings('home')); ?></code>.
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_auto_create">Automatically create accounts</label></th>
		  <td>
			<input type="checkbox" name="cas_auto_create" id="cas_auto_create" <?php echo checked($auto_create, 'on'); ?>" value="on" />
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_allow_anon">Allow anonymous users</label></th>
		  <td>
			<input type="checkbox" name="cas_allow_anon" id="cas_allow_anon" <?php echo checked($allow_anon, 'on'); ?>" value="on" />
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_ldap_server">The URL to the LDAP server</label></th>
		  <td>
			<input type="text" name="cas_ldap_server" id="cas_ldap_server" value="<?php echo htmlspecialchars($ldap_server); ?>" size="50" />
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_ldap_base">The base DN of the LDAP server</label></th>
		  <td>
			<input type="text" name="cas_ldap_base" id="cas_ldap_base" value="<?php echo htmlspecialchars($ldap_base); ?>" size="50" />
		  </td>
		</tr>

		<tr valign="top">
		  <th scope="row"><label for="cas_ldap_filter">Additional filter for new users</label></th>
		  <td>
			<input type="text" name="cas_ldap_filter" id="cas_ldap_filter" value="<?php echo htmlspecialchars($ldap_filter); ?>" size="50" />
		  </td>
		</tr>

	  </table>
	</fieldset>
	<p class="submit">
	  <input type="submit" name="Submit" value="Update Options &raquo;" />
	</p>
  </form>
</div>
<?php
		}
	}
}

$cas_plugin = new CASAuthenticationPlugin();
