package nl.uvt.commons.io;

import java.io.*;
import java.net.*;
import java.util.*;

import org.apache.log4j.Logger;

import nl.uvt.commons.io.Http;
import nu.xom.*;

/**
 * Service class for HTTP related stuff.
 * 
 */
public class Http {

	private static Logger logger = Logger.getLogger(Http.class);

	/**
	 * Returns a parsed XML document based on a HTTP GET request with
	 * parameters.
	 * 
	 * @param url the base URL for the HTTP GET request
	 * @param args list of request parameters
	 * @param props HTTP properties
	 * @return parsed document
	 * 
	 * @throws UnsupportedEncodingException if a parameter value cannot be encoded into ISO 8859-1
	 * @throws MalformedURLException if a URL is not valid
	 * @throws IOException if the HTTP GET fails
	 * @throws ParsingException 
	 * @throws ValidityException 
	 */
	public static nu.xom.Document httpGetDocument( URL url, Properties args, Properties props )
	throws UnsupportedEncodingException, IOException, ParsingException, ValidityException {
		nu.xom.Document responseDocument = null;
		InputStream is = HttpGetInputStream(url, args, props );
		nu.xom.Builder builder = new Builder();
		responseDocument =  builder.build(is);
		is.close();
		return responseDocument;
	}

	public static nu.xom.Document httpGetDocument(URL url, Properties args )
	throws UnsupportedEncodingException, MalformedURLException,
	IOException, ValidityException, ParsingException {
		return httpGetDocument( url, args, null );
	}

	/**
	 * Perform a HTTP GET request and return InputStream.
	 * Based on: Jason Hunter & William Crawford, Java Servlet Programming, 2nd Edition,
	 * pp 316-319, O'Reilly
	 * 
	 * @param url URL of web service
	 * @param args key/value pairs
	 * @param httpProps HTTP properties as key/value pairs
	 * @return InputStream with response
	 */

	public static InputStream HttpGetInputStream( URL baseUrl, Properties args, Properties httpProps ) 
	throws UnsupportedEncodingException, IOException {
		URL fullUrl = getFullUrl( baseUrl, args );
		logger.debug("HttpGetInputStream from URL = '" + fullUrl + "'");
		URLConnection con = fullUrl.openConnection();
		if ( httpProps != null ){
			String useCache = httpProps.getProperty("useCache", "false");
			if ( useCache.equalsIgnoreCase("true") )
				con.setUseCaches(true);
			else
				con.setUseCaches(false);
			String keepAlive = httpProps.getProperty("Keep-Alive", "300");
			con.setRequestProperty("Keep-Alive", keepAlive  );
			String timeout = httpProps.getProperty("urlTimeout", "1500"); // take max 1.5 seconds, or what is configured
			con.setReadTimeout( Integer.parseInt(timeout) );
		}
		else {
			con.setRequestProperty("Keep-Alive", "300");
			con.setConnectTimeout(1500);
			con.setUseCaches(false);
		}
		return con.getInputStream();
	}


	public static InputStream HttpGetInputStream(URL url, Properties args)
	throws UnsupportedEncodingException, MalformedURLException,
	IOException {
		return HttpGetInputStream( url, args, null );
	}

	/**
	 * Transforms base URL + parameters into full URL.
	 * 
	 * @param url URL of web service
	 * @param args key/value pairs
	 * @return full URL
	 */
	public static URL getFullUrl(URL url, Properties args)
	throws UnsupportedEncodingException, MalformedURLException {
		String argString = "";
		if (args != null) {
			if(url.toString().contains("?")) {
				argString = "&" + toEncodedString(args);
			}
			else {
				argString = "?" + toEncodedString(args);	
			}
		}
		URL fullUrl = new URL(url.toExternalForm() + argString);
		return fullUrl;
	}

	/**
	 * Transforms key/value pairs into query string.
	 * 
	 * @param args Series of key/value pairs
	 * @return String like 'a=1&amp;b=2', without question mark
	 */

	public static String toEncodedString(Properties args, String encoding )
	throws UnsupportedEncodingException {
		StringBuffer buf = new StringBuffer();
		Enumeration<?> names = args.propertyNames();
		while (names.hasMoreElements()) {
			String name = (String) names.nextElement();
			String value = args.getProperty(name);
			buf.append(URLEncoder.encode(name, encoding ) + "="
					+ URLEncoder.encode(value, encoding ));
			if (names.hasMoreElements())
				buf.append("&");
		}
		return buf.toString();
	}

	/**
	 * 
	 * @param args Series of key/value pairs
	 * @return toEncodedString with UTF8 encoding
	 * @throws UnsupportedEncodingException
	 */
	public static String toEncodedString(Properties args)
	throws UnsupportedEncodingException {
		return toEncodedString(args, "UTF-8" );
	}

	public static Document buildDiagnosticDocument(Class<?> c, InputStream is) throws IOException {
		Element diagnosticElement = new Element("diagnostic");
		Element messageElement = new Element("message");
		messageElement.appendChild(c.getName() + " occurred");
		diagnosticElement.appendChild(messageElement);
		/*
		 * Unfortunately the following piece of code doesn't show the failing
		 * document in 'more' if wildcards are causing problems. Maybe with
		 * larger documents it will work:
		 */
		byte[] buffer = new byte[1024];
		int bytesRead;

		is.reset();
		if ((bytesRead = is.read(buffer)) != -1) {
			Element moreElement = new Element("more");
			StringBuffer more = new StringBuffer("");
			for (int i = 0; i < bytesRead; i++)
				more.append(buffer[i]);
			moreElement
			.appendChild("Part of response that couldn't be parsed: "
					+ more.toString());
			diagnosticElement.appendChild(moreElement);
		}


		Document document = new Document(diagnosticElement);
		return document;
	}

	/**
	 * Checks if given url exists.
	 * 
	 * @param url string representation of URL to test
	 * @return true if it exists (responses to HEAD request) and false if not
	 */
	public static boolean pageExists(URL url) throws MalformedURLException, IOException {
		HttpURLConnection con = (HttpURLConnection) url.openConnection();
		con.setRequestMethod("HEAD");
		return (con.getResponseCode() == HttpURLConnection.HTTP_OK);
	}

	/**
	 * Get mimetype from url
	 * @param url string representation of URL to test
	 * @return String mimetype
	 * 
	 */
	public static String getMimeType(URL url) throws MalformedURLException, IOException {
		HttpURLConnection con = (HttpURLConnection) url.openConnection();
		con.setRequestMethod("HEAD");
		return con.getContentType();
	}
	/**
	 * Sends a document to a service via HTTP POST.
	 * 
	 * @param url the service
	 * @param document document to send to the service
	 * @return parsed response from the service
	 * 
	 * @throws IOException
	 * @throws ParsingException
	 * @throws ValidityException
	 */
	public static nu.xom.Document httpPostDocument(URL url, Document document)
	throws IOException, ParsingException, ValidityException {
		HttpURLConnection connection = (HttpURLConnection) url.openConnection();
		connection.setRequestMethod("POST");
		connection.setDoOutput(true);
		connection.setDoInput(true);
		connection.setUseCaches(false);
		connection.setAllowUserInteraction(false);
		connection.setRequestProperty("Content-type", "text/xml; charset=UTF-8");

		OutputStream out = connection.getOutputStream();
		Writer writer = new OutputStreamWriter(out, "UTF-8");
		writer.write(document.toXML());
		writer.close();
		out.close();

		InputStream in = connection.getInputStream();
		Builder parser = new Builder();
		Document response = parser.build(in);
		in.close();

		connection.disconnect();

		return response;
	}
}
