# $Id: Directory.pm 48415 2019-10-10 16:06:09Z anton $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/pwdmodifier/libpwdmodifier/lib/UvT/PwdModifier/Directory.pm $

use strict;
#use warnings FATAL => 'all';

package UvT::PwdModifier::Directory::Account;

use Data::Dumper;
use MIME::Base64;
use Digest::SHA;
use Spiffy -Base;

field 'ldapEntry';
field 'cf';
field 'ldap';
field debug => 0;
field usePagedResults => 0;


sub orgstats {
	my %res;

	my $cf = $self->cf;
	my @config = $cf->value('ldap_orgstat');
	die $cf->error if $cf && $cf->error;

	my $attr = shift @config;
	foreach my $status ($self->ldapEntry->get_value($attr)) {
		foreach my $line (@config) {
			my ($name, $regex) = split(' ', $line, 2);
			undef $res{$name}
				if $status =~ /$regex/i;
		}
	}
	return (keys %res);
}

sub laus {
	return $self->ldapEntry->get_value('uvt-lau');
}

sub uid {
	return scalar $self->ldapEntry->get_value('uid');
}

sub mail {
	return scalar $self->ldapEntry->get_value('mail');
}

sub anr {
	return scalar $self->ldapEntry->get_value('employeeNumber');
}

sub accountStatus {
	return $self->ldapEntry->get_value('accountStatus');
}

sub accountSuspended {
	return $self->ldapEntry->get_value('accountSuspended');
}

sub privateEmail {
	return $self->ldapEntry->get_value('privateEmail');
}

sub tiasPrivateEmail {
	return $self->ldapEntry->get_value('tiasPrivateEmail');
}

sub accountTemporarilyDisabled {
	return $self->ldapEntry->get_value('accountTemporarilyDisabled');
}


sub stuff {

	my $joinArrays = shift;

#	warn "stuff: " ;
#	if ($joinArrays){
#		warn "joinArrays!";
#	} else {
#		warn "no joinArays!";
#	}
#	
	my $entry = $self->ldapEntry;

	my %userinfo;
	foreach my $attr (
		qw(
        accountBlocked
        accountStatus
        accountSuspended
        accountTemporarilyDisabled
        cn
        datePasswdChanged
        employeeNumber
        givenName
        mail
        mailhost
        passwordExpirationStage
        passwordExpirationFinalDay
        passwordExpirationAttention
        passwordExpirationWarning
        passwordExpirationReminder
        passwordExpirationLastReminder
        passwordExpirationExpired
        passwordExpirationAdminRequired
		passwordInvalidated
        passwordChangeCount
        passwordChangeDate
        passwordStatus
        privateEmail
        rcryptPassword
        sn
        tiasPrivateEmail
        uid            
        unpaidFees
         )) {
		$userinfo{$attr} = $entry->get_value($attr);
	}

    $userinfo{dn} = $entry->dn;
    $userinfo{orgstatus} = [$entry->get_value('organizationalStatus;lang-nl')];

	my $cf = $self->cf;
    $userinfo{orgstats} = $self->orgstats if $cf->value('ldap_orgstat');

    $userinfo{ou} = [$entry->get_value('ou;lang-nl')];
	$userinfo{'uvt-lau'} = [$entry->get_value('uvt-lau')];
	$userinfo{'uvt-auth'} = [$entry->get_value('uvt-auth')];

	$userinfo{entry} = $entry;

    my $uid = $userinfo{uid};
    $userinfo{studentaccount} = $uid =~ /^(s\d{6}|u\d{7})$/i;
	
	while (my ($key, $value) = each (%userinfo)){

#		warn "key: $key";
		if (defined($value)){
#			warn "value: $value";
			if (defined($joinArrays)) {
				if (ref($value) eq 'ARRAY'){
					$value = join(', ', @{$value});
					$value = '' unless $value;
					warn "new value: $value";
				}
			}
			if (ref($value eq 'SCALAR')){
				utf8::decode($value) or utf8::upgrade($value);
			}
			
		}
		$userinfo{$key} = $value;
	}

	return \%userinfo;
}

package UvT::PwdModifier::Directory;
use Net::LDAP::Control::Paged;
use Net::LDAP::Constant qw(LDAP_CONTROL_PAGED); 

use Data::Dumper;
use Carp qw(cluck carp confess);
use Spiffy -Base;

use Net::LDAP;
use Net::LDAP::Filter;

field 'cf';
field 'ldap';
field 'server';
field 'retries';
field 'capath';
field 'bindAccountFilter';
field 'bindAccount';
field 'bindPassword';
field 'ldapbase';
field debug => 0;
field usePagedResults => 0;
field 'ssl' => 1;
field 'tls' => 1;

sub ldapError {
	#callback heeft geen $self
	my $res = $self;
	warn "ldapError: ", $res->code;
	# No confess: a stack trace shows parameters like username and password printed on failure
	#  code 4 is sizelimit exceeded
	die( "ldapError: ". $res->error) unless $res->code == 4;
	return $res;
}

sub ldap_connection {
	my $cf = $self->cf;
	my $cfhash = {};
	$cfhash = $cf->hash() if $cf;

	my $ldap = $self->ldap;
	unless($ldap) {
		my $server = $cfhash->{'ldaphost'} || $self->server;
		my $capath = $cfhash->{'capath'} || $self->capath || '/etc/ssl/certs/';
		my $retries = $cfhash->{'ldapRetries'} || $self->retries || 1;
		my $ldapbase = $cfhash->{'ldapbase'} || $self->{ldapbase};
		$self->ldapbase($ldapbase);

		my $tls = $self->tls || 'off';
		my $ssl = $self->ssl || 'on';

		die "Inconsistent config: please choose ssl OR tls"
			if ($tls eq 'on' && $ssl eq 'on');
		
		my $state;
		my $debug = $self->debug;
		warn "debug: $debug" if $debug;
		warn "Connecting to ldapserver: $server" if $self->debug;

	  # onerror is o.a. NODIG om te reageren op foutieve authenticatie         
	  # nadeel is gvd dat je niet meer kunt zien waar je error vandaan komt :(

		
		if ($ssl eq 'on' ) {
			warn "ssl connection to: $server" if $self->debug;
			$ldap = Net::LDAPS->new(
					$server,
					onerror => \&ldapError,
					timeout => 10,
					verify => 'require',
					capath => $capath,
				);
	
		} else {		
			warn "PLANE connection to: $server" if $self->debug;
		 	$ldap = new Net::LDAP(
					$server,
					onerror => \&ldapError,
					timeout => 10,
					verify => 'require',
					raw => qr/(?i:^jpegPhoto|;binary)/,
					capath => $capath);
		
			$state = $@;
		
			if ($tls eq 'on') {
				warn "switching to TLS for server: $server" if $self->debug;
				$ldap->start_tls(verify => 'require', capath => $capath)
					unless $ldap->cipher;
			}
		}
		die "Error connecting to $server: $state" unless $ldap;
		warn "Succesful ldap connection with: $server: $ldap";
		$self->ldap($ldap);
	}
	my $bindAccountFilter = $self->bindAccountFilter;
	my $bindAccount = $self->bindAccount;
	
	if ($bindAccountFilter || $bindAccount) {
		unless ($bindAccount) {
			warn "Using filter: \"$bindAccountFilter\" to find bindAccount" if $self->debug;
			my $base = $self->ldapbase;
			my $msg = $ldap->search(base => $base, filter => $self->bindAccountFilter);
			my @entries = $msg->entries;
			
			return undef, "no match found for filter: '$bindAccountFilter'"
				unless @entries;
			
			return undef, "multiple entries found for filter: '$bindAccountFilter'"
				if @entries > 1;
			
			$bindAccount = $entries[0]->dn();
		}

		warn "bindAccount: " . $bindAccount if $self->debug; 
		$ldap->bind($bindAccount, password => $self->bindPassword);
		warn "Bind successful for: $bindAccount" if $self->debug; 

	} else {
		$ldap->bind;
		warn "Anonymous bind successful" if $self->debug;
	}
	return $ldap;
}


sub ldapSimpleSearch {
	 my ($filter) = @_;
	 my $ldap = $self->ldap_connection();
	 my $cf = $self->cf;
	 my $base = $self->ldapbase;
	 die $cf->error if $cf && $cf->error;
	 warn "ldapSimpleSearch: filter: $filter, base: $base" if $self->debug;
	 my $msg = eval {
		  $ldap->search(base => $base, filter => $filter);
	 };
	 my $err = $@;
	 if ($err) {
		 warn "ldapSimpleSearch Error: ,", Dumper($err);
	 }
	 return $msg;
}

sub ldap_search_filter {
	my ($filter) = @_;

	my $msg = $self->ldapSimpleSearch(@_);
	my @entries = $msg->entries;

	return undef, "no match found for filter: '$filter'"
		unless @entries;

	return undef, "multiple entries found for filter: '$filter'"
		if @entries > 1;

	return $entries[0], undef;
}

sub ldap_search_uid {
	my ($uid) = @_;

	my $filter = bless(
			{equalityMatch => {attributeDesc => 'uid', assertionValue => $uid}},
		'Net::LDAP::Filter')->as_string;

	return $self->ldap_search_filter($filter);
}

sub ldap_search_anr {
	my ($anr) = @_;

	my $filter = bless(
			{equalityMatch => {attributeDesc => 'employeeNumber', assertionValue => $anr}},
		'Net::LDAP::Filter')->as_string;

	return $self->ldap_search_filter($filter);
}



sub verify_password {
	my ($filter, $password, $trySHA) = @_;
	my $ldap = $self->ldap_connection();
	my $debug = $self->debug;

	if ($filter !~ /\=/) {
		$filter = "uid=$filter";
	}

	my ($entry, $err) = $self->ldap_search_filter($filter);
	return $err unless ($entry);

	my $dn = $entry->dn;
	eval { 
		local $SIG{__DIE__};
		$ldap->bind($dn, password => $password) 
	};
	$err = $@;
	# fixme, verander dit in 'getFallbackPassword'

	warn "Error: $err" if $err;

	if ($err and $trySHA) {
		my $ctx = Digest::SHA->new;
		$ctx->add($password.$trySHA);
		my $digest = $ctx->digest;
		my $converted = MIME::Base64::encode_base64($digest);
		# vergeet niet de \n eraf te slopen!
		chomp $converted;
		warn "converted: $converted" if $debug;
		eval { 
			local $SIG{__DIE__};
			$ldap->bind($dn, password => $converted) 
		};
		$err = $@;
	}
	# Maak er weer een anonymous bind van
	$ldap->bind;
	return $err;
}


sub getAccount {
	my $uid = shift;
	my ($entry, $err) = $self->ldap_search_uid($uid);

	return undef, $err
		unless $entry;

	return new UvT::PwdModifier::Directory::Account
		(ldapEntry => $entry, cf => $self->cf, ldap => $self, debug => $self->debug, usePagedResults => $self->usePagedResults), 
		undef;
}

sub check_uvt_auth {
	my ($admin, $auths) = @_;
	my $ldap = $self->ldap_connection();

	# (&(uid=bob)(|(uvt-auth=pwd_change/foo)(uvt-auth=pwd_change/foo/bar)))
	my $filter = bless(
			{and => [{equalityMatch => {attributeDesc => 'uid', assertionValue => $admin}},
				{or => [map { {equalityMatch => {attributeDesc => 'uvt-auth', assertionValue => $_}} } @$auths]}
		]}, 'Net::LDAP::Filter')->as_string;

	# 3) zoek op dat filter
	my ($entry, $err) = $self->ldap_search_filter($filter);
	return $entry ? undef : $err;
}

sub getLdapEntriesByFilter {
 	my ($filter, $attrs) = @_;
	$attrs = [] unless $attrs;
	my $cf = $self->cf;

	my $ldap = $self->ldap_connection();
	my $base = $self->ldapbase;

	warn "getLdapEntriesByFilter; filter: $filter" if $self->debug;
	# LET OP, retourneert een REFerence naar lijst met entries
	my $entries =  $self->searchLdap(base => $base, filter => $filter, attrs => $attrs);
}

sub getUIDsByFilter {
	my ($filter) = @_;
	my $entries = $self->getLdapEntriesByFilter($filter, ['uid']);
	return [map {$_->get_value('uid')} @$entries];
}


sub listTemporarilyDisabled {
	my $filter = '(AccountTemporarilyDisabled=*)';
	return $self->getUIDsByFilter($filter);
}

sub listSuspended {
	# verander * eventueel in een anr om te testen
#	my $filter = '(&(accountSuspended=*)(userpassword=*)(employeeNumber=*))';
	my $filter = '(&(accountSuspended=*)(!(rcryptpassword=suspended*))(employeeNumber=*))';

	return $self->getUIDsByFilter($filter);
}

sub listUnsuspended {
#	my $filter = '(&(objectclass=uvtUser)(employeeNumber=*)(!(accountSuspended=*))(!(userpassword=*)))';
	my $filter = '(&(employeeNumber=*)(!(accountSuspended=*))(|(!(userpassword=*))(rcryptpassword=suspended*)))';
	return $self->getUIDsByFilter($filter);
}

sub searchLdap {
	my $ldap = $self->ldap_connection();
	my @aggregatedEntries = ();

	if ($self->usePagedResults) {
		my $page = Net::LDAP::Control::Paged->new( size => 500 );
		while (1) {
			my $search = $ldap->search(@_, control => [$page]);
			push (@aggregatedEntries, $search->entries);
			my ($resp) = $search->control( LDAP_CONTROL_PAGED ) or last;
			my $cookie = $resp->cookie or last;
			$page->cookie($cookie);
		}
		
	} else {
		my $search = $ldap->search(@_);
		my @entries = $search->entries;
		push(@aggregatedEntries, @entries);
	}
	return \@aggregatedEntries;
}

sub getEntriesByUID {
	my $uid = shift;
	warn "getEntriesByUid: $uid" if $self->debug;
	my $ldap = $self->ldap_connection();
	my $cf = $self->cf;
	my $base = $self->ldapbase;
	die $cf->error if $cf && $cf->error;

	my $filter = bless(
			{equalityMatch => {attributeDesc => 'uid', assertionValue => $uid}},
		'Net::LDAP::Filter')->as_string;

	my $entries = $self->searchLdap(base => $base, filter => $filter);
	die "no match found for filter: '$filter'"
		unless @$entries;

	die "multiple entries found for filter: '$filter'"
		if @$entries > 1;

	return @$entries[0];
}


sub getEntryByUID {
	my ($uid) = @_;
	my $ldap = $self->ldap_connection();
	return ($self->getEntriesByUID($uid), $ldap);
}

sub getRcryptPassword {
	my ($uid) = @_;
	my ($entry, $ldap) = $self->getEntryByUID($uid);
	return $entry->get_value('rcryptPassword');
}

sub toggleAccountState {
	my ($attrib, $uid, $on, $by) = @_;
	# ignore label
	my ($entry, $ldap) = $self->getEntryByUID($uid);

	if ($on) {
		my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime(time);
		$year += 1900;  $mon += 1;
		my $content = sprintf("$year-%02d-%02d %02d:%02d $by",$mon, $mday, $hour, $min);
		$entry->replace($attrib, $content);
	} else {
		# throws an exception it the attrib does not exist \o/
		$entry->delete($attrib) if $entry->get_value($attrib);
	}

	my $msg = $entry->update($ldap);
	if ($msg->code) {
		warn("toggleAccountState failed:", $msg->error());
	} else {
		warn "toggleAccountState succeeded" if $self->debug;
	}
}

sub replaceAttribute {
	my ($uid, $attrib, $value) = @_;
#	warn "replaceAttribute; uid: $uid, attrib: \"$attrib\", value: \"$value\"";
	my ($entry, $ldap) = $self->getEntryByUID($uid);
	if (defined($value)) {
		$entry->replace($attrib, $value);
	} else {
		$entry->delete($attrib);
	}
	my $msg = $entry->update($ldap);
	
	if ($msg->code) {
		die("Error setting attrib: $attrib", $msg->error());
	}
}


sub resetPasswordExpirationStatus {
	my $userinfo = shift;
	my $uid = $userinfo->{uid};
	if ($userinfo->{passwordExpirationStage}) {
		map {$self->replaceAttribute($uid, $_, undef) if $userinfo->{$_}} 
		qw(passwordExpirationAttention
		   passwordExpirationWarning
		   passwordExpirationReminder
		   passwordExpirationLastReminder
		   passwordExpirationExpired
		   passwordExpirationAdminRequired
		   passwordExpirationFinalDay
		   passwordExpirationStage
          );
	}
}

sub removeEnclosingBrackets {
	local $_ = shift;

	while (/^\(/ and /\)$/){
		s/^\(//;
		s/\)$//;
	}
	$_;
}

sub ldapEncodeFilter {
	#
	# GEPIKT van ldapbrowser:Ldapsearch.pm:ldapsearchencode
	#
	#   http://www.cse.ohio-state.edu/cgi-bin/rfc/rfc2254.html
	#
	#   If a value should contain any of the following characters
	#
	#             Character       ASCII value
	#             ---------------------------
	#             *               0x2a
	#             (               0x28
	#             )               0x29
	#             \               0x5c
	#             NUL             0x00
	#
	#     the character must be encoded as the backslash '\' character (ASCII
	#     0x5c) followed by the two hexadecimal digits representing the ASCII
	#     value of the encoded character.
    # comfort emacs ;(
	#  map {$specialchars->{$_}=1 } split (//,'\()\*');

	# Het simpelst is gewoon ALLES coderen, maar dat vinden mensenogen niet fijn.
	# .. who cares?..
	#  my $encoded='\\'.join('\\', map {sprintf ("%x",$_)} unpack('U*',$str));

	#
	# alternatief: codeer alleen de noneword characters,
	# NB. blijf van de wildcard '*' af!
	#
    my $str = shift;
	my $res = '';
	if ($str) {
		foreach my $char (split (//,$str)) {
			if ($char =~ /\W/ and $char !~ /\*/) {
				$res .= '\\' . sprintf("%x",ord($char));
			} else {
				$res .= $char;
			}
		}
	}
	$res;
}




0;
