# $Id: Conduit.pm 40762 2013-12-20 12:29:19Z anton $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/pwdmodifier/libpwdmodifier/lib/UvT/PwdModifier/Conduit.pm $
use strict;
use warnings FATAL => 'all';

package UvT::PwdModifier::Conduit;

use Spiffy -Base;
use XML::LibXML;
use IO::Pipe;
use POSIX qw(dup2 STDIN_FILENO STDOUT_FILENO STDERR_FILENO _exit :sys_wait_h);

field 'command', -init => 'die "no command configured\n"';

sub handle_status() {
	my $status = shift;
	my $prog = join(' ', @_);
	if(WIFEXITED($status)) {
		my $status = WEXITSTATUS($status);
		die sprintf("%s exited with status %d\n", $prog, $status)
			if $status;
	} elsif(WIFSIGNALED($status)) {
		my $sig = WTERMSIG($status);
		die sprintf("%s killed with signal %d%s\n", $prog, $sig & 127, ($sig & 128) ? ' (core dumped)' : '')
	} elsif(WIFSTOPPED($status)) {
		my $sig = WSTOPSIG($status);
		warn sprintf("%s stopped with signal %d\n", $prog, $sig)
	}
}

sub xml2hash() {
	my $doc = shift;
	my %res;
	my $root = $doc->documentElement;
#	warn "xml2hash";
#	warn $doc->toStringHTML();
	foreach my $node ($root->childNodes) {
		my $name = $node->nodeName;
		next unless $name;
		die "node '$name' must be unique"
			if exists $res{$name};
		$res{$name} = $node->textContent;
	}

	return $root->nodeName, \%res;
}

use Carp qw(cluck);

use Data::Dumper;
sub hash2xml() {
	my $message = shift;
	my $hash =  shift;

#	cluck ("Message: \"$message\"");
	utf8::upgrade($message);
#	warn Dumper($message);

	my $doc = new XML::LibXML::Document('1.0', 'UTF-8');
	my $root = $doc->createElement($message);
	$doc->setDocumentElement($root);

	while(my ($key, $value) = each(%$hash)) {
		utf8::upgrade($key);
		$value = '' unless defined ($value);
		utf8::upgrade($value);
		$root->appendTextChild($key, $value);
	}
	return $doc;
}

sub start_feeder {
	my $in = shift;

#	warn "toString";
#	warn $in->toString;

	my $stdin = new IO::Pipe;

	my $feeder = fork();
	die "fork(): $!" unless defined $feeder;

	if($feeder) {
		$stdin->reader;
		return $feeder, $stdin;
	}

	eval {
		$stdin->writer;
		binmode($stdin) or die $!;
		$in->toFH($stdin)
			or die $!;
		$stdin->flush;
		_exit(0);
	};
	warn $@;
	_exit(1);
}

sub start_logger {
	my $stderr = new IO::Pipe;

	my $logger = fork();
	die "fork(): $!" unless defined $logger;

	if($logger) {
		$stderr->writer;
		return $logger, $stderr;
	}

	eval {
		$stderr->reader;
		while(defined(my $line = $stderr->getline)) {
			warn $line;
		}
		_exit(0);
	};
	warn $@;
	_exit(1);
}

sub start_runner {
	my ($stdin, $stderr) = @_;
	my $stdout = new IO::Pipe;
	my $runner = fork();
	die "fork(): $!" unless defined $runner;

	if($runner) {
		$stdout->reader;
		return $runner, $stdout;
	}

	eval {
		$stdout->writer;

		dup2($stdin->fileno, STDIN_FILENO)
			or die $!;

		dup2($stdout->fileno, STDOUT_FILENO)
			or die $!;

		dup2($stderr->fileno, STDERR_FILENO)
			or die $!;

		exec @{$self->command};
		die $!;
	};
	warn $@;
	_exit(1);
}

sub engage {
	my ($feeder, $logger, $runner, $output);
	do {
		($feeder, my $stdin) = $self->start_feeder(hash2xml(@_));
		($logger, my $stderr) = $self->start_logger;
		($runner, $output) = $self->start_runner($stdin, $stderr);
	};

	my $parser = new XML::LibXML;
	$parser->pedantic_parser(1);
	$parser->validation(0);
	$parser->load_ext_dtd(0);
	$parser->expand_entities(0);
	$parser->keep_blanks(0);
	$parser->line_numbers(1);

	binmode($output) or die $!;
	my $out = $parser->parse_fh($output);

	die "waitpid(): $!"
		if waitpid($feeder, 0) == -1;
	my $feeder_status = $?;

	die "waitpid(): $!"
		if waitpid($runner, 0) == -1;
	my $runner_status = $?;

	die "waitpid(): $!"
		if waitpid($logger, 0) == -1;
	my $logger_status = $?;

	handle_status($runner_status, @{$self->command});
	handle_status($feeder_status, 'feeder');
	handle_status($logger_status, 'logger');

	return xml2hash($out);
}
