# $Id: ControlTower.pm 41360 2014-04-17 14:44:59Z anton $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/pwdmodifier/libpwdmodifier/lib/UvT/PwdModifier/ControlTower.pm $
use strict;
package UvT::PwdModifier::ControlTower;
use Spiffy -Base;
use Carp qw(cluck);
use warnings FATAL => 'all';
use Data::Dumper;
field 'userinfo';
field 'dir';
field debug => 0;
field readonly => 0;

# Commands:
# keymaster.in

my $passwordExpirationHash = {
	'attention' => 0,
	'warning' => 1,
	'last-warning' => 2,
	'expired' => 3,
	'admin-required' => 4,
};

sub needlessRequest {
	my ($uid, $passwordStatus, $command) = @_;
	$self->{fields}->{message} = 'NeedlessRequest';
	$self->{fields}->{details} = "Command: $command not needed for uid: $uid, since passwordStatus is already $passwordStatus";
}

sub error {
	return $self->{fields}->{error};
}

sub passwordStatusFromExpiration {
	my $userinfo = $self->userinfo;
	my $passwordExpirationStage = $userinfo->{passwordExpirationStage};
	if ($self->debug) {
		warn "passwordExpirationStage";
		warn $passwordExpirationStage if $passwordExpirationStage;
	}

	return 'normal' unless defined ($passwordExpirationStage);
	return 'set-only' if $passwordExpirationStage eq 'expired';
	return 'closed' if $passwordExpirationStage eq 'admin-required';
	return 'normal';
}

sub setPasswordStatus {
	# Dit wordt aangeroepen nadat een commando met succes is uitgevoerd om het attribuut 
	# passwordStatus te zetten conform dat commando.
	# 
	my $command = shift;
	my $userinfo = $self->userinfo;
	warn "no userinfo" && return undef unless ($userinfo);

	my $passwordStatus = $self->{fields}->{passwordStatus} // '';
	my $passwordExpirationStage = $userinfo->{passwordExpirationStage} // '';
	my $requestedPasswordStatus = $passwordStatus;

  commandSwitch: {
	  $command =~ /^blockaccount|^disablePassword/i && do {$requestedPasswordStatus = 'closed'; last commandSwitch};
	  $command =~ /^settmppwd/i && do {$requestedPasswordStatus = 'hijacked'; last commandSwitch};
	  $command =~ /^unblockaccount|^unsettmppwd/i && do {$requestedPasswordStatus = 'normal'; last commandSwitch};
	  $command =~ /^setpwd|^syncpassword/i && do {$requestedPasswordStatus = 'normal'; last commandSwitch};
	  $command =~ /^pwdexpire/i && do {$requestedPasswordStatus = $self->passwordStatusFromExpiration(); last commandSwitch};
	  warn "ControlTower: command \"$command\" does not influence passwordStatus" if $self->debug;
	  return;
	}

	my $dir = $self->dir;
	my $uid = $userinfo->{uid};
	if ($self->debug) {
		warn "ControlTower:SetPasswordStatus, command: $command";
		cluck "passwordStatus-> $passwordStatus requestedPasswordStatus: $requestedPasswordStatus";
	}

	$dir->replaceAttribute($uid, 'passwordStatus', $requestedPasswordStatus) 
		if ($requestedPasswordStatus ne $passwordStatus);
}

sub regulateTraffic {
	my $command = shift;
	warn "ControlTower::regulateTrafic; command: $command" if $self->debug;

	my $userinfo = $self->userinfo;
	my $passwordStatus = $userinfo->{passwordStatus} // '';
	my $passwordExpirationStage = $userinfo->{passwordExpirationStage} // '';
	my $accountStatus = $userinfo->{accountStatus} // '';
	my $accountBlocked = $userinfo->{accountBlocked};
	my $accountTemporaryDisabled = $userinfo->{accountTemporaryDisabled};
	my $unpaidFees = $userinfo->{unpaidFees};
	my $uid = $userinfo->{uid};

	my $admin = 0; 
	if ($userinfo->{changed_by}) {
		$admin = 1 if $userinfo->{changed_by} ne '_self_';

		if ($self->debug){
			warn "changed_by: $userinfo->{changed_by}";
			warn "adminStatus: $admin";
		}
	}

	my $implicitPasswordStatus;
	
  switch: {
	  $accountStatus =~ /disabled|exit/i && do { $implicitPasswordStatus = 'closed'; last switch};
	  $accountBlocked && do { $implicitPasswordStatus = 'closed'; last switch};
	  $unpaidFees && do { $implicitPasswordStatus = 'closed'; last switch};
	  $accountTemporaryDisabled && do { $implicitPasswordStatus = 'hijacked'; last switch};
	  $passwordExpirationStage && do { $implicitPasswordStatus = $self->passwordStatusFromExpiration(); last switch};
	  $implicitPasswordStatus = 'normal';
	}

	warn "implicitPasswordStatus: $implicitPasswordStatus" if $self->debug;

	if ($passwordStatus) {
		if ($passwordStatus ne $implicitPasswordStatus){
			warn "Unexpected passwordStatus: $passwordStatus, expected: $implicitPasswordStatus";
		}
	} else {
		$passwordStatus = $implicitPasswordStatus;
	}
	

	unless($self->readonly) { 
		unless ($passwordStatus) {
			# No passwordStatus in ldap yet: write it now.
			warn "Writing new passwordStatus: $implicitPasswordStatus for $uid";
			$self->dir->replaceAttribute($uid, 'passwordStatus', $implicitPasswordStatus);
		}

		# Clean up traces of previous versions.
		my $rcryptPassword = $userinfo->{rcryptPassword};
		if ($rcryptPassword && $rcryptPassword =~ s/^\w+\://) {
			$self->dir->replaceAttribute($uid, 'rcryptPassword', $rcryptPassword);
		}
	}
	
#	$self->{fields}->{passwordStatus} = $passwordStatus;
	warn("ControlTower:regulateTraffic; passwordStatus: $passwordStatus") if $self->debug;
	my @denied = ();
	
  statusSwitch: {
	  $passwordStatus =~ /^closed/i 
		  && do {
			  last statusSwitch if $command =~ /qamstatus/i and $admin;

			  if ($command =~ /^disablePassword/i) { 
				  $self->needlessRequest($uid, $passwordStatus, $command);
				  last statusSwitch;
			  }

			  if ($accountBlocked and $command !~ /unblockAccount|getTblock/i) {  
				  push(@denied, "Account Blocked at: $accountBlocked");
				  push(@denied, 'contactHelpdesk') unless $admin;
				  last statusSwitch;
			  }

			  if ($accountStatus =~ /^disabled|^exit/i ) {
				  push(@denied, "accountStatus $accountStatus");
				  push(@denied, 'contactHelpdesk') unless $admin;
				  last statusSwitch;
			  } 

			  if ($unpaidFees){
				  push(@denied, "UnpaidFees $unpaidFees");
				  push(@denied, 'contactStudentDesk');
			  }
			  last statusSwitch;
	  };

	  $passwordStatus =~ /set-only/i 
		  && do {
			  push(@denied, 'contactStudentDesk') if $command =~ /setQAM/i;
			  # laat verder alles doorgaan
			  last statusSwitch;
	  };

	  $passwordStatus =~ /admin\-required/i 
		  && do { 
			  last statusSwitch if $command =~ /qamstatus/i;
			  push(@denied, 'adminRequired') unless $admin;
			  last statusSwitch;
	  };

	  $passwordStatus =~ /^normal/i 
		  && do { 
			  $self->needlessRequest($uid, $passwordStatus, $command) if $command =~ /^unblockAccount|^unsetTmpPwd/i; 
			  last statusSwitch;
	  };

	  $passwordStatus =~ /^hijacked/i 
		  && do {
			  my $date = $accountTemporaryDisabled;

			  last statusSwitch  
				  if ( $command =~ /^unsetTmpPwd|^getTsetpwd|^getTblock|^blockAccount|^disablePassword|qamstatus/i 
					   and 
					   $admin
				  );

			  push(@denied, "Account temporary disabled at: $date");
			  push(@denied, 'adminRequired') unless $admin;
			  last statusSwitch;

	  };
	}

	if (@denied) {
#		my $reasons = join(' ', @denied);
		$self->{fields}->{details} = "RegulateTrafficReport: Command: \"$command\" denied for account: \"$uid\"";
		my @messages;
		push (@messages, 'requestDenied');
		push (@messages, @denied);
		$self->{fields}->{message} = [@messages];

		$self->{fields}->{error} = "requestDenied" unless $self->{fields}->{error};
	} else {
		warn "RegulateTraficReport: Command: $command on $uid is allowed." if $self->debug;
	}
}
