package nl.uvt.locator;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Iterator;
import java.util.List;
import java.util.Collections;
import java.util.Properties;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.Comparator;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nu.xom.Document;
import nu.xom.Element;
import nu.xom.Node;
import nu.xom.Nodes;
import nu.xom.Text;
import nu.xom.XPathContext;

import org.apache.log4j.*;

public class Locator extends HttpServlet {

	public static final String SOAPNamespace   = "http://schemas.xmlsoap.org/soap/envelope/";
	public static final String SRWNamespace    = "http://www.loc.gov/zing/srw/";
	public static final String diagNamespace   = "http://www.loc.gov/zing/srw/diagnostic/";

	static XPathContext sruContext = new XPathContext();

	private void fillContext(){
		sruContext.addNamespace("SOAP-ENV", SOAPNamespace);
		sruContext.addNamespace("srw", SRWNamespace);
		sruContext.addNamespace("diag", diagNamespace);
	}

	public static Properties sruDefaultProps = new Properties();
	public static Properties locatorConfig = new Properties();
	public static Properties httpConfig = new Properties();

	public static Logger logger = Logger.getLogger(Locator.class);

	static private String [] prefixPaths;
	static String myName;
	static String myHostName;
	static String myVersion;
	static String myConfigName;

	static Set<String> DOImethods = null;
	static Set<String> knownCollections = null;
	static Set<String> knownDepots = null;
	static Set<String> stopWords = null;
	static String filterChars = null;
	static Set<String> forgetAbout = null; // this global option is for testing purposes only!

	static boolean doMeresco = true;
	static boolean htlIndexes = false;
	static boolean doAvailability = true;
	static boolean cascadingSearch = false;

	private void setSRUDefaults(){
		sruDefaultProps.setProperty("operation", "searchRetrieve");
		sruDefaultProps.setProperty("version", "1.1");
		if ( locatorConfig.getProperty("sruMaximumRecords") == null )
			sruDefaultProps.setProperty("maximumRecords", "5");
		else
			sruDefaultProps.setProperty( "maximumRecords", 
					locatorConfig.getProperty("sruMaximumRecords"));
	}

	static public File getFile( String name ){
		return getFile( prefixPaths, name );
	}

	static	private File getFile( String[] prefixes, String name ){
		try {
			int i=0;
			while ( i < prefixes.length ){
				String fullName = prefixes[i] + File.separator + name;
				File file = new File( fullName );
				if ( file.exists() ){
					return file;
				}
				++i;
			}
		}
		catch ( Exception e ){
		}
		return null;
	}

	private void readConfig( String name, Properties p ){
		readConfig( name, p, false );
	}

	private void readMainConfig( String name, Properties p ){
		readConfig( name, p, true );
	}

	static Map<String,Long > modTimes = new HashMap<String,Long>();

	private void readConfig( String name, Properties result, Boolean main ){
		try {
			File configFile = getFile( prefixPaths, name );
			if ( configFile != null ){
				long moDate = configFile.lastModified();
				long oldDate = 0;
				if ( modTimes.containsKey(name) ){
					oldDate = modTimes.get(name);
				}
				//				logger.debug("file " + name + " old date was: " + oldDate + " moDate=" + moDate );
				if ( moDate > oldDate ){
					logger.debug("(re)read configfile:" + name );
					FileInputStream is = new FileInputStream(configFile);
					result.clear();
					result.load( is );
					if ( main ){
						myConfigName = configFile.getAbsolutePath();
						DOImethods = null;
						knownCollections = null;
						knownDepots = null;
						stopWords = null;
						filterChars = null;
						forgetAbout = null;
						doMeresco = true;
						htlIndexes = false;
						doAvailability = true;
						cascadingSearch = false;
					}
					modTimes.put(name, moDate);
					logger.debug("config:" + name + "=" + result.toString());
				}
			}		
		}
		catch ( Exception e ){
			logger.error("readConfig problem: "+e.getMessage() );
		}
	}	

	// For tests:
	public static void setKnownCollections(Set<String> collections) {
	  knownCollections = collections;
	}
	
	public void init() throws ServletException{
		myName = getServletContext().getInitParameter("artifactId" );
		myVersion = getServletContext().getInitParameter("version" );
		String prefixPath = getServletContext().getInitParameter("prefixPath" );
		if ( prefixPath.equals("") )
			throw new ServletException("no prefixPath in web.xml found" );
		prefixPaths = prefixPath.split(":");
		if ( prefixPaths.length == 0 ){
			throw new ServletException("no prefix paths for configuration found" );
		}
		String configName = null;
		File l4j = getFile( "log4j.properties");
		if ( l4j != null){
			configName = l4j.getAbsolutePath();
			PropertyConfigurator.configureAndWatch( configName );
		}
		else {
			BasicConfigurator.configure();
		}
		logger.info("initialize Locator, using config from PATH=" + prefixPath );
		if ( configName != null )
			logger.debug("log4j configured from " + configName );
		else
			logger.warn("couldn't get log4j.properties. logging to System.out");
		fillContext();
		logger.info("started " + myName + "-" + myVersion );
	}

	enum serveAction {explainAct, lookupAct };

	public void doGet( HttpServletRequest request, 
			HttpServletResponse response ) throws ServletException, IOException {
		myHostName = request.getServerName();
		OutputStream os = response.getOutputStream();
		PrintStream out = new PrintStream( os, true, "UTF8" );
		String fileName = "locator.cfg";
		readMainConfig( fileName, locatorConfig );
		if ( locatorConfig.isEmpty() ){
			out.println("unable to read configfile " + fileName );
			logger.error("reading configfile '" + fileName + "' failed");
			return;
		}
		String httpSettings = locatorConfig.getProperty("httpDefaults");
		if ( httpSettings != null ){
			fileName = httpSettings;
			readConfig(fileName, httpConfig );
			if ( httpConfig.isEmpty() ){
				logger.warn("reading http configfile '" + fileName  + "' failed. (using defaults)");
			}
		}
		setSRUDefaults();
		response.setContentType("text/XML");
		String prop = locatorConfig.getProperty("doMeresco", "true");
		if ( prop.equalsIgnoreCase("false") ){
			doMeresco = false;
		}
		else {
			doMeresco = true;
		}
		prop = locatorConfig.getProperty("doAvailability", "true");
		if ( prop.equalsIgnoreCase("false") ){
			doAvailability = false;
		}
		else 
			doAvailability = true;
		prop = locatorConfig.getProperty("htlIndexes", "false");
		if ( prop.equalsIgnoreCase("true") ){
			htlIndexes = true;
		}
		else 
			htlIndexes = false;
		prop = locatorConfig.getProperty("cascadingSearch", "false");
		if ( prop.equalsIgnoreCase("true") ){
			cascadingSearch = true;
		}
		else
			cascadingSearch = false;
		Document doc = null;
		String holding = request.getParameter( "holdingDb" );
		if ( holding != null ){
			if ( holding.equalsIgnoreCase("tucat") ){
				doMeresco = false;
			}
			else if ( holding.equalsIgnoreCase("meresco") ){
				doMeresco = true;
			}
			else {
				logger.warn("invalid holdingDb:"+holding);
				doc = errorDoc( "invalid holdingDb:" + holding );
			}
		}
		String avail = request.getParameter( "doAvailability" );
		if ( avail != null ){
			if ( avail.equalsIgnoreCase("true") ){
				doAvailability = true;
			}
			else if ( avail.equalsIgnoreCase("false") ){
				doAvailability = false;
			}	
			else {
				logger.warn( "invalid doAvailability:" + avail );
				doc = errorDoc( "invalid doAvailability:" + avail );
			}
		}
		serveAction act = serveAction.lookupAct; 	
		if ( doc == null ){
			String action = request.getParameter( "action" ); 	 
			if ( action != null ){ 	 
				logger.debug("special_action="+action); 	 
				if ( action.equals("explain") ) 	 
					act = serveAction.explainAct; 	 
				else if ( action.equals("cascading") ){
					cascadingSearch = true;
				}
				else  {
					doc = errorDoc( "invalid action:" + action );
				}
			}
		}
		if ( doc == null ){
			String query = request.getQueryString();
			if ( act == serveAction.explainAct || query == null ){
				doc = explainDoc();
			}
			else { // lookupAct
				QueryData pQ = new QueryData();
				try {
					if ( pQ.parseQuery( request ) ){
						logger.debug( "parsed query: " + pQ );
						if ( pQ.genre.equals("journal") && pQ.epn.isEmpty() )
							doc = errorDoc("genre=journal without an EPN is not handled by Locator!");
						else if ( pQ.genre.equals("article") )
							doc = tryToLocateArticle( pQ );
						else
							doc = tryToLocateBook( pQ );
						if ( doc == null ){
							IllRequestor illRequestor = new IllRequestor();
							doc = illRequestor.getDocument( pQ ); 
						}
					}
					else
						doc = errorDoc("parseQuery failed");
				}
				catch (LocatorException e) {
					doc = errorDoc( "locator failed: ", e.getMessage());
				}
			}
		}
		if ( locatorConfig.getProperty("transactionLogging", "false" ).equals("true") )
			logger.info("resulting Doc: " + doc.toXML());
		out.println( doc.toXML() );		
	}

	private Document tryToLocateArticle( QueryData pQ ){
		List<LocatorInfo> locInfo = null;
		try {
			LinkDbConnection linkDb = new LinkDbConnection();
			locInfo = linkDb.query( pQ );
		}
		catch ( LocatorException e ){
			logger.warn(e.getMessage());
			return errorDoc( "linkDb failed", e.getMessage() );
		}
		if ( locInfo != null ){
			for ( Iterator<LocatorInfo> it=locInfo.iterator(); it.hasNext(); ){
				LocatorInfo li = it.next();
				if ( !li.applyMethod( pQ ) ){
					it.remove();
				}
			}
		}
		List<LocatorInfo> tuInfo = null;
		if ( cascadingSearch || locInfo == null || locInfo.isEmpty() ){
			if ( cascadingSearch )
				logger.debug("cascading search continues with holding db" );
			else
				logger.debug("no valid linkdb results, try holding db");
			try {
				tuInfo = locateHolding( pQ );
			} catch (LocatorException e) {
				return errorDoc( "holding db failed", e.getMessage() );
			}
		}
		if ( tuInfo != null ){
			if (locInfo != null )
				locInfo.addAll( tuInfo );
			else
				locInfo = tuInfo;
		}
		if ( locInfo != null && !locInfo.isEmpty() ){
			return toDoc(locInfo );
		}
		else 
			return null;
	}

	private Document tryToLocateBook( QueryData pQ ){
		List<LocatorInfo> locInfo = null;
		try {
			locInfo = locateHolding( pQ );
		}
		catch ( LocatorException e ){
			return errorDoc( "holding db lookup failed", e.getMessage() );
		}
		if ( locInfo != null && !locInfo.isEmpty() ){
			return toDoc(locInfo );
		}
		else
			return null;
	}

	private List<LocatorInfo> locateHolding( QueryData pQ ) throws LocatorException { 
		List<LocatorInfo> locInfo = null;
		if ( doMeresco ){
			MerescoConnection bzv = new MerescoConnection();
			locInfo = bzv.query( pQ );
		}
		else {
			TucatConnection tucat = new TucatConnection();
			locInfo = tucat.query( pQ );
		}
		if ( locInfo != null ){
			for ( Iterator<LocatorInfo> it=locInfo.iterator(); it.hasNext(); ){
				LocatorInfo li = it.next();
				if ( !li.applyMethod( pQ ) ){
					it.remove();
				}
			}
		}
		return locInfo;
	}

	class rankComparator implements Comparator<LocatorInfo> {
		public int compare( LocatorInfo le1, LocatorInfo le2 ){
			int rank1 = ((LocatorInfo)le1).rank;
			int rank2 = ((LocatorInfo)le2).rank;
			if ( rank1 > rank2 )
				return 1;
			else if ( rank1 < rank2 )
				return -1;
			else
				return 0;
		}
	}

	public Document toDoc( List<LocatorInfo> locInfo ){
		Element root = new Element("records");
		Document doc = new Document( root );
		if ( locInfo != null ){
			Collections.sort( locInfo, (Comparator<LocatorInfo>) new rankComparator() ); 
			for ( int i=0; i< locInfo.size(); ++i ){
				Node liNode = locInfo.get(i).toRecordNode();
				root.appendChild( liNode );
			}
		}
		return doc;
	}

	static public int extractHits( Document doc ){
		Nodes numOfRecords = doc.query("//srw:numberOfRecords", sruContext);
		int hits =0;
		if( numOfRecords.size() > 0 ){
			Element numHitsElement = (Element) numOfRecords.get(0);
			String hitString = numHitsElement.getValue();
			if ( !hitString.equals("") )
				hits=Integer.parseInt(hitString);
		}
		return hits;
	}

	public static Document explainDoc( ) {
		Element explainNode = new Element("explain");
		Element subNode = new Element("artifactId");
		Text txt = new Text( myName );
		subNode.appendChild( txt );
		explainNode.appendChild( subNode);
		subNode = new Element("version");
		txt = new Text( myVersion );
		subNode.appendChild( txt );
		explainNode.appendChild( subNode);
		subNode = new Element("hostname");
		txt = new Text( myHostName );
		subNode.appendChild( txt );
		explainNode.appendChild( subNode);
		subNode = new Element("configFile");
		txt = new Text( myConfigName );
		subNode.appendChild( txt );
		explainNode.appendChild( subNode);
		return new Document( explainNode );
	}

	public static Document errorDoc( String mess, String addI ) {
		Diagnostic diag = new Diagnostic( 0, mess, addI );
		Element diagNode = diag.toElement();
		return new Document( diagNode );
	}

	public static Document errorDoc( String mess ) {
		return errorDoc( mess, "" );
	}

	static final long serialVersionUID=1;
}
