package nl.uvt.locator;

import java.io.*;
import java.net.*;
import java.math.*;
import java.security.*;
import java.util.*;
import java.util.regex.*;

import nu.xom.*;
import nu.xom.xslt.XSLException;
import nu.xom.xslt.XSLTransform;
import nl.uvt.commons.io.*;

public class LinkDbInfo extends LocatorInfo {

	List<Threshold> thresholds;
	String methodName;
	
	LinkDbInfo(){
		thresholds = new LinkedList<Threshold>();
		methodName = "unknown";
	}

	public Node toRecordNode(){
		Element record = new Element("record");
		record.appendChild( makeNode( "linkIssn", linkIssn ) );
		record.appendChild( makeNode( "linkTitle", linkTitle ) );
		record.appendChild( makeNode( "linkUrl", linkUrl) );
		record.appendChild( makeNode( "linkMethod", linkMethod ) );
		record.appendChild( makeNode( "methodName", methodName ) );
		record.appendChild( makeNode( "rank", Integer.toString(rank) ) );
		record.appendChild( makeNode( "generatedUrl", generatedUrl) );
		record.appendChild( makeNode( "publisher", linkPublisher ) );
		return record;
	}

	public boolean inRange( QueryData pQ) {
		try{
			return Threshold.inRange( thresholds, pQ );
		}
		catch (LocatorException e ){
			logger.debug( "threshold failed: "+  e.getMessage() );
			return false;
		}
	}

	boolean fill( Node rec ){
		boolean result = true;
		String res = getValue( rec, "parameters/issn");
		if ( res.length() == 0 ){
			res = getValue( rec, "issn");		
		}
		if ( res.length() > 0 ){
			Pattern issnPat = Pattern.compile("([\\dX]*)");
			Matcher issnMatcher = issnPat.matcher(res);
			if ( issnMatcher.find() ){
				linkIssn = issnMatcher.group();
//				logger.debug( "matched!"+issn);
			}
		}
		res = getValue( rec, "title" );
		if ( res.length() > 0 )
			linkTitle = res;
		res = getValue( rec, "url");
		if ( res.length() > 0 )
			linkUrl = res;
		res = getValue( rec, "template" );
		if ( res.length() > 0 )
			linkMethod = res;
		res = getValue( rec, "publisher");
		if ( res.length() > 0 )
			linkPublisher = res;
		Nodes nodes = rec.query( "threshold ");
		for ( int i=0; i < nodes.size(); ++i ){
			Threshold th = new Threshold();
			if ( !th.fill( nodes.get(i) ) )
				result = false;
			thresholds.add( th );
		}
		return result;
	}

	protected String getValue( Node node, String val ){
		Nodes nodes = node.query( val );
		if ( nodes.size() > 0 ){
			return nodes.get(0).getValue();
		}
		else
			return "";
	}

	public String toString(){
		String result = "LinkDbInfo:";
		result += " issn='" + linkIssn;
		result += "' title='" + linkTitle;
		result += "' url='" + linkUrl;
		result += "' template='" + linkMethod;
		result += "' publisher='" + linkPublisher;
		result += "' thresholds=";
		for ( int i=0; i < thresholds.size(); ++i ){
			result += thresholds.get(i).toString();
		}
		return result;
	}

	private class linkDbProperties{
		String methodName;
		boolean linkDbOnly;
		boolean supportsDOI;
		boolean DOIhasFallback;
		int defaultRank;
		linkDbProperties( String name, boolean lo, boolean sd, boolean fb, int r ){
			methodName = name;
			linkDbOnly = lo;
			supportsDOI = sd;
			DOIhasFallback = fb;
			defaultRank = r;
		}
	}

	static Map<String, linkDbProperties> properties = null;
	static XSLTransform transformer = null;

	static long oldPropDate = 0;
	static long oldSheetDate = 0;

	private void fillProperties() {
		String filename = Locator.locatorConfig.getProperty("linkDbProperties");
		File file = Locator.getFile( filename );
		if ( file != null && file.exists() ){
			long moDate = file.lastModified();
//			logger.debug("file " + filename + " old date was: " + oldPropDate + " moDate=" + moDate );
			if ( moDate > oldPropDate ){
				logger.debug("(re)read file:" + filename );
				try {
					properties = new HashMap<String,linkDbProperties>();
					FileInputStream is = new FileInputStream(file);
					BufferedReader in = new BufferedReader( new InputStreamReader(is));
					while ( in.ready() ){
						String line = in.readLine();
						if ( !line.equals("") && !line.startsWith("#") ){
							String [] fields = line.split(",");
							if (fields.length == 4 ){
								try {
									for ( int i=0; i < 4; i++ )
										fields[i] = fields[i].trim();
									String name = fields[1];
									String type = fields[2];
									boolean lo = true;
									boolean sd = false;
									boolean fb = false;
									if ( type.equals("DOI") ){
										lo = false;
										sd = true;
									}
									else if ( type.equals("DOI_FALLBACK") ){
										lo = false;
										sd = true;
										fb = true;
									}
									else if ( type.equals("LINK_ONLY") ){
										lo = true;
										sd = false;
									}
									else if ( type.equals("DEFAULT") ){
										lo = false;
										sd = false;
										fb = false;
									}
									else {
											logger.warn("Invalid value for linkType: " + type );
											continue; //while
									}
									int r = Integer.parseInt( fields[3] );
/*									logger.debug( "add propperties for method=" + fields[0] + " (" + fields[1] + ")" );
									logger.debug( "   linkOnly = " + Boolean.toString(lo));
									logger.debug( "   is DOI   = " + Boolean.toString(sd));
									logger.debug( "   fallBack = " + Boolean.toString(fb));
									logger.debug( "   defaultrank = " + r);
									
*/									
									linkDbProperties prop = properties.get( fields[0] );
									if ( prop != null ){
										logger.warn("Multiple entry for method: "+ fields[0] + " in linkDb properties file! (Ignored)" );
									}
									else {
										prop = new linkDbProperties( name, lo, sd, fb, r );
										properties.put( fields[0], prop );
									}
								}
								catch ( Exception e){
									logger.warn("format problem in '" + filename + "'\nline='" + line + "'\nSKIPPED!" );
								}
							}	
						}
						oldPropDate = moDate;
					}
				} catch (Exception e) {
					logger.equals("problems reading:" + filename );
				}
			}
		}
		else {
			logger.debug("unable to find linkDb properties file");
			properties = null;
		}
	}

	private void loadTransformer(){
		String sheetName = Locator.locatorConfig.getProperty("linkDbTransfomer");
		File file = Locator.getFile( sheetName );
		if ( file != null && file.exists() ){
			long moDate = file.lastModified();
//			logger.debug("file " + sheetName + " old date was: " + oldSheetDate + " moDate=" + moDate );
			if ( moDate > oldSheetDate ){
				logger.debug("(re)read file:" + sheetName );
				Builder builder = new Builder(false);
/*
 we would like to validate the xslt script against a schema. But I can't find one yet
 Does such a beast is exist?
				SchemaFactory factory = null;
				try {
					factory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
				}
				catch( Exception e ){
					logger.warn("dammit: " + e.getMessage());
				}
				Validator validator = null;
				Schema schema = null;
				try{ 
					schema = factory.newSchema( new File("/etc/locator1/xslt.xsd" ));
				}
				catch (Exception e ){
					logger.warn("PROBLEMS! with schema " + e.getMessage() );
				}
				if ( schema != null ){
					try{ 
						validator = schema.newValidator();
						validator.validate(new StreamSource(file));
					}
					catch (Exception e ){
						logger.warn("more PROBLEMS! " + e.getMessage() );
					}
				}
				Builder builder = new Builder(true);
*/				try {
					Document stylesheet = builder.build(file);
					transformer = new XSLTransform(stylesheet);
					oldSheetDate = moDate;
					logger.debug( "created transformer: " + sheetName );
				}
				catch ( IOException e ){
					logger.warn( "creating transformer failed: "+ sheetName );
					logger.warn("reason: " + e.getMessage() );
					if ( transformer != null )
						logger.warn("keeping old version");
				}
				catch ( XSLException e ){
					logger.warn( "creating transformer failed: "+ sheetName + " XSLT problem");
					logger.warn("reason: " + e.getMessage() );
					if ( transformer != null )
						logger.warn("keeping old version");
				}

				catch ( ValidityException e ){
					logger.warn( "creating transformer failed: "+ sheetName + " Invalid XSLT" );
					logger.warn("reason: " + e.getMessage() );
					if ( transformer != null )
						logger.warn("keeping old version");
				}
				catch ( ParsingException e ){
					logger.warn( "creating transformer failed: "+ sheetName + " parsing failed");
					logger.warn("reason: " + e.getMessage() );
					if ( transformer != null )
						logger.warn("keeping old version");
				}
			}
		}
	}
	
	Document toTempDoc( QueryData qd){
		Element root = new Element("record");
		Document result = new Document( root );
		root.appendChild( makeNode( "linkIssn", linkIssn ) );
		root.appendChild( makeNode( "linkTitle", linkTitle ) );
		root.appendChild( makeNode( "linkUrl", linkUrl) );
		root.appendChild( makeNode( "linkMethod", linkMethod ) );
		root.appendChild( makeNode( "issn", qd.issn ) );
		root.appendChild( makeNode( "genre", qd.genre ) );
		try {
			root.appendChild( makeNode( "atitle", URLEncoder.encode( qd.atitle, "ISO-8859-1" )) );
			root.appendChild( makeNode( "title", URLEncoder.encode( qd.title , "ISO-8859-1" )) );
		} catch (UnsupportedEncodingException e) {
			logger.fatal( "encoding title failed", e );
		}
		if ( qd.date > 0 )
			root.appendChild( makeNode( "year", Integer.toString(qd.date) ) );
		root.appendChild( makeNode( "volume", qd.volume ) );
		root.appendChild( makeNode( "issue", qd.issue) );
		if ( qd.thIssue > 0 )
			root.appendChild( makeNode( "thIssue", Integer.toString(qd.thIssue)) );
		if ( qd.thVolume > 0 )
			root.appendChild( makeNode( "thVolume", Integer.toString(qd.thVolume)) );
		root.appendChild( makeNode( "spage", qd.spage ) );
		root.appendChild( makeNode( "rank", Integer.toString(rank) ) );
		return result;
	}

	public boolean applyMethod( QueryData pQ ){
		if ( !inRange( pQ ) )
			return false;
		/* 
 	RANKINGS:
	 	1.  pdf/html
	 	2.  artikelpagina (abstract o.i.d. + link naar full text)
		3.  issuepagina (bevat overzicht artikelen)
		4.  volumepagina (bevat overzicht issues)
		5.  tijdschriftpagina (bevat overzicht volumes)
		6.  site aanbieder (bevat overzicht tijdschriften of zoekbox)
		 */
		if ( linkMethod.equals("OLC") ) {
			// OLC is not handled by us!
			// should link to the docserver
			logger.warn( "method OLC is not handled by the Locator, docserver link needed");
		}
		else {
			fillProperties();
			linkDbProperties prop = null;
			if ( properties != null )
				prop = properties.get( linkMethod );
			boolean done = false;
			if ( prop != null ){
				logger.debug("found linkDb properties for method " +linkMethod );
				methodName = prop.methodName;
				if ( prop.linkDbOnly ){
					rank = prop.defaultRank;
					generatedUrl = linkUrl;
					done = true;
				}
				else if ( prop.supportsDOI ){
					logger.debug("it's a well-known DOI method");
					if ( !pQ.doi.equals("") ){
						logger.debug("A DOI was provided in the Query!: " + pQ.doi );
						rank = 1;
						try {
							generatedUrl = 	"http://dx.doi.org/" + URLEncoder.encode(pQ.doi, "ISO-8859-1" );
						} catch (UnsupportedEncodingException e) {
							logger.fatal("encoding doi failed");
						}
						done = true;
					}
					else {
						String myUrl = generateDOI( pQ );
						if ( !myUrl.equals("") ){
							rank = 1;
							generatedUrl = myUrl;
							done = true;
						}
						else {
							if ( prop.DOIhasFallback ){
								logger.debug("DOI failed, but try fallback.");
							}
							else {
								generatedUrl = linkUrl;
								rank = prop.defaultRank;
								done = true;
							}
						}
					}
				}
			}
			else {
				methodName = "unknown linkMethod: "+linkMethod;
				logger.debug( "didn't find properties for method " +linkMethod );
				logger.debug( "try a fallback anyway" );
			}
			if ( !done ) {
				if ( linkMethod.equals("SD")){
					String myUrl = "";
					if ( !pQ.volume.equals("") && !pQ.spage.equals("") ){
						logger.debug( "create md5 hashed SD link");
						String SDorigin="UTILBURG";
						String SDsalt = "(CxZ:=Jrxs;hXYWS!KVpdj!jgeO7NSdJ";
						String volKey = linkIssn;
						volKey += "#" + pQ.volume;
						volKey += "#" + pQ.spage;
						if ( pQ.thIssue >= 0 )
							volKey += "#" + pQ.thIssue;
						String md5source = "_ob=GatewayURL&_origin=" + SDorigin 
						+ "&_method=citationSearch&_volkey=" + volKey
						+ "&_version=1" + SDsalt;
//						logger.debug("md5:'" + md5source + "'" );
						MessageDigest md5;
						String md5Hash = "";
						try {
							md5 = MessageDigest.getInstance("MD5");
							md5.update(md5source.getBytes(), 0, md5source.length());
							byte mdBytes[] = md5.digest();
							md5Hash = new BigInteger(1,mdBytes).toString(16);
							if ( md5Hash.length() < 32 )
								md5Hash = '0' + md5Hash; 
//							logger.debug("md5 hash:'" + md5Hash + "'" );
						} catch (NoSuchAlgorithmException e) {
							logger.fatal("MD5 hasher", e);
						}
						try {
							myUrl = "http://www.sciencedirect.com/science?_ob=GatewayURL&_origin="
								+ SDorigin + "&_method=citationSearch&_volkey="
								+ URLEncoder.encode(volKey, "ISO-8859-1" ) + "&_version=1&md5=" + md5Hash;
							rank = 1;
						} catch (UnsupportedEncodingException e) {
							logger.fatal( "encoding failed", e );
						}
					}
					if ( !myUrl.equals("") )
						generatedUrl = myUrl;
					else {
						generatedUrl = linkUrl;
						rank = 5;
					}
						
				}
				else {
					// leave it to the xslt
					if ( prop != null )
						rank = prop.defaultRank;
					else
						rank = 6; // fear the worst
					logger.debug("Method= " + linkMethod );
					Document doc = toTempDoc( pQ );
					loadTransformer();
					if ( transformer != null ){
						logger.debug( "start transform" );
						logger.debug( "input doc:" + doc.toXML());
						Document resultDoc = null;
						try {
							Nodes output = transformer.transform( doc );
							resultDoc = XSLTransform.toDocument(output);
							logger.debug( "transformed doc:" + resultDoc.toXML());
						}
						catch ( Exception e){
							logger.warn( "transforming failed:" + e.getMessage() );
						}
						if ( resultDoc != null ){
							Nodes nodes = resultDoc.query("//generatedUrl" );
							if ( nodes.size() > 0 ){
								generatedUrl = nodes.get(0).getValue().toString();
								nodes = resultDoc.query("//rank" );
								if ( nodes.size() > 0 ){
									rank = Integer.parseInt( nodes.get(0).getValue().toString().trim() );
								}
							}
						}
					}
					else {
						logger.warn("unable te find a transformer script");
					}
				}
			}
		}
		logger.debug("generated URL: "+ generatedUrl );
		return generatedUrl != "";
	}

	private String generateDOI( QueryData pQ ){
		String crossrefUri = Locator.locatorConfig.getProperty("crossrefUri");
		try {
			URL url = new URL( crossrefUri );
			Properties prop = new Properties();
			String qData = "";
			qData += linkIssn + "|";
			qData += pQ.atitle +"||";
			qData += pQ.thVolume + "|";
			qData += pQ.issue + "|";
			qData += pQ.spage + "|";
			qData += pQ.date + "|||";
			prop.setProperty("qdata", qData );
			prop.setProperty( "pwd", "crpw021");
			prop.setProperty( "usr", "tuli");
			InputStream result = Http.HttpGetInputStream( url, prop );
			logger.debug( "call crossref: " + url );
			BufferedReader in = new BufferedReader( new InputStreamReader(result) );
			logger.debug( "crossref returns:");
			String doi = "";
			while ( in.ready() ){
				String line = in.readLine();
				logger.debug( line );
				if ( line.contains("|") ){
					String [] arr = line.split( "\\|" );
					int num = arr.length;
					logger.debug("split into "+ num );
					if ( num > 8 && arr[num-1] != "" )
						doi = "http://dx.doi.org/" +  URLEncoder.encode(arr[num-1], "ISO-8859-1" );
				}
			}
			if ( doi.equals("") && !pQ.aulast.equals("") && !pQ.atitle.equals("")){
				// try fall back to Title/Author search
				logger.debug( "DOI failed, try alternative");
				doi = generateTADOI( pQ );
			}
			return doi;
		}
		catch (Exception e) 
		{
			logger.debug( "crossref failed: " +  e.getMessage() );
			return "";
		}
	}
	
	private String generateTADOI( QueryData pQ ){
		String crossrefUri = Locator.locatorConfig.getProperty("crossrefUri");
		try {
			URL url = new URL( crossrefUri );
			Properties prop = new Properties();
			String qData = "";
			qData += pQ.atitle +"|";
			qData += pQ.aulast + "||key|";
			prop.setProperty("qdata", qData );
			prop.setProperty("type", "a" );
			prop.setProperty( "pwd", "crpw021");
			prop.setProperty( "usr", "tuli");
			logger.debug( "call crossref: " + url );
			InputStream result = Http.HttpGetInputStream( url, prop );
			BufferedReader in = new BufferedReader( new InputStreamReader(result) );
			logger.debug( "crossref returns:");
			String doi = "";
			while ( in.ready() ){
				String line = in.readLine();
				logger.debug( line );
				if ( line.contains("|") ){
					String [] arr = line.split( "\\|" );
					int num = arr.length;
					logger.debug("split into "+ num );
					if ( num > 9 && arr[num-1] != "" )
						doi = "http://dx.doi.org/" +  URLEncoder.encode(arr[num-1], "ISO-8859-1" );
				}
			}
			return doi;
		}
		catch (Exception e) 
		{
			logger.debug( "crossref failed: " + e.getMessage() );
			return "";
		}
	}

}
