package nl.uvt.locator;

import java.io.File;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import nu.xom.*;
import nu.xom.xslt.XSLTransform;

public class HoldingInfo extends LocatorInfo {
	String epn;
	String status;
	String callNumber;
	String url;
	String material;
	String collection;
	String action;
	List<ThRange> ranges; 
	Document availabilityDoc;

	private class ThRange {
		List<Threshold> rangeItems;
		ThRange(){
			rangeItems = new LinkedList<Threshold>();
		}

		public String toString(){
			String s = "[";
			for( int i=0; i<rangeItems.size(); ++i ){
				s += rangeItems.get(i).toString();
				if ( i < rangeItems.size()-1 )
					s += ",";
			}
			s += "]";
			return s;
		}
	}

	HoldingInfo(){
		epn = "";
		status = "";
		callNumber =  "";
		url = "";
		material = "";
		action = "";
		collection = "";
		ranges = new LinkedList<ThRange>();
		availabilityDoc = null;
	}

	static XPathContext modsContext = Locator.sruContext;

	public static final String modsNamespace = "http://www.loc.gov/mods/v3";
	public static final String schemaNamespace = "http://www.w3.org/2001/XMLSchema-instance";
	public static final String extNamespace = "http://drcwww.uvt.nl/~place/search%20engine/extension";


	static public List<LocatorInfo> getModsResults( Document doc, QueryData pQ, String sheetName ){
		List<LocatorInfo> meResult = null;
		Builder builder = new Builder();
		try {
			HoldingInfo.modsContext.addNamespace("mods", HoldingInfo.modsNamespace);
			HoldingInfo.modsContext.addNamespace("xsi", HoldingInfo.schemaNamespace);
			HoldingInfo.modsContext.addNamespace("a", HoldingInfo.extNamespace);
			Locator.logger.debug( "use xslt sheet:" + sheetName );
			File f = Locator.getFile( sheetName );
			if ( f != null ){
				nu.xom.Document stylesheet = builder.build(f);
				XSLTransform transform = new XSLTransform(stylesheet);
				//				Locator.logger.debug( "input doc:" + doc.toXML());
				Nodes output = transform.transform( doc );
				nu.xom.Document standardDoc = XSLTransform.toDocument(output);
				//				Locator.logger.debug( "transformed doc:" + standardDoc.toXML());
				Nodes records = standardDoc.query("//mods:mods", HoldingInfo.modsContext);
				if ( records.size() > 0 ) {
					meResult = new LinkedList<LocatorInfo>();
					Locator.logger.debug("found " + records.size() + " real records");
					for ( int i=0; i < records.size(); ++i ){
						// every record may contain several occurrences
						// split these in separate records
						Locator.logger.debug(" try record: " + (i+1)  );
						HoldingInfo.fillMultipleRecord( records.get(i), meResult );
					}
				}
			}
			else {
				Locator.logger.fatal("unable to get a stylesheet:" + sheetName );
			}
		}
		catch( Exception e )
		{
			Locator.logger.error( "mods exception", e );
		}
		return meResult;
	}

	public static void fillMultipleRecord(Node inputRecord, List<LocatorInfo> meResult) {
		Locator.logger.debug( "split one record per occurrence");
//		Locator.logger.debug( "input record: " + inputRecord.toXML() );
		HoldingInfo data = new HoldingInfo();
		data.linkMethod = "holding";
		Nodes nodes= inputRecord.query("./mods:recordInfo/mods:recordIdentifier", HoldingInfo.modsContext );
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				data.linkIssn = nodes.get(0).getChild(0).getValue().toString();
		}
		nodes = inputRecord.query("./mods:titleInfo/mods:title", HoldingInfo.modsContext);
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				data.linkTitle = nodes.get(0).getChild(0).getValue().toString();
		}
		nodes = inputRecord.query("./mods:originInfo/mods:publisher", HoldingInfo.modsContext );
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				data.linkPublisher = nodes.get(0).getChild(0).getValue().toString();
		}
		nodes = inputRecord.query("./mods:physicalDescription/mods:form", HoldingInfo.modsContext );
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) {
				data.material = nodes.get(0).getChild(0).getValue().toString();
				Locator.logger.debug( "found material=" + data.material );
			}
		}
		nodes = inputRecord.query("./mods:location", HoldingInfo.modsContext )	;
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) {
				//
				// occurrence can be found as a note in holdingSimple or as a tag in holdingExternal
				// there is no guarantee that occurrences are present in both, are continuous or whatever
				// therefore, we just collect all found 'occurrences' in a String set and try them all
				//
				Nodes occNodes = nodes.get(0).query("./mods:holdingSimple/mods:copyInformation/mods:note[@type='occurrence']", HoldingInfo.modsContext );
				// search the simpelHolding
				SortedSet<String> occs = new TreeSet<String>();
				if ( occNodes.size() > 0){
					Locator.logger.debug("there are " + occNodes.size() + " occurence nodes in Simple" );
					for ( int i=0; i < occNodes.size(); ++i ){
						//						Locator.logger.debug("adding occ = " + occNodes.get(i).getValue() );
						occs.add(occNodes.get(i).getValue());
					}
				}
				occNodes = nodes.get(0).query("./mods:holdingExternal/a:copies/a:copy/a:occurrence", HoldingInfo.modsContext );
				// search the external Holding
				boolean haveSomeExt = false;
				if ( occNodes.size() > 0){
					haveSomeExt = true;
					Locator.logger.debug("there are " + occNodes.size() + " occurence nodes in External" );
					for ( int i=0; i < occNodes.size(); ++i ){
						//						Locator.logger.debug("adding occ = " + occNodes.get(i).getValue() );
						occs.add(occNodes.get(i).getValue());
					}
				}

				// now loop all occurences. First try External. then Simple.
				Iterator<String> occIt = occs.iterator();
				while ( occIt.hasNext() ){
					String occS = occIt.next();
					Locator.logger.debug("handling occ="+ occS );
					Nodes extHolding = nodes.get(0).query("./mods:holdingExternal/a:copies/a:copy[a:occurrence='"+ occS +"']", HoldingInfo.modsContext);
					if ( extHolding.size() == 0 ){
						Locator.logger.debug( "found no externalHolding for occ=" + occS );
						Nodes simpleHolding = nodes.get(0).query("./mods:holdingSimple/mods:copyInformation[mods:note[@type='occurrence' and .='"+occS+"']]", HoldingInfo.modsContext);
						if ( simpleHolding.size() == 0 )
							Locator.logger.debug("Also found NO simpleHolding!");
						else if( haveSomeExt )
							Locator.logger.debug("We ignore simpleHoldings, because there are also externalHoldings for other occurrences.");
						else {
							Node holding = simpleHolding.get(0);
							Locator.logger.debug( "but found simpleHolding for occ=" + occS );
							//							Locator.logger.debug( holding.toXML());							
							HoldingInfo rec = new HoldingInfo();
							rec.action = "";
							rec.linkIssn = data.linkIssn;
							rec.linkTitle = data.linkTitle;
							rec.linkPublisher = data.linkPublisher;
							rec.material = data.material;
							Nodes notes = holding.query("./mods:note[@type='epn']", HoldingInfo.modsContext);
							if( notes.size() > 0 ){
								String epn = notes.get(0).getValue().toString();
								rec.epn = epn;
								Locator.logger.debug("got epn:" + epn );
							}
							notes = holding.query("./mods:note[@type='status']", HoldingInfo.modsContext);
							if( notes.size() > 0 ){
								String status = notes.get(0).getValue().toString();
								rec.status = status;
								Locator.logger.debug("got status: " + status );
							}
							notes = holding.query("./mods:shelfLocator", HoldingInfo.modsContext );
							if( notes.size() > 0 ){
								String call = notes.get(0).getValue().toString();
								rec.callNumber = call;
							}
							notes = holding.query("./mods:electronicLocator", HoldingInfo.modsContext );
							if( notes.size() > 0 ){
								String url = notes.get(0).getValue().toString();
								rec.url = url;
							}
							if ( rec.collection.equals("") )
								rec.collection = guessCollection( rec.callNumber );
							Locator.logger.debug( "rec.collection=" + rec.collection );
							Locator.logger.debug( "rec.callNumber=" + rec.callNumber );
							meResult.add( rec );
						}
					}
					else {
						Node holding = extHolding.get(0);
						Locator.logger.debug( "found externalHolding for occ=" + occS );
//						Locator.logger.debug( holding.toXML());
						HoldingInfo rec = new HoldingInfo();
						rec.action = "";
						rec.linkIssn = data.linkIssn;
						rec.linkTitle = data.linkTitle;
						rec.linkPublisher = data.linkPublisher;
						rec.material = data.material;
						Nodes info = holding.query("./a:epn", HoldingInfo.modsContext );
						if ( info.size() > 0 ){
							if ( info.get(0).getChildCount() > 0 ) 
								rec.epn = info.get(0).getChild(0).getValue().toString();
						}
						info = holding.query("./a:collection", HoldingInfo.modsContext);
						if ( info.size() > 0 ){
							if ( info.get(0).getChildCount() > 0 ) 
								rec.collection = info.get(0).getChild(0).getValue().toString();
						}
						info = holding.query("./mods:enumerationAndChronology", HoldingInfo.modsContext);
						if ( info.size() > 0 ){
							String coll = "";
							if ( info.get(0).getChildCount() > 0 ) 
								coll = info.get(0).getChild(0).getValue().toString();
							rec.collection = guessCollection(coll);
						}
						info = holding.query("./a:status", HoldingInfo.modsContext);
						if ( info.size() > 0 ){
							if ( info.get(0).getChildCount() > 0 ) 
								rec.status = info.get(0).getChild(0).getValue().toString();
						}
						info = holding.query("./mods:shelfLocator", HoldingInfo.modsContext);
						if ( info.size() > 0 ){
							if ( info.get(0).getChildCount() > 0 ) {
								String call = info.get(0).getChild(0).getValue().toString();
								rec.callNumber = call;
							}
						}
						info = holding.query("./mods:electronicLocator", HoldingInfo.modsContext );
						if( info.size() > 0 ){
							String url = info.get(0).getValue().toString();
							rec.url = url;
						}
						info = holding.query( "./a:range", HoldingInfo.modsContext);
						for ( int j=0; j < info.size(); ++j ){
							Locator.logger.debug("get range:"+j);
							ThRange th = data.new ThRange();
							Nodes thNodes = info.get(j).query( "./a:threshold", HoldingInfo.modsContext);
							for ( int k=0; k < thNodes.size(); ++k ){
								Threshold thSub = new Threshold();
								thSub.fill( thNodes.get(k), "a", HoldingInfo.modsContext );
								th.rangeItems.add( thSub );
							}
							//							Locator.logger.debug("read:" + th.toString());
							rec.ranges.add( th );
						}
						if ( rec.collection.equals("") )
							rec.collection = guessCollection( rec.callNumber );
						Locator.logger.debug( "rec.callNumber=" + rec.callNumber );
						Locator.logger.debug( "rec.collection=" + rec.collection );
						meResult.add( rec );
					}
				}
			}
		}
		else {
			Locator.logger.debug( "Did not find usefull Holding Info in input record: " + inputRecord.toXML() );	
		}
	}


	public static String guessCollection( String call ){
		//
		// try to guess the collection from the call
		// rather naive: handles calls like: 'CBM TF B 55657', 'JUR X45C BIER 2009' '+ Reg. jrg. 1-33 (1949-1981) (BRA)'
		// I wonder about: 'CBM LET P SELL 2000' (first hit is taken...)
		Locator.logger.debug("guess from: " + call );
		String[] parts = call.split("[ \\[\\(\\)\\]]");
		for ( int i=0; i < parts.length; ++i ){
			String coll = parts[i];
			if ( isKnownCollection( coll ) )
				return coll;
			else if ( isKnownCollection( parts[i] ) )
				return parts[i];
		}	
		return "";	
	}	

	public static boolean isKnownCollection( String collection ) {
		if ( Locator.knownCollections == null ){
			Locator.knownCollections = new HashSet<String>();
			String colls = Locator.locatorConfig.getProperty("knownCollections","");
			if ( colls.equals("") ){
				// hardwired default
				Locator.logger.warn("PLEASE FIX THIS: NO knownCollections found in configfile." );
				colls = "CBM, CIF, BRA, DISC, ECO, GES, JUR, LET, SOC, TFL";
				Locator.logger.warn("PLEASE FIX THIS: we use hardcoded default: " + colls );
			}
			String [] doiS = colls.split(",");
			for ( int i=0; i < doiS.length; ++i ){
				Locator.knownCollections.add(doiS[i].trim());
			}
			Locator.logger.debug("filled Known collections list: " + Locator.knownCollections.toString() );
		}
		return Locator.knownCollections.contains( collection );
	}

	void fill( Node rec ){
		logger.debug( "fill from record:" + rec.toXML() );
		linkMethod = "holding";
		Nodes nodes= rec.query("./mods:recordInfo/mods:recordIdentifier", modsContext);
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				linkIssn = nodes.get(0).getChild(0).getValue().toString();
		}
		nodes = rec.query("./mods:titleInfo/mods:title", modsContext);
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				linkTitle = nodes.get(0).getChild(0).getValue().toString();
		}
		nodes = rec.query("./mods:location/mods:url", modsContext);
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				linkUrl = nodes.get(0).getChild(0).getValue().toString();
		}
		nodes = rec.query("./mods:originInfo/mods:publisher", modsContext);
		if ( nodes.size() > 0 ){
			if ( nodes.get(0).getChildCount() > 0 ) 
				linkPublisher = nodes.get(0).getChild(0).getValue().toString();
		}
	}		

	public Node toRecordNode(){
		Element record = new Element( "record" );
		record.appendChild( makeNode( "linkIssn", linkIssn ) );
		record.appendChild( makeNode( "linkTitle", linkTitle ) );
		record.appendChild( makeNode( "linkUrl", linkUrl ) );
		record.appendChild( makeNode( "generatedUrl", generatedUrl ) );
		record.appendChild( makeNode( "linkMethod", linkMethod ) );
		record.appendChild( makeNode( "action", action ) );
		record.appendChild( makeNode( "collection", collection ) );
		record.appendChild( makeNode( "epn", epn ) );
		record.appendChild( makeNode( "callNumber", callNumber ) );
		record.appendChild( makeNode( "material", material ) );
		record.appendChild( makeNode( "status", status ) );
		Element av = new Element( "availability" );
		if ( availabilityDoc != null )
			av.appendChild( availabilityDoc.getRootElement().copy() );
		record.appendChild( av );
		return record;
	}

	public boolean inRange( QueryData pQ) {
		if ( pQ.genre.equals("journal"))
			return true;
		try {
			if ( ranges.size() == 0 ) {
				if ( pQ.genre.equals("article") ){
					logger.debug( "article requires range, but no range here...");
					return false;
				}
				else
					return true;
			}
			for ( int i=0; i < ranges.size(); ++i ){
				if ( Threshold.inRange( ranges.get(i).rangeItems, pQ ) )
					return true;
			}
			return false;
		} catch (LocatorException e) {
			logger.debug("too bad");
			return false;
		}
	}

	public boolean applyMethod( QueryData pQ ){
		logger.debug("handle holding Information for call: " + callNumber );
		if ( material.equals("electronic")){
			logger.info("HOLDING Lookup for Electronic medium!");
			logger.info("query was:" + pQ.toString() );
			logger.info("PLEASE FIX in the LinkDb");
		}
		else if ( !inRange( pQ ) ){
			logger.debug( "rejected because of threshold");
			return false;
		}
		logger.debug( "collection=" + collection );
		logger.debug( "epn=" + epn );
		logger.debug( "status=" + status );
		logger.debug( "callNumber=" + callNumber );
		logger.debug( "url=" + url );
		if ( !linkUrl.equals("") ){
			action = "url";
			generatedUrl = linkUrl;
//			return true;
		}
		else if ( !url.equals("") ){
			action = "url";
			generatedUrl = url;
//			return true;
		}
		if ( !epn.equals("") ) {
			if ( Locator.doAvailability ){
				AvailabilityConnection av = new AvailabilityConnection();
				try {
					availabilityDoc = av.query(epn);
					if ( availabilityDoc != null )
						logger.debug("availability server returned " + availabilityDoc.toXML());
				}
				catch( LocatorException e ){
					logger.warn("Availibility connection failed!");
					// but we just continue...
				}
			}
			else {
				logger.debug( "doAvail=false!???");
			}
		}
		if ( !action.equals("url")){
			if ( collection.equals("") ){
				action = "unknown";
			}
			else if ( isDepot( collection ) ){
				action = "depot";
			}
			else {
				action = "stack";
			}
		}
		logger.debug("accepted: action=" + action );
		return true;
	}

	public static boolean isDepot(String collection ) {
		if ( Locator.knownDepots == null ){
			Locator.knownDepots = new HashSet<String>();
			String colls = Locator.locatorConfig.getProperty("knownDepots","");
			if ( colls.equals("") ){
				// hardwired default
				Locator.logger.warn("PLEASE FIX THIS: NO knownDepots found in configfile." );
				colls = "CBM, KOD, TRE";
				Locator.logger.warn("PLEASE FIX THIS: we use hardcoded default: " + colls );
			}
			String [] doiS = colls.split(",");
			for ( int i=0; i < doiS.length; ++i ){
				Locator.knownDepots.add(doiS[i].trim());
			}
			Locator.logger.debug("filled Known depots list: " + Locator.knownDepots.toString() );
		}
		return Locator.knownDepots.contains( collection );
	}


}
