# $Id$
# $URL$

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::NOC::Util::NetworkResourceUtil;

use UvT::NOC::Util::StringUtil;
use UvT::NOC::Util::FileUtil;

use UvT::NOC::InfraStructure::HostList;
use UvT::NOC::InfraStructure::SwitchList;
use UvT::NOC::Tracking::UserTrackingData;
use UvT::NOC::Tracking::Network;
use UvT::NOC::Tracking::Vlan;
use UvT::NOC::Tracking::MACAddress;
use UvT::NOC::Tracking::TrackingData;
use UvT::NOC::MacRegistration::UvTRegList;
use UvT::NOC::MacRegistration::WMBRegList;

use XML::LibXML;
use Data::Dumper;

use Exporter qw(import);

our @EXPORT = qw(readHostsFile readInfra readUserTrackingData readIntreg readWMBReg);
our @EXPORT_OK = @EXPORT;

# Functienaam : readHostsFile
#
# Parameters : (1x)
#       1) de locatie van de hosts file
#
# Return : (1x)
#       1) een object van het type HostList :
#
#        bless( {
#                 'hosts' => {
#                              '172.20.20.7' => bless( {
#                                                        'ip' => '172.20.20.7',
#                                                        'name' => [
#                                                                    'ciscoprime',
#                                                                    'lms',
#                                                                    'watson'
#                                                                  ]
#                                                      }, 'UvT::NOC::InfraStructure::Host' ),
#                              '172.20.20.13' => bless( {
#                                                         'ip' => '172.20.20.13',
#                                                         'name' => [
#                                                                     'matahari'
#                                                                   ]
#                                                       }, 'UvT::NOC::InfraStructure::Host' )
#                            }
#               }, 'UvT::NOC::InfraStructure::HostList' );
#
#
# Omschrijving :
#       Functie om hosts file uit te lezen en de koppeling ip-adres op te slaan als
#       object Host in het object HostList welke wordt geretourneerd

sub readHostsFile
{
	my $filename = shift;

	my $data = stringFromFile($filename);
    
	my $hostList = new UvT::NOC::InfraStructure::HostList();

	foreach my $line (split('\n', $data))
    {
		# Enkel de regels behandelen die beginnen een geldig ip-adres
		# (a.b.c.d wordt in deze beschouwd als een geldig ip address)
		if ($line =~ /^\d+\.\d+\.\d+\.\d+/)
		{
			my($ip, $hostname) = split(/\s+/, trim($line), 2);
			$hostList->addHost(new UvT::NOC::InfraStructure::Host($ip, $hostname));
		}
	}

	return $hostList;
}


# Functienaam : readInfra
#
# Parameters : (1x)
#       1) de locatie van de infra file
#
# Return : (1x)
#       1) een data structure met de infra-gegevens :  
#
#        bless( {
#                 'switches' => {
#                                 'fd-g240b' => bless( {
#                                                        'ports' => {
#                                                                     'Gi1/0/3' => bless( {
#                                                                                           'fwdb' => '3',
#                                                                                           'name' => 'Gi1/0/3',
#                                                                                           'patch' => 'G 228.263',
#                                                                                           'description' => undef,
#                                                                                           'ifIndex' => '10103'
#                                                                                         }, 'UvT::NOC::InfraStructure::Port' ),
#                                                                     'Gi1/0/2' => bless( {
#                                                                                           'fwdb' => '2',
#                                                                                           'name' => 'Gi1/0/2',
#                                                                                           'patch' => 'G 228.261',
#                                                                                           'description' => undef,
#                                                                                           'ifIndex' => '10102'
#                                                                                         }, 'UvT::NOC::InfraStructure::Port' )
#                                                                   },
#                                                        'ip' => '172.20.20.98',
#                                                        'name' => 'fd-g240b'
#                                                      }, 'UvT::NOC::InfraStructure::Switch' ),
#                                 'bd-a60' => bless( {
#                                                      'ports' => {
#                                                                   'Gi1/0/1' => bless( {
#                                                                                         'fwdb' => '1',
#                                                                                         'name' => 'Gi1/0/1',
#                                                                                         'patch' => 'bd-a60-uplink-1',
#                                                                                         'description' => undef,
#                                                                                         'ifIndex' => '10101'
#                                                                                       }, 'UvT::NOC::InfraStructure::Port' ),
#                                                                   'Gi1/0/2' => bless( {
#                                                                                         'fwdb' => '2',
#                                                                                         'name' => 'Gi1/0/2',
#                                                                                         'patch' => 'fd-a60a',
#                                                                                         'description' => undef,
#                                                                                         'ifIndex' => '10102'
#                                                                                       }, 'UvT::NOC::InfraStructure::Port' )
#                                                                 },
#                                                      'ip' => '172.20.20.80',
#                                                      'name' => 'bd-a60'
#                                                    }, 'UvT::NOC::InfraStructure::Switch' )
#                               }
#               }, 'UvT::NOC::InfraStructure::SwitchList' );     
#          
#
# Omschrijving :
#       Functie om infra file uit te lezen en de gegevens op te slaan in een datastructure SwitchList welke wordt geretourneerd

sub readInfra
{
	my $filename = shift;
	my $hostList = shift;

	my $data = stringFromFile($filename);

	my $switchList = new UvT::NOC::InfraStructure::SwitchList();

	foreach my $line (split('\n', $data))
	{
		# De regel moet bestaan uit 8 velden gescheiden met een '|' (dus 7x '|')
		if ($line =~ /([^|]*\|){7}/)
		{
			# LET'S THINK ABOUT IT AGAIN ...
			my @data = split(/\|/, trim($line));
			my $device_port = trim($data[0]);						# switchpoort/telpoort (i.e. : 'fd-g240a-Fa2/0/36', 'tel-e157-60')
			my $ifIndex = trim($data[1]);   		$ifIndex =~ s/[^\d.]//g;
			my $fwdb = trim($data[2]);  			$fwdb =~ s/[^\d.]//g;
			my $aansluiting = trim($data[7]);		if ($aansluiting eq $device_port) { $aansluiting = undef; }	

			if ($device_port !~ /^tel/ && $device_port !~ /^td/)
			{
				my $device_name = substr($device_port, 0, rindex($device_port, "-"));
				my $device_interface = substr($device_port, rindex($device_port, "-") + 1, length($device_port));
				
				unless ($switchList->getSwitchByName($device_name))
				{
					my $ip = "Not found in hosts-file!";
					if ($hostList->getHostByName($device_name)) { $ip = $hostList->getHostByName($device_name)->ip; }
					else { warn ("IP lookup for '$device_name' failed!"); }
					$switchList->addSwitch(new UvT::NOC::InfraStructure::Switch($device_name, $ip));
				}
				
				# Poort toevoegen aan switch ...
				if ($switchList->getSwitchByName($device_name)->getPortByName($device_interface))
				{
					warn "Duplicate port $device_interface found on switch $device_name!";
				}
				else
				{
					my $newPort = new UvT::NOC::InfraStructure::Port($device_interface, undef, $ifIndex, $fwdb, $aansluiting);
					$switchList->getSwitchByName($device_name)->addPort($newPort);
				}
			}
		}
	}

	return $switchList;
}


# Functienaam : readUserTrackingData
#
# Parameters : (1x)
#       1) de locatie van de ciscoworks usertracking file
#
# Return : (1x)
#       1) een data structure met de usertracking-gegevens vanuit ciscoworks :  
#
#
#        bless( {
#                 'trackingServer' => 'WATSON',
#                 'network' => {
#                                'UvT' => bless( {
#                                                  'name' => 'UvT',
#                                                  'vlan' => {
#                                                              '95' => bless( {
#                                                                             'macAddress' => { 
#                                                                                               '00219b39e010' => bless( {
#                                                                                                                        'macAddress' => '00219b39e010',
#                                                                                                                        'trackingData' => {
#                                                                                                                                            '1365501703' => bless( {
#                                                                                                                                                                     'timestamp' => 1365501703,
#                                                                                                                                                                     'switchPort' => 'Gi1/0/1',
#                                                                                                                                                                     'switchIPAddress' => '172.20.20.98',
#                                                                                                                                                                     'hostIPAddress' => '137.56.126.100'
#                                                                                                                                                                   }, 'UvT::NOC::Tracking::TrackingData' ),
#                                                                                                                                            '1365595303' => bless( {
#                                                                                                                                                                     'timestamp' => 1365595303,
#                                                                                                                                                                     'switchPort' => 'Gi1/0/1',
#                                                                                                                                                                     'switchIPAddress' => '172.20.20.98',
#                                                                                                                                                                     'hostIPAddress' => '137.56.126.100'
#                                                                                                                                                                   }, 'UvT::NOC::Tracking::TrackingData' )
#                                                                                                                                          }
#                                                                                                                      }, 'UvT::NOC::Tracking::MACAddress' )
#                                                                                             },
#                                                                             'id' => '95'
#                                                                           }, 'UvT::NOC::Tracking::Vlan' ),
#                                                            }
#                                                }, 'UvT::NOC::Tracking::Network' ),
#                                'WMB' => bless( {
#                                                  'name' => 'WMB',
#                                                  'vlan' => {
#                                                              '3' => bless( {
#                                                                            'macAddress' => {
#                                                                                              'f20c9a787324' => bless( {
#                                                                                                                       'macAddress' => 'f20c9a787324',
#                                                                                                                       'trackingData' => {
#                                                                                                                                           '1362336813' => bless( {
#                                                                                                                                                                    'timestamp' => 1362336813,
#                                                                                                                                                                    'switchPort' => 'Fa1/0/21',
#                                                                                                                                                                    'switchIPAddress' => '172.20.22.138',
#                                                                                                                                                                    'hostIPAddress' => ''
#                                                                                                                                                                  }, 'UvT::NOC::Tracking::TrackingData' )
#                                                                                                                                         }
#                                                                                                                     }, 'UvT::NOC::Tracking::MACAddress' )
#                                                                                            },
#                                                                            'id' => '3'
#                                                                          }, 'UvT::NOC::Tracking::Vlan' )
#                                                            }
#                                                }, 'UvT::NOC::Tracking::Network' )
#                              },
#                 'creationDate' => 1365656409
#               }, 'UvT::NOC::Tracking::UserTrackingData' );
#
#
# Omschrijving :
#       Functie om ciscoworks file uit te lezen en de gegevens op te slaan in een datastructure UserTrackingData welke wordt geretourneerd

sub readUserTrackingData
{
	my $filename = shift;

	my $parser = XML::LibXML->new();
	my $doc    = $parser->parse_string(stringFromFile($filename));

	my $trackingServer = trim($doc->findnodes('/UTDetails/CMServer')->string_value);
	my $creationDate = trim($doc->findnodes('/UTDetails/CreatedAt')->string_value);

	my $trackingData = new UvT::NOC::Tracking::UserTrackingData($trackingServer, $creationDate);

	foreach my $utdata ($doc->findnodes('/UTDetails/UTData'))
	{
		# check netwerk op basis van VTPDomain
		my $vtpDomain = trim($utdata->findnodes('./VTPDomain')->string_value);

		my $network = "unknown";
		if ($vtpDomain =~ m/^KUB*/) { $network = "UvT"; }
		if ($vtpDomain =~ m/^Breeburg*/) { $network = "WMB"; }

		# als het netwerk nog niet bestaat in de tracking data structure -> toevoegen
		unless ($trackingData->hasNetwork($network))
		{
			my $newNetwork = new UvT::NOC::Tracking::Network($network);
			$trackingData->addNetwork($newNetwork);
		}

		# hellup, de inhoud van ciscoworks is niet altijd correct wat de inhoud van node VLAN en VLANId betreft
		# hiervoor deze fix :
        my $cwVlan = trim($utdata->findnodes('./VLAN')->string_value);
		my $cwVlanId = trim($utdata->findnodes('./VLANId')->string_value);

		my $vlan = $cwVlan =~ /^\d+$/ ? $cwVlan : $cwVlanId;


		# als het vlan niet bestaat in het betreffende netwerk object -> toevoegen
		unless ($trackingData->getNetworkByName($network)->hasVlan($vlan))
		{
			my $newVlan = new UvT::NOC::Tracking::Vlan($vlan);
			$trackingData->getNetworkByName($network)->addVlan($newVlan);
		}

		my $mac = trim($utdata->findnodes('./MACAddress')->string_value);
		$mac =~ s/[^a-zA-Z0-9 ]//g;

		# als het mac address niet bestaat in het betreffende vlan object -> toevoegen
		unless ($trackingData->getNetworkByName($network)->getVlanById($vlan)->hasMacAddress($mac))
		{
			my $newMac = new UvT::NOC::Tracking::MACAddress($mac);
			$trackingData->getNetworkByName($network)->getVlanById($vlan)->addMacAddress($newMac);	
		}

		# op dit punt bestaat het betreffende mac adres in de het trackingData object -> TrackingData hieraan toevoegen
		my $switchIp = trim($utdata->findnodes('./Device')->string_value);
		my $switchPort = trim($utdata->findnodes('./Port')->string_value);
		my $timestamp = trim($utdata->findnodes('./LastSeen')->string_value);
		my $hostIp = trim($utdata->findnodes('./IPAddress')->string_value);

		my $tData = new UvT::NOC::Tracking::TrackingData($switchIp, $switchPort, $timestamp, $hostIp);
		$trackingData->getNetworkByName($network)->getVlanById($vlan)->getMacAddress($mac)->addTrackingData($tData);
	}

	return $trackingData;
}


# Functienaam : readIntreg
#
# Parameters : (1x)
#       1) de locatie van de intreg file
#
# Return : (1x)
#       1) een object van het type UvTRegList :
#
#        bless( {
#                 'registrations' => {
#                                      '00-0F-1F-90-65-10' => bless( {
#                                                                      'department' => 'FRW',
#                                                                      'hostname' => 'RG0447',
#                                                                      'description' => 'GX 280',
#                                                                      'vlanCode' => 'RG',
#                                                                      'username' => 'Sijtsma, I.',
#                                                                      'vlanId' => '17',
#                                                                      'vlanName' => 'frw-giw-fwsm-inside',
#                                                                      'mac' => '000f1f906510'
#                                                                    }, 'UvT::NOC::MacRegistration::UvTRegistration' ),
#                                      '00-24-E8-36-56-54' => bless( {
#                                                                      'department' => 'LIS',
#                                                                      'hostname' => 'XP0837',
#                                                                      'description' => '',
#                                                                      'vlanCode' => 'XP',
#                                                                      'username' => 'Studentenwerkplek',
#                                                                      'vlanId' => '21',
#                                                                      'vlanName' => 'its-prac',
#                                                                      'mac' => '0024e8365654'
#                                                                    }, 'UvT::NOC::MacRegistration::UvTRegistration' )
#                                    }
#               }, 'UvT::NOC::MacRegistration::UvTRegList' );
#
#
# Omschrijving :
#       Functie om hosts file uit te lezen en de koppeling ip-adres op te slaan als
#       object Host in het object HostList welke wordt geretourneerd

sub readIntreg
{
	my $filename = shift;

	my $data = stringFromFile($filename);
    
	my $regList = new UvT::NOC::MacRegistration::UvTRegList();

	foreach my $line (split('\n', $data))
    {
		# De regel moet bestaan uit 12 velden gescheiden met een '|' (dus 11x '|')
		if ($line =~ /([^|]*\|){11}/)
		{
			my @data = split(/\|/, trim($line));
			my $mac = trim($data[0]);				$mac =~ s/[^a-zA-Z0-9 ]//g;
			my $department = trim($data[1]);
			my $vlanCode = trim($data[2]);
			my $hostname = trim($data[3]);
			my $username = trim($data[4]);
			my $description = trim($data[10]);

			my @vlandata = split(/\*/, trim($data[11]));
			my $vlanId = $vlandata[1] ? trim($vlandata[0]) : "";
			my $vlanName = $vlandata[1] ? trim($vlandata[1]) : "";

			if ($vlanId eq "" && $vlanCode eq "X") { $vlanId = "1" }

			$regList->addRegistration(new UvT::NOC::MacRegistration::UvTRegistration(lc($mac), $department, $vlanId, $vlanCode, $vlanName, $hostname, $username, $description));
		}
	}

	return $regList;
}


# Functienaam : readWMBReg
#
# Parameters : (1x)
#       1) de locatie van de intreg file
#
# Return : (1x)
#       1) een object van het type UvTRegList :
#
#		bless( {
#                 'addresses' => {
#                                  'V Doorenstr 38 21' => bless( {
#                                                                  'number' => '38 21',
#                                                                  'street' => 'V Doorenstr',
#                                                                  'registrations' => {
#                                                                                       'Figaroa-00-16-0A-22-3E-95--13-3-2013-niet-UvT-studenten' => bless( {
#                                                                                                                                                             'regDate' => '13-3-2013',
#                                                                                                                                                             'mac1' => '00-16-0A-22-3E-95',
#                                                                                                                                                             'vlanName' => 'niet-UvT-studenten',
#                                                                                                                                                             'mac2' => '',
#                                                                                                                                                             'name' => 'Figaroa'
#                                                                                                                                                           }, #'UvT::NOC::MacRegistration::WMBRegistration' ),
#                                                                                       'Albers-64-D1-A3-0D-7A-DF--27-11-2013-UvT-studenten' => bless( {
#                                                                                                                                                        'regDate' => '27-11-2013',
#                                                                                                                                                        'mac1' => '64-D1-A3-0D-7A-DF',
#                                                                                                                                                        'vlanName' => 'UvT-studenten',
#                                                                                                                                                        'mac2' => '',
#                                                                                                                                                        'name' => 'Albers'
#                                                                                                                                                      }, #'UvT::NOC::MacRegistration::WMBRegistration' )
#                                                                                     }
#                                                                }, 'UvT::NOC::MacRegistration::WMBAddress' ),
#                                  'Daltonerf 37 01' => bless( {
#                                                                'number' => '37 01',
#                                                                'street' => 'Daltonerf',
#                                                                'registrations' => {
#                                                                                     'Jong-00-A0-C9-42-FD-95---UvT-studenten' => bless( {
#                                                                                                                                          'regDate' => '',
#                                                                                                                                          'mac1' => '00-A0-C9-42-FD-95',
#                                                                                                                                          'vlanName' => 'UvT-studenten',
#                                                                                                                                          'mac2' => '',
#                                                                                                                                          'name' => 'Jong'
#                                                                                                                                        }, 'UvT::NOC::MacRegistration::WMBRegistration' )
#                                                                                   }
#                                                              }, 'UvT::NOC::MacRegistration::WMBAddress' )
#                                }
#               }, 'UvT::NOC::MacRegistration::WMBRegList' );
#
# Omschrijving :
#       Functie om de WMB registratie file uit te lezen en op te slaan in de datastructure WMBRegList welke wordt geretourneerd

sub readWMBReg
{
	my $filename = shift;

	my $data = stringFromFile($filename);
    
	my $regList = new UvT::NOC::MacRegistration::WMBRegList();

	foreach my $line (split('\n', $data))
	{
		# De regel moet bestaan uit 8 velden gescheiden met een '|' (dus 7x '|')
		if ($line =~ /([^|]*\|){7}/)
		{
			my @data = split(/\|/, trim($line));
			my $mac1 = trim($data[3]);				$mac1 =~ s/[^a-zA-Z0-9 ]//g;
			my $mac2 = trim($data[4]);				$mac2 =~ s/[^a-zA-Z0-9 ]//g;
			my $name = trim($data[0]);
			my $street = trim($data[5]);
			my $number = trim($data[6]);
			my $regDate = trim($data[1]);
			my $vlanName = trim($data[7]);

			#$regList->addRegistration(new UvT::NOC::MacRegistration::WMBRegistration(lc($mac1), lc($mac2), $name, $street, $number, $regDate, $vlanName));
			$regList->addRegistration($street, $number, $name, $mac1, $mac2, $regDate, $vlanName);
		}
	}

	return $regList;
}

1;
