# $Id: FileTransfer.pm 6701 2012-10-09 12:24:35Z joostvb $
# $URL: https://pong.uvt.nl/its-nettel/Netwerken/Procedures/libnoc-infra-perl/lib/NOC/FileTransfer.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package NOC::FileTransfer;

use NOC::Util::FileUtil;
use Net::Lite::FTP;
use Net::FTP;
use LWP::UserAgent;
use XML::LibXML;

our $debug;


# Functienaam : downloadFileUsingFTP
#
# Parameters : (3x)
#       1) een hash met de volgende ftp client-gegevens :
#		- host
#		- username
#		- password
#		- passive (1 of 0)
#		- filedir (locatie van de file op de ftp server), indien niet meegegeven wordt de standaard dir gebruikt
#	2) naam van het op te halen bestand
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory)	 
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftp

sub downloadFileUsingFTP
{
	my $conf_ref = shift;
    my $filename = shift;
	my $destinationDir = shift;
	my $destinationFilename = shift;

	#chdir($destination) or die("Cannot change to $destination : $!");

	if ($debug) { warn "Retreiving $filename from $conf_ref->{host} ...\n"; }

	eval
	{
		if ($debug) { warn "Connecting to host : $conf_ref->{host}\n"; }
		my $ftp = Net::FTP->new($conf_ref->{host}, Debug => $debug, Passive => $conf_ref->{passive})
			or die "Cannot connect to $conf_ref->{host}";

		if ($debug) { warn "Login\n"; }
		$ftp->login($conf_ref->{username}, $conf_ref->{password})
			or die "Cannot login : ", $ftp->message;

		if ($conf_ref->{filedir})
		{
			if ($debug) { warn "Change to directory : $conf_ref->{filedir}\n"; }
			$ftp->cwd($conf_ref->{filedir})
				or die "Cannot change to working directory : ", $ftp->message;
		}

		if ($debug) { warn "Get file : $filename\n"; }
		$ftp->get($filename, "$destinationDir/$destinationFilename")
			or die "Cannot get file : ", $ftp->message;

		if ($debug) { warn "Closing connection\n"; }
		$ftp->quit()
			or die "Cannot close connection";
	};
	die "Error retreiving file $filename : $@" if $@;
}


# Functienaam : downloadFileUsingHTTPS
#
# Parameters : (3x)
#       1) een hash met de volgende https client-gegevens :
#		- host
#		- port
#		- realm
#		- username
#		- password
#	2) de url van het op te halen bestand
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftp

sub downloadFileUsingHTTPS
{
	my $conf_ref = shift;
	my $url = shift;
	my $destination = shift;

	eval
	{
		$ENV{HTTPS_DEBUG} = $debug;

		my $useragent = LWP::UserAgent->new();
		$useragent->credentials($conf_ref->{host} . ":" . $conf_ref->{port}, $conf_ref->{realm}, $conf_ref->{username}, $conf_ref->{password});
		my $request = HTTP::Request->new(GET => "$url");
		my $response = $useragent->request($request);
		
		unless ($response->is_success()) { die "Cannot get url $url : ", $response->status_line; }

		NOC::Util::FileUtil::stringToFile($destination, $response->content);
	};
        if($@)
	{
                warn "Error retreiving $url : $@\n";
	};
}


# Functienaam : downloadUserTrackingUsingHTTPS
#
# Parameters : (2x)
#       1) een hash met de volgende gegevens om te verbinden met de user tracking data extraction engine van ciscoworks :
#		- username
#		- password
#		- url
#		- command
#	2) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om de user tracking gegevens van ciscoworks te downloaden

sub downloadUserTrackingUsingHTTPS
{
	my $conf_ref = shift;
	my $destination = shift;

	eval
	{
		my $payload = constructUserTrackingPayload($conf_ref);

		if ($debug) { warn "Retreive user-tracking data with payload : " . $payload->toString(); }

		my $useragent = LWP::UserAgent->new();
		my $request = HTTP::Request->new(POST => $conf_ref->{url});
		$request->content_type('text/html');
		$request->content($payload->toString());
		my $response = $useragent->request($request);

		unless ($response->is_success()) { die "Cannot get url $conf_ref->{url} : ", $response->status_line; }
	
		my $output = $response->content;

        	# Strip out embedded newlines and returns if line doesn't end in tag.
        	$output =~ s/\r//g;
        	$output =~ s/([^>])\n/$1/g;

        	# Take out leading and trailing spaces
        	$output =~ s/\>\  /\>/g;
        	$output =~ s/\  \</\</g;

		NOC::Util::FileUtil::stringToFile($destination, $output);
        };
        if($@)
        {
                warn "Error retreiving usertracking data from ciscoprime : $@\n";
        };
}


# Functienaam : constructUserTrackingPayload
#
# Parameters : (1x)
#       1) een hash met de volgende gegevens om te verbinden met de user tracking data extraction engine van ciscoworks :
#		- username
#		- password
#		- url
#		- command
#
# Return : (1x)
#
#	1) payload in xml-formaat
#
# Omschrijving :
#       Functie om de payload in xml-formaat te genereren voor de data extraction engine van ciscoworks

sub constructUserTrackingPayload
{
	my $conf_ref = shift;

	my $payload = XML::LibXML::Document->new();
	my $root = $payload->createElement('payload');
	$payload->setDocumentElement($root);

	my $username = $payload->createElement('username');
	$username->appendText($conf_ref->{username});
	$root->appendChild($username);

	my $password = $payload->createElement('password');
        $password->appendText($conf_ref->{password});
        $root->appendChild($password);	

    	my $command = $payload->createElement('command');
        $command->appendText($conf_ref->{command});
        $root->appendChild($command);

	return $payload;
}


# Functienaam : debug
#
# Parameters : (1x)
#       1) debugwaarde, 1 = aan, 0 = uit
#
# Return : -
#
# Omschrijving :
#       Functie om debugging aan te zetten

sub debug
{
	$debug = shift;
}

1;
