use utf8;
use strict;
use warnings FATAL => 'all';

package KUB::Aselect::Request;

use Aselect::Client;
use Aselect::Util qw(aselect_split);

use KUB::Request -self;

field crypto => sub { shift->cfg->crypto };

field cur_aselect_ticket => sub {
	my $self = shift;
	if(my $ticket = $self->cookie('aselect')) {
		my ($uid, $org) = eval { aselect_verify_ticket($ticket) };
		if($@) {
			warn "aselect_verify_ticket: $@";
		} else {
			$self->aselect_login($uid);
			return $ticket;
		}
	}
	return '';
};

field new_aselect_ticket => sub { shift->cur_aselect_ticket };

sub aselect_self_url {
	my $url = $self->self_url;
	if(my $domain = $self->cfg->httpsdomain) {
		$url =~ s{^http://[a-z0-9-]+([/?]|$)}{https://$domain$1}i;
		$url =~ s{^http:}{https:}i;
	}
	$url =~ s{\?(.*)}{} or return $url;
	my $query = '';
	foreach my $q (split(qr{[;&]+}, $1)) {
		next if $q =~ /^(?:aselect_credentials|rid|a-select-server)(?:=|$)/;
		$query .= "&$q";
	}
	$query =~ s{^&}{?};
	return $url.$query;
}

field aselect_url => \&aselect_self_url;

param rid;
param aselect_credentials => sub {
	return undef unless defined;
	my $self = shift;

	my $cred = $_;
	my $rid = $self->rid;
	die $self->error('aselect_rid') unless defined $rid;
	my $ticket = eval { aselect_verify_credentials($rid, $cred) };
	if($@) {
		warn "aselect_verify_credentials: $@";
	} else {
		my %ticket = aselect_split($ticket);
		$self->aselect_login($ticket{uid});
		$self->new_aselect_ticket($ticket);
	}
	$_ = 1;
};

field privileged_user => sub {
	my $self = shift;

	my $user = $self->remote_user;
	return undef unless defined $user;
	my $luser = lc $user;

	my $users = $self->cfg->privileged_users;
	unless(exists $users->{$luser}) {
		warn "Unknown privileged user '$user'\n";
		return undef;
	}
	my $u = $users->{$luser};

	my $uid = $self->param('uid');
	unless(defined $uid && $uid ne '') {
		warn "Privileged user '$user' failed to supply a uid\n";
		return undef;
	}

	$uid = lc($uid);

	if(defined $u) {
		unless(exists $u->{$uid}) {
			warn "Privileged user '$user' not authorized for '$uid'\n";
			return undef;
		}
	} else {
		$uid =~ tr/a-z0-9_-//cd;
		if($uid eq '') {
			warn "Privileged user '$user' failed to supply a proper uid\n";
			return undef;
		}
	}

	warn "Privileged user '$user' acting on behalf of '$uid'\n";
	return $uid;
};

field aselect_login => sub {
	my $self = shift;

	if(my $uid = $self->privileged_user) {
		return $uid;
	}

	if(my $ticket = $self->cookie('aselect')) {
		my ($uid, $org) = eval { aselect_verify_ticket($ticket) };
		return $uid unless $@;
		warn "aselect_verify_ticket: $@";
	}

	return undef;
};

sub aselect_force_login {
	if(my $uid = $self->aselect_login) {
		return $uid;
	}
	my $cfg = $self->cfg;
	my $sso = aselect_authenticate($cfg->aselect_id, $self->aselect_self_url);
	$self->aselect_url($sso);
	my $doc = new KUB::Aselect::Document(req => $self);
	die $doc->response;
}

param nonce => sub {
	my $self = shift;
	return 1 if $self->privileged_user;
	my $ticket = $self->cur_aselect_ticket;
	if($ticket eq '') {
		$self->aselect_force_login;
		confess("internal error: unable to acquire valid ticket");
	}
	my $noise = 'n'.$self->remote_addr.$;.$ticket;
	$_ = eval { $self->crypto->check_token($noise, $_, $self->cfg->nonce_expiry); 1 };
};

sub create_nonce {
	return undef if $self->privileged_user;
	my $ticket = $self->new_aselect_ticket;
	if($ticket eq '') {
		$self->aselect_force_login;
		confess("internal error: unable to acquire valid ticket");
	}
	my $noise = 'n'.$self->remote_addr.$;.$ticket;
	return scalar $self->crypto->create_token($noise);
}
