# $Id: Alias.pm 38022 2012-10-22 08:17:04Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki/Database/Alias.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Kiki::Database::Alias;

use Email::Address;
use JSON;

use Clarity -self;

use constant type => 'alias';

field db;

field id;
field name;
field domain;
field addressbook;
field destinations => sub {
	my $self = shift;
	return $self->db->destinations_for($self->id);
};

field referrers => [];

field original => sub {
	my $clone = shift->clone;
	$clone->original(undef);
	return $clone;
};

sub names {
	return [$self];
}

sub clone {
	my $class = ref $self;
	my $clone = $class->new(
		id => $self->id,
		name => $self->name,
		domain => $self->domain,
		addressbook => $self->addressbook,
	);
	if(my $dst = $self->destinations) {
		$clone->destinations($dst);
	}
	if(my $ref = $self->referrers) {
		$clone->referrers($ref);
	}
	return $clone;
}

sub full {
	return $self->name . '@' . $self->domain->name;
}

sub unique {
	my $db = $self->db;
	my $a = $db->fetch_alias($self->full);
	return 1 unless $a;
	my $id = $self->id;
	return 1 if defined $id && $id == $a->id;
	return 0;
}

sub update_name {
	my $newname = shift;
	return [error => 'empty-name'] if $newname eq '';
	if(my $original = $self->original) {
		my $oldname = $original->name;

		return [ok => 'unchanged', $newname]
			if $newname eq $oldname;

		if(lc($newname) eq lc($oldname)) {
			$self->name($newname);
			return [ok => 'case', $newname];
		}
	}

	return [error => 'malformed']
		if $newname !~ /^[a-z0-9_]+(?:[.-][a-z0-9_]+)*$/i;

	$self->name($newname);
	return [ok => 'changed', $newname];
}

sub update_domain {
	my $newdomain = shift;
	return [error => 'empty-domain'] if $newdomain eq '';

	$newdomain = $self->db->domain_by_name($newdomain);
	return [error => 'unknown'] unless $newdomain;

	if(my $original = $self->original) {
		my $olddomain = $original->domain;
		return [ok => 'unchanged', $olddomain->name]
			if $newdomain == $olddomain;
	}

	$self->domain($newdomain);
	return [ok => 'changed', $newdomain->name];
}

sub update_addressbook {
	my $newaddressbook = shift;
	return [error => 'empty-addressbook'] unless defined $newaddressbook;

	if(my $original = $self->original) {
		my $oldaddressbook = $original->addressbook;
		return [ok => 'unchanged', $oldaddressbook ? JSON::true : JSON::false]
			if $newaddressbook == $oldaddressbook;
	}

	$self->addressbook($newaddressbook);
	return [ok => 'changed', $newaddressbook ? JSON::true : JSON::false];
}

sub update_destinations {
	my $destinations = shift;

	if(my $original = $self->original) {
		$original->destinations;
	}

	my $db = $self->db;

	my @notes;
	my %uniq;
	my $fail;

	foreach(@$destinations) {
		if($_ eq '') {
			# leeg
			push @notes, [ok => 'empty', ''];
			next;
		} elsif(/^\d{6}$/) {
			# anr, check of-ie bestaat
			my $a = $db->person_for_anr($_);
			if($a) {
				my $addr = $a->full;
				$uniq{$addr} = $a;
				push @notes, [ok => 'anr', $addr];
			} else {
				$fail = 1;
				push @notes, [error => 'unknown-anr'];
			}
		} elsif(/\@/) {
			my ($x, $y) = Email::Address->parse($_);
			if(defined $y) {
				$fail = 1;
				push @notes, [error => 'multiple'];
				next;
			}
			unless(defined $x && $x->original eq $_) {
				$fail = 1;
				push @notes, [error => 'invalid-address'];
				next;
			}

			my $a = $db->fetch_alias($x->address);
			if($a) {
				my $addr = $a->full;
				$uniq{$addr} = $a;
				push @notes, [ok => $a->type, $addr];
			} elsif($db->domain_by_name($x->host)) {
				$fail = 1;
				push @notes, [error => 'unknown-alias'];
			} else {
				my $addr = $x->address;
				$addr =~ s/\@.*/\L$&/;
				if($db->hostresolvable($addr)) {
					my $e = new UvT::Kiki::Database::External(db => $db, address => $addr);
					$uniq{$addr} = $e;
					push @notes, [ok => 'external', $addr];
				} else {
					$fail = 1;
					push @notes, [error => 'unknown-domain'];
				}
			}
		} else {
			# localpart, check of-ie bestaat en uniek is
			my $res = $db->fetch_alias_by_localname($_);
			if(@$res < 1) {
				$fail = 1;
				push @notes, [error => 'unknown-alias'];
			} elsif(@$res > 1) {
				$fail = 1;
				push @notes, [error => 'not-unique'];
			} else {
				my ($a) = @$res;
				my $addr = $a->full;
				$uniq{$addr} = $a;
				push @notes, [ok => $a->type, $addr];
			}
		}
	}

	unless($fail) {
		if(%uniq) {
			$self->destinations([values %uniq]);
		} else {
			return [[error => 'empty-destinations']];
		}
	}

	return \@notes;
}

sub remove {
	if($self->db->has_referrers($self)) {
		return ['error', 'has-referrers'];
	} else {
		$self->db->remove_alias($self);
		return ['ok', 'removed'];
	}
}

sub store {
	if($self->id) {
		$self->db->update_alias($self);
	} else {
		$self->db->create_alias($self);
	}
}

sub toString {
	return $self->full . ': ' . join(', ', map { $_->full } @{$self->destinations});
}

sub json_boolean() {
	my $val = shift;
	return JSON::null unless defined $val;
	return $val ? JSON::true : JSON::false;
}

sub TO_JSON {
	my %res = (type => $self->type);
	$res{name} = $self->name
		if $self->name_isset;
	$res{domain} = $self->domain->name
		if $self->domain_isset;
	$res{addressbook} = json_boolean($self->addressbook)
		if $self->addressbook_isset;
	$res{destinations} = $self->destinations
		if $self->destinations_isset;
	$res{referrers} = $self->referrers
		if $self->referrers_isset;
	return \%res;
}
