/* $Id: plugin.c 5270 2015-02-19 13:06:24Z wsl $ */
/* $URL: https://svn.non-gnu.uvt.nl/uvt-dev/trunk/sources/dovecot-prjquota/src/plugin.c $ */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/quota.h>
#include <sys/stat.h>
#include <xfs/xqm.h>

#include <dovecot/config.h>
#include <dovecot/lib.h>
#include <dovecot/str.h>
#include <dovecot/hash.h>
#include <dovecot/istream.h>
#include <dovecot/imap-common.h>
#include <dovecot/imap-client.h>
#include <dovecot/imap-commands.h>
#include <dovecot/imap-quote.h>

#include "src/plugin.h"

struct project {
	const char *name;
	unsigned int id;
	dev_t dev;
};
const struct project project_0 = {0};

static pool_t pool;
static struct hash_table *projects_by_name;
static struct hash_table *projects_by_box;
static struct hash_table *names_by_num;

static void *p_memdup(pool_t p, void *src, size_t len) {
	void *dst = p_malloc(p, len);
	memcpy(dst, src, len);
	return dst;
}

static unsigned int uint_hash(const void *p) {
	return *(const unsigned int *)p;
}

static int uint_cmp(const void *a, const void *b) {
	unsigned int x = *(const unsigned int *)a;
	unsigned int y = *(const unsigned int *)b;
	return x < y ? -1 : x > y ? 1 : 0;
}

static int (*const str_cmp)(const void *, const void *) =
	(int (*)(const void *, const void *))strcmp;

static const char *getname(unsigned int num) {
	const char *name, *line;
	char *off;
	int fd;
	unsigned int u, lineno;
	static bool projid_failed = FALSE;
	struct istream *istream;

	name = hash_table_lookup(names_by_num, &num);
	if(name)
		return name;

	if(projid_failed)
		return NULL;

	fd = open("/etc/projid", O_RDONLY|O_NOCTTY);
	if(fd == -1) {
		i_error("Can't open /etc/projid: %s", strerror(errno));
		projid_failed = TRUE;
		return NULL;
	}
	istream = i_stream_create_fd(fd, 4096, TRUE);
	while((line = i_stream_read_next_line(istream))) {
		lineno++;
		while(isspace(*line))
			line++;
		if(!*line || *line == '#')
			continue;
		off = strrchr(line, ':');
		if(!off) {
			i_warning("Bad line at /etc/projid:%u", lineno);
			continue;
		}
		*off++ = '\0';
		u = (unsigned int)strtoul(off, NULL, 10);
		if(!u) {
			i_warning("Bad line at /etc/projid:%u", lineno);
			continue;
		}
		if(!hash_table_lookup(names_by_num, &u))
			hash_table_update(names_by_num,
				p_memdup(pool, &u, sizeof u),
				p_strdup(pool, line));
	}
	i_stream_destroy(&istream);

	return hash_table_lookup(names_by_num, &num);
}

static inline void *notconst(const void *c) {
	union {
		void *p;
		const void *c;
	} x;
	x.c = c;
	return x.p;
}

static const struct project *getproject(const char *path) {
	struct project *prj;
	int fd;
	struct stat st;
	struct fsxattr fsx = {0};
	const char *name;

	fd = open(path, O_RDONLY|O_NOCTTY);
	if(fd == -1)
		return i_warning("open(%s): %s", path, strerror(errno)), NULL;

	if(fstat(fd, &st) == -1)
		return i_warning("stat(%s): %s", path, strerror(errno)), NULL;

	if(ioctl(fd, XFS_IOC_FSGETXATTR, &fsx) == -1)
		return i_warning("ioctl(%s, XFS_IOC_FSGETXATTR): %s",
			path, strerror(errno)), NULL;

	if(!fsx.fsx_projid)
		return i_warning("%s does not have a project ID assigned", path), NULL;

	name = getname(fsx.fsx_projid);
	if(!name)
		return i_warning("%s (id %u) does not have a name assigned", path, (unsigned int)fsx.fsx_projid), NULL;

	prj = hash_table_lookup(projects_by_name, name);
	if(!prj) {
		prj = p_malloc(pool, sizeof *prj);
		*prj = project_0;
		prj->name = name;
		prj->id = fsx.fsx_projid;
		prj->dev = st.st_dev;
		hash_table_insert(projects_by_name, notconst(name), prj);
	}
	return prj;
}

static void sendquotaline(struct client_command_context *cmd, const struct project *prj) {
	string_t *str;
	char dev[100];
	fs_disk_quota_t q = {0};
	dev_t st_dev = prj->dev;

	sprintf(dev, "/dev/block/%u:%u",
		(unsigned int)(st_dev >> 8),
		(unsigned int)(st_dev & 0xFF));

	if(quotactl(QCMD(Q_XGETQUOTA, XFS_PROJ_QUOTA), dev, (int)prj->id, (caddr_t)&q)) {
		i_error("Can't get quota for project %u (%s): %s",
			prj->id, prj->name, strerror(errno));
		client_send_line(cmd->client, "* BAD System error getting quota");
		return;
	}

	str = t_str_new(128);
	str_append(str, "* QUOTA ");
	imap_quote_append_string(str, prj->name, FALSE);
	str_printfa(str, " (STORAGE %lu %lu)",
		(unsigned long)q.d_bcount / 2,
		(unsigned long)q.d_blk_softlimit / 2);
	client_send_line(cmd->client, str_c(str));
}

static bool cmd_getquota(struct client_command_context *cmd) {
	const char *name;
	const struct project *prj;

    if(!client_read_string_args(cmd, 1, &name))
        return FALSE;
	if(!name)
		return FALSE;

	prj = hash_table_lookup(projects_by_name, name);
	if(prj) {
		sendquotaline(cmd, prj);
		client_send_tagline(cmd, "OK GETQUOTA completed.");
	} else {
		i_warning("GETQUOTA called for wrong quota root (%s)", name);
		client_send_tagline(cmd, "NO Quota root unknown.");
	}
	return TRUE;
}

static bool cmd_setquota(struct client_command_context *cmd) {
	const struct imap_arg *args;
	const char *root;

	/* <quota root> <resource limits> */
	if(!client_read_args(cmd, 2, 0, &args))
		return FALSE;

	if(!imap_arg_get_nstring(&args[0], &root) || !root || args[1].type != IMAP_ARG_LIST)
		return client_send_command_error(cmd, "Invalid arguments."), TRUE;

	client_send_tagline(cmd, "NO Quota setting not supported.");
	return TRUE;
}

static bool cmd_getquotaroot(struct client_command_context *cmd) {
	const char *mailbox, *decoded, *storage, *path;
	const struct project *prj;
	struct mail_namespace *namespace;
	string_t *str;

	/* <mailbox> */
	if(!client_read_string_args(cmd, 1, &mailbox))
		return FALSE;

	prj = hash_table_lookup(projects_by_box, mailbox);
	if(!prj) {
		decoded = mailbox;
		namespace = client_find_namespace(cmd, &decoded);
		if(!namespace)
			return TRUE;

		storage = mailbox_list_get_storage_name(namespace->list, decoded);
		if(!storage)
			return client_send_tagline(cmd, "NO Quota not supported for this mailbox."), TRUE;

		path = mailbox_list_get_path(namespace->list, storage, MAILBOX_LIST_PATH_TYPE_DIR);
		if(!path)
			return client_send_tagline(cmd, "NO Quota not supported for this mailbox."), TRUE;

		prj = getproject(path);
		if(!prj)
			return client_send_tagline(cmd, "NO Quota root not found for this mailbox."), TRUE;

		hash_table_insert(projects_by_box, p_strdup(pool, mailbox), notconst(prj));
	}

	str = t_str_new(128);
	str_append(str, "* QUOTAROOT ");
	imap_quote_append_string(str, mailbox, FALSE);
	str_append_c(str, ' ');
	imap_quote_append_string(str, prj->name, FALSE);
	client_send_line(cmd->client, str_c(str));

	sendquotaline(cmd, prj);

	client_send_tagline(cmd, "OK GETQUOTAROOT completed.");

	return TRUE;
}

static struct module *imap_prjquota_module;
static imap_client_created_func_t *next_hook_client_created;

static void imap_prjquota_client_created(struct client **client) {
	if(mail_user_is_plugin_loaded((*client)->user, imap_prjquota_module))
		str_append((*client)->capability_string, " QUOTA");

	if(next_hook_client_created)
		next_hook_client_created(client);
}

void imap_prjquota_plugin_init(struct module *module) {
	pool = pool_alloconly_create("imap_prjquota", 65536);

	names_by_num = hash_table_create(system_pool, pool, 1024, uint_hash, uint_cmp);
	projects_by_box = hash_table_create(system_pool, pool, 128, str_hash, str_cmp);
	projects_by_name = hash_table_create(system_pool, pool, 128, str_hash, str_cmp);

	command_register("GETQUOTAROOT", cmd_getquotaroot, 0);
	command_register("GETQUOTA", cmd_getquota, 0);
	command_register("SETQUOTA", cmd_setquota, 0);

	imap_prjquota_module = module;
	next_hook_client_created = imap_client_created_hook_set(imap_prjquota_client_created);
}

void imap_prjquota_plugin_deinit(void) {
	imap_client_created_hook_set(next_hook_client_created);
	imap_prjquota_module = NULL;

	command_unregister("SETQUOTA");
	command_unregister("GETQUOTA");
	command_unregister("GETQUOTAROOT");

	hash_table_destroy(&projects_by_name);
	hash_table_destroy(&projects_by_box);
	hash_table_destroy(&names_by_num);

	pool_unref(&pool);
}
