use strict;
use warnings FATAL => 'all';

package Clarity::Util;
$INC{"Clarity/Util.pm"} = !undef;

use Exporter qw(import);
our @EXPORT_OK = qw(parse_arguments);

sub parse_arguments {
	my %flags;
	my @args;
	while(@_) {
		my $key = shift;
		if(defined $key && !ref $key && ord($key) == 45) { # -
			undef $flags{substr($key, 1)};
		} else {
			push @args, $key;
		}
	}
	return \%flags, @args;
}

package Clarity::Internals;

use Filter::Util::Call;
use Carp qw(croak carp confess cluck);

sub all_my_bases {
	my %bases;
	my @bases;
	no strict 'refs';
	while(@_) {
		my $class = shift;
		next if exists $bases{$class};
		undef $bases{$class};
		push @bases, $class;
		push @_, @{"${class}::ISA"};
	}
	return wantarray ? @bases : \%bases;
}

our $supercode = <<'EOT';
	sub super {
		my $method;
		my $frame = 1;
		while($method = (caller($frame++))[3]) {
			my $i = rindex($method, '::');
			next if $i == -1;
			substr($method, 0, $i, 'SUPER');
			next if $method eq 'SUPER::__ANON__';
			my ($self, @args) = DB::super_args($frame);
			return $self->$method(@_ ? @_ : @args);
		}
		Carp::confess("can't find super method");
	}
EOT

sub filter {
	my $preamble = "use utf8; use strict; use warnings FATAL => 'all';";
	filter_add(sub {
		my $status = Filter::Util::Call::filter_read();
		return $status if $status < 0;
		if(!$status) {
			$_ = ";1;$_";
			filter_del();
			return 1;
		}
		if(/^package\s/) {
			filter_del();
			return 1;
		}
		if(/^__(?:END|DATA)__\s*$/) {
			$_ = ";1;\n$_";
			filter_del();
			return 1;
		}
		s/^(sub\s+\w+\s*\{)(.*\n)/ $1 my \$self = shift; $2/;
		s/^(sub\s+\w+)\s*\(\s*\)\s*(\{)/ $1 $2/;
		$_ = "$preamble$_" if $preamble;
		undef $preamble;
		return 1;
	});
}

sub mixin {
	my $mix = shift;
	my $pkg = shift;

	my $mixed = join('-', $pkg, $mix, sort @_);

	no strict 'refs';

	my %skip;
	my @select = grep { ord != 33 or undef $skip{substr($_, 1)} } @_;
	if(@select) {
		foreach my $key (@select) {
			next if exists $skip{$key};
			my $val = $mix->can($key)
				or die "$pkg: no method '$key' in '$mix'\n";
			undef $skip{$key};
			*{"${mixed}::$key"} = $val;
		}
	} else {
		my $bases = all_my_bases($pkg);

		foreach my $base (all_my_bases($mix)) {
			next if exists $bases->{$base};
			my %symbols = %{"${base}::"};
			while(my ($key, $val) = each %symbols) {
				next unless defined &$val;
				next if exists $skip{$key};
				next if ord($key) == 95; # _
				my $can = $pkg->can($key);
				next if $can && $can == \&$val;
				undef $skip{$key};
				#warn "mixing $key() from $mix into $pkg ($mixed)";
				*{"${mixed}::$key"} = $mix->can($key);
			}
		}
	}

	@{"${mixed}::ISA"} = @{"${pkg}::ISA"};
	@{"${pkg}::ISA"} = ($mixed);
}

sub stub {
	my $package = shift;
	my $name = shift;
	my $message = @_ ? shift
		: "Method $name in package $package must be subclassed";
	my $code = "package $package; sub $name { Carp::confess(\$message) }";
	my $err = do { local $@; eval $code; $@ };
	confess($err) if $err;
}

sub field {
	my $package = shift;
	my $name = shift;
	my $initcode;
	my $init;
	if(@_) {
		$init = shift;
		if(ref $init) {
			if(Scalar::Util::reftype($init) eq 'CODE') {
				$initcode = '
					unless(@_ = $init->($self, $name, $package)) {
						return $self->{$name} if exists $self->{$name};
						return;
					}
					return $self->{$name} = shift;
				';
			} else {
				$init = Clarity::Factory::factory($init);
				$initcode = 'return $self->{$name} = $init->();';
			}
		} else {
			$initcode = 'return $self->{$name} = $init;';
		}
	} else {
		$init = "field $name in package $package used uninitialized";
		$initcode = "Carp::confess(\$init);\n";
	}
	my $code = "
		package $package;
		sub $name {
			my \$self = shift;
			return \$self->{\$name} = shift if \@_;
			return \$self->{\$name} if exists \$self->{\$name};
			$initcode
		}
		sub ${name}_isset { exists shift->{\$name} }
		sub ${name}_reset { delete shift->{\$name} }
	";
	my $err = do { local $@; eval $code; $@ };
	confess($err) if $err;
}

sub weakfield {
	my $package = shift;
	my $name = shift;
	my $initcode;
	my $init;
	my $assign = '
		my $val = shift;
		$self->{$name} = $val;
		Scalar::Util::weaken($self->{$name}) if ref $val;
		return $val;
	';
	if(@_) {
		$init = shift;
		if(ref $init) {
			if(Scalar::Util::reftype($init) eq 'CODE') {
				$initcode = '
					unless(@_ = $init->($self, $name, $package)) {
						return unless exists $self->{$name};
						Scalar::Util::weaken($self->{$name})
							if ref $self->{$name};
						return $self->{$name};
					}
				'.$assign;
			} else {
				$init = Clarity::Factory::factory($init);
				$initcode = '@_ = $init->();'.$assign;
			}
		} else {
			$initcode = 'return $self->{$name} = $init;';
		}
	} else {
		$init = "field $name in package $package used uninitialized";
		$initcode = "Carp::confess(\$init);\n";
	}
	my $code = "
		package $package;
		sub $name {
			my \$self = shift;
			if(\@_) { $assign }
			return \$self->{\$name} if exists \$self->{\$name};
			$initcode
		}
		sub ${name}_isset { exists shift->{\$name} }
		sub ${name}_reset { delete shift->{\$name} }
	";
	my $err = do { local $@; eval $code; $@ };
	confess($err) if $err;
}

sub const {
	my $package = shift;
	my $name = shift;
	my $initcode;
	my $init;
	if(@_) {
		$init = shift;
		if(ref $init) {
			if(Scalar::Util::reftype($init) eq 'CODE') {
				$initcode = '
					unless(@_ = $init->($self, $name, $package)) {
						return $self->{$name} if exists $self->{$name};
						return;
					}
					return $self->{$name} = shift;
				';
			} else {
				$init = Clarity::Factory::factory($init);
				$initcode = 'return $self->{$name} = $init->();';
			}
		} else {
			$initcode = 'return $self->{$name} = $init;';
		}
	} else {
		$init = "const field $name in package $package used uninitialized";
		$initcode = "return \$self->{\$name} = shift if \@_;
			Carp::confess(\$init);\n";
	}
	my $code = "
		package $package;
		sub $name {
			my \$self = shift;
			return \$self->{\$name} if exists \$self->{\$name};
			$initcode
		}
		sub ${name}_isset { exists shift->{\$name} }
		sub ${name}_reset { delete shift->{\$name} }
	";
	my $err = do { local $@; eval $code; $@ };
	confess($err) if $err;
}

package Clarity::Factory;
$INC{"Clarity/Factory.pm"} = !undef;

use Exporter qw(import);
use Carp qw(croak carp confess cluck);
use Scalar::Util qw(blessed reftype refaddr weaken isweak);

our @EXPORT_OK = qw(factory register_special_classes);

our %special_classes = (Regexp => undef);

sub seen { shift->{seen} //= {} }
sub weak { shift->{weak} //= {} }
sub vars { shift->{vars} //= {} }
sub leaves { shift->{leaves} //= {} }
sub cache { shift->{cache} //= {} }
sub fixcode { shift->{fixcode} //= [] }
sub weakcode { shift->{weakcode} //= [] }
sub order { shift->{order} //= [] }
sub done { shift->{done} //= {} }
sub toplevel { exists shift->{toplevel} }

sub new { bless {}, shift }

sub id {
	my $self = shift;
	my $name = join('', @_);
	return "\$$name" . ++$self->{id}{$name};
}

sub register_special_classes {
	@special_classes{@_} = ();
}

sub register_leaf {
	my $self = shift;
	my $case = shift;
	if(ref $case) {
		my $addr = refaddr $case;
		my $cache = $self->cache;
		return $cache->{$addr}
			if exists $cache->{$addr};
		my $leaves = $self->leaves;
		my $n = $self->id('LEAF');
		$leaves->{$n} = $case;
		return $cache->{$addr} = $n;
	} else {
		my $leaves = $self->leaves;
		my $n = $self->id('LEAF');
		$leaves->{$n} = $case;
		return $n;
	}
}

sub register_var {
	my $self = shift;
	my $case = shift;
	my $id = $self->id('VAR');
	return $self->vars->{refaddr $case} = $id;
}

sub register_order {
	my $self = shift;
	my $case = shift;
	my $vars = $self->vars;
	return if exists $vars->{refaddr $case};
	unshift @{$self->order}, $case;
	return $self->register_var($case);
}

sub analyze_SCALAR {
	my $self = shift;
	my $case = shift;

	if(isweak($$case) && !$self->toplevel) {
		$self->register_order($case);
		undef $self->{toplevel};
		$self->analyze($$case);
		delete $self->{toplevel};
	} else {
		$self->analyze($$case);
	}
}

*analyze_REF = *analyze_SCALAR;

sub analyze_HASH {
	my $self = shift;
	my $case = shift;
	# FIXME: scan for weak references
	while(my ($k, $c) = each(%$case)) {
		if(isweak($case->{$k}) && !$self->toplevel) {
			$self->register_order($case);
			undef $self->{toplevel};
			$self->analyze($c);
			delete $self->{toplevel};
		} else {
			$self->analyze($c);
		}
	}
}

sub analyze_ARRAY {
	my $self = shift;
	my $case = shift;
	# FIXME: scan for weak references
	foreach my $c (@$case) {
		if(isweak($c)) {
			if(!$self->toplevel) {
				$self->register_order($case);
				undef $self->{toplevel};
				$self->analyze($c);
				delete $self->{toplevel};
				return;
			}
		}
		$self->analyze($c);
	}
}

sub analyze {
	my $self = shift;
	my $case = shift;
	if(ref $case) {
		my $addr = refaddr $case;
		my $seen = $self->seen;
		if(exists $seen->{$addr}) {
			$self->register_order($case);
		} else {
			my $analyzer = 'analyze_' . reftype $case;
			return unless $self->can($analyzer);
			undef $seen->{$addr};
			$self->$analyzer($case);
		}
	}
}

sub codegen_SCALAR {
	my $self = shift;
	my $case = shift;

	my $c = $$case;

	if(my @el = $self->codegen($c, @_)) {
		my @code;
		# cheating
		if($el[0] =~ /^[\[\(\\\$]/) {
			@code = ("\\", @el);
		} else {
			@code = ("\\(", @el, ")");
		}
		if(isweak($$case)) {
			my $vars = $self->vars;
			my $me;
			if(exists $vars->{my $addr = refaddr $case}) {
				$me = $vars->{$addr};
			} else {
				$me = $self->register_var($case);
				$self->done->{$me} = $case;
				unshift @code, "my $me = ";
			}
			my $weakcode = $self->weakcode;
			push @$weakcode, "\tScalar::Util::weaken(\$$me);\n"
		}
		return @code;
	}

	my $vars = $self->vars;
	my $ref = $self->register_var(\undef);
	my $n = $vars->{refaddr $c};
	my $fixcode = $self->fixcode;
	push @$fixcode, "\t$ref = $n;\n";
	my $weakcode = $self->weakcode;
	push @$weakcode, "\tScalar::Util::weaken($ref);\n"
		if isweak($$case);
	return "\\my $ref";
}

*codegen_REF = \&codegen_SCALAR;

sub codegen_list {
	my ($self, $case, $fixups, $weak, $open, $close, @fixups) = @_;
	return unless %$fixups || @$weak;

	@fixups = keys %$fixups
		unless @fixups;

	my @code;

	my $me;
	my $vars = $self->vars;
	my $addr = refaddr $case;
	if(exists $vars->{$addr}) {
		$me = $vars->{$addr};
	} else {
		$me = $self->register_var($case);
		$self->done->{$me} = $case;
		unshift @code, "my $me = ";
	}

	my $fixcode = $self->fixcode;
	if(@fixups == 1) {
		my ($i, $c) = each(%$fixups);
		my $n = $vars->{refaddr $c};
		push @$fixcode, "\t$me\->$open$i$close = $n;\n";
	} elsif(@fixups) {
		push @$fixcode,
			"\t\@$me$open", join(', ', @fixups), "$close = (",
			join(', ', map { $vars->{refaddr $fixups->{$_}} } @fixups),
			");\n";
	}

	my $weakcode = $self->weakcode;
	push @$weakcode, map { "\tScalar::Util::weaken($me->$open$_$close);\n" } @$weak;

	return @code;
}

sub codegen_ARRAY {
	my $self = shift;
	my $case = shift;

	my $i = $[;
	my $sep = '';
	my @code = ('[');
	my %fixups;
	my @fixups;
	my @weak;

	foreach my $c (@$case) {
		push @weak, $i if isweak($c);
		push @code, $sep;
		$sep = ', ';
		if(my @el = $self->codegen($c, @_)) {
			push @code, @el;
		} else {
			push @code, 'undef';
			push @fixups, $i;
			$fixups{$i} = $c;
		}
		$sep = ', ';
		$i++;
	}

	return $self->codegen_list($case, \%fixups, \@weak, '[', ']', @fixups), @code, ']';
}

sub codegen_HASH {
	my $self = shift;
	my $case = shift;

	my $me;
	my $sep = '';
	my @code = ('{');
	my %fixups;
	my @weak;

	while(my ($k, $v) = each(%$case)) {
		my $i = $self->register_leaf($k);
		push @weak, $i if isweak($case->{$k});
		if(my @el = $self->codegen($v, @_)) {
			push @code, "$sep$i => ", @el;
			$sep = ', ';
		} else {
			$fixups{$i} = $v;
		}
	}

	return $self->codegen_list($case, \%fixups, \@weak, '{', '}'), @code, '}';
}

sub codegen_force {
	my $self = shift;
	my $case = shift;
	my $mutable = shift;
	if(ref $case) {
		my $codegen = 'codegen_' . reftype $case;
		return $self->register_leaf($case)
			unless $self->can($codegen);
		if(my $class = blessed $case) {
			return $self->register_leaf($case)
				if exists $special_classes{$class};
			my $n = $self->register_leaf($class);
			return 'bless(', $self->$codegen($case, 1), ", $n)";
		}
		return $self->$codegen($case);
	} elsif($mutable || defined $case) {
		return $self->register_leaf($case);
	} else {
		return 'undef';
	}
}

sub codegen {
	my $self = shift;
	my $case = shift;
	my $vars = $self->vars;
	if(my $n = ref $case && $vars->{refaddr $case}) {
		my $done = $self->done;
		return unless exists $done->{$n};
		return $n;
	}

	return $self->codegen_force($case, @_);
}

sub cleanroom_eval {
	do { local $@; @_ = eval shift; push @_, $@ };
	my $err = pop;
	die $err if $err;
	return wantarray ? @_ : pop;
}

sub factory {
	my $self = new Clarity::Factory;
	my $case = shift;

	$self->analyze($case);

	my $vars = $self->vars;
	my $leaves = $self->leaves;
	my $order = $self->order;
	my $done = $self->done;

	my @code = ("sub {\n");

	foreach my $v (@$order) {
		my $n = $vars->{refaddr $v};
		push @code, "\tmy $n = ", $self->codegen_force($v), ";\n";
		$done->{$n} = $v;
	}

	push @code, @{$self->fixcode}, @{$self->weakcode},
		"\treturn ", $self->codegen($case), ";\n}\n";
	warn join('', @code) if $ENV{CLARITY_DEBUG};

	if(%$leaves) {
		my @eval = ("sub {\n",
				"\tmy (", join(', ', keys %$leaves), ") = \@_;\n",
				"\tlocal \$\@;\n",
				"\tpush \@_, eval pop;\n",
				"\tdie \$\@ if \$\@;\n",
				"\treturn pop;\n",
			"}\n");
		#warn join('', @eval);

		my $eval = cleanroom_eval join('', @eval);
		return $eval->(values %$leaves, join('', @code));
	} else {
		return cleanroom_eval join('', @code);
	}
}

package Clarity;

use Carp qw(croak carp confess cluck);
use Exporter;
use Scalar::Util;

our @EXPORT_BASE = qw(self stub field weakfield const croak carp confess cluck);

sub stub(*@) { Clarity::Internals::field(scalar caller, @_) }
sub field(*@) { Clarity::Internals::field(scalar caller, @_) }
sub weakfield(*@) { Clarity::Internals::weakfield(scalar caller, @_) }
sub const(*@) { Clarity::Internals::const(scalar caller, @_) }

sub self {
	my $self = DB::self_args(2);
	Carp::confess("can't find my self")
		unless Scalar::Util::blessed($self);
	return $self;
}

sub import {
	my $pkg = caller;

	my $file = "$pkg.pm";
	$file =~ s/::/\//g;
	$INC{$file} = !undef;

	if(@_ > 1 && ord($_[1]) == 45) { # -
		my $me = shift;

		my $mode = substr(shift, 1);

		if($mode eq 'mixin') {
			return Clarity::Internals::mixin($me, $pkg, @_);
		} elsif($mode eq 'self') {
			Clarity::Internals::filter;
		} elsif($mode ne 'base') {
			confess("unknown mode '-$mode'");
		}

		my $code = "package $pkg; use base \$me; $Clarity::Internals::supercode";
		my $err = do { local $@; eval $code; $@ };
		confess($err) if $err;

		no strict 'refs';
		foreach my $base (Clarity::Internals::all_my_bases($me)) {
			my @exports = @{"${base}::EXPORT_BASE"}
				or next;
			local @{"${base}::EXPORT_OK"} = @exports;
			Exporter::export($base, $pkg, @exports);
		}

		return 1;
	}

	local $Exporter::ExportLevel = 1;
	return &Exporter::export;
}

sub new {
	my $self = bless {}, shift;
	while(@_) {
		my $key = shift;
		next unless defined $key;
		next if ref $key;
		if(ord($key) == 45) { # -
			substr($key, 0, 1, '');
			if($self->can($key)) {
				$self->$key();
			} else {
				undef $self->{$key};
			}
		} else {
			if($self->can($key)) {
				$self->$key(shift);
			} else {
				$self->{$key} = shift;
			}
		}
	}
	return $self;
}

sub DESTROY {}

sub AUTOLOAD {
	my $self = shift;
	my $name = our $AUTOLOAD
		or confess("AUTOLOAD called but \$AUTOLOAD not set");
	my $off = rindex($name, '::');
	confess("no package name in '$name'")
		if $off == -1;
	my $pkg = substr($name, 0, $off + 2, '');
	unless(exists $self->{$name}) {
		substr($pkg, -2, 2, '');
		local $Carp::CarpLevel = 1;
		confess("Can't locate object method \"$name\" via package \"$pkg\"");
	}
	return $self->{$name} = shift if @_;
	return $self->{$name};
}

package DB;

{
	no warnings 'redefine';
	sub self_args { 
		my (undef) = caller(shift); 
		return shift @DB::args;
	}
	sub super_args { 
		my (undef) = caller(shift); 
		return @DB::args;
	}
}

1;

__END__

=pod

=encoding utf8

=head1 NAME

Clarity - base class for Perl objects

=head1 SYNOPSIS

 use Clarity -self;

=head1 INTRODUCTION

Clarity is a base class for Perl objects that provides a number of
convenience functions that enable you to write clean, concise, readable
code.

Its design is very heavily influenced by Ingy döt Net's awesome Spiffy.pm.
So heavily in fact, that in many cases either can be a drop-in replacement
for the other (or with only minimal changes, see the list below).

=head1 BASIC USE

To use Clarity as the base for your class:

 package MyClass;
 use Clarity -self;

This will enable all features, including the automatic $self filtering.
To use Clarity as a base without the filtering, write:

 package MyClass;
 use Clarity -base;

If you want to subclass your class further, you can simply write:

 package MySubClass;
 use MyClass -self;

You can use Clarity based packages as if they're Exporters:

 use MyClass qw(foo bar !baz);

=head1 FEATURES

=head2 accessor methods

These functions allow you to access fields of your object in a more
structured way. Calling one of these function with a string as an argument
will create a method with that string as its name. Calling the method
with an argument will set the field to that value, calling it without an
argument will return the previously set value.

 field color;
 ...
 sub example {
     ...
     # set the color:
     $self->color('blue');
     print $self->color;

You can provide default values for each field like this:

 field color => 'white';
 field long_options => ['--help', '--version'];

Note that if you supply a reference, each time the default value is
assigned a complete deep copy is created for you. This way default values
aren't inadvertently shared between instances. And if you supply a code
reference, that code will be called (with $self as an argument) to provide
the default value. (If you really want to supply a coderef as a default
value, you can wrap it in an anonymous sub.)

If you did not supply a default value and the method is called, it will die
with an informational message. Simply supply a default like C<undef> if
this is undesirable.

 field quest => 'glory';

You can trigger the default by deleting the field from $self:

     $self->quest('the holy grail');
     print $self->quest;                # prints 'the holy grail'
     delete $self->{quest};             # or $self->quest_unset
     print $self->quest;                # prints 'glory' again

You can freely intermix these calls with direct accesses to the fields:

     field name;
     @$self{qw(name color quest)} = qw(Bob red fame);
     print $self->name;                 # will print 'Bob'

However, accessors have a few advantages:

=over

=item spell checking

if you mistype the function name you will get a clear error message

=item default values

relieve you of the need to initialize fields in your new() method

=item future proof

can be replaced with real methods later on, as your software evolves

=back

=head3 Accessor methods

Accessor methods are:

=head4 $self->I<name>()

Get or set the field.

=head4 $self->I<name>_isset()

Returns a boolean indicating whether the value has been set or otherwise
initialized.

=head4 $self->I<name>_unset()

Returns the field to its uninitialized state.

=head3 Accessor creation

The various accessor creation functions are:

=head4 field()

Creates a simple field, as described above.

=head4 weakfield()

Creates a weak field, which is like field() but with the additional feature
that Scalar::Util::weaken() is called every time the field is set.

=head4 const()

A field created using const() will gleefully ignore any attempts to set it.
There's exactly only exception: when no default is provided and the field
is still uninitialized it is allowed to set it once.

=head3 Default values

Clarity supports using default values of arbitrary complexity. It tries
fairly hard to efficiently perform a deep copy (by precompiling a generator
function). Some objects are uncopyable however, in particular XS wrappers
and regular expressions. For this reason you can declare that some classes
should be copied only shallowly:

=head4 Clarity::register_special_classes($classname ...)

Objects from a class registered with this function will be copied by
reference, instead of the usual deep copy.

 # do not attempt a deep copy of XML::LibXML::Document objects
 Clarity::register_special_classes('XML::LibXML::Document');

=head3 Automatic accessors

Clarity will automatically handle accessors for you if you call a method
which does not exist but for which a corresponding value in $self does
exist.

 # look ma, no field declaration!

 sub example {
     ...
     $self->{volume} = 11;
     print $self->volume;               # prints '11'
 }

This way you don't have to create accessors if you prefer to initialize
your fields in your new() method.

It also allows you to create Clarity instances directly, useful if all
you want to do is store some data.

 my $c = new Clarity(volume => 11, loud => 1);
 print $c->volume;                      # prints '11'

=head2 Automatic $self 

If you use Clarity with the -self option, Clarity will apply a few simple
(textual) transformations to your code. It will add the following at the
top:

 use strict;
 use warnings FATAL => 'all';
 use utf8;

It will add the requisite C<1;> at the bottom of your file.

But the main feature is that it will add a C<my $self = shift;> to every
method. For methods that should not have this, add an (empty) prototype:

 sub new() {
     my ($class, @args) = @_;
     ...
 }

Be aware that due to the simple line-based nature of the transformation,
the opening brace of the method needs to be on the same line as the sub
keyword and the name. Also, the sub keyword must be at the very start of
the line.

=head2 super() function

Clarity provides all of its derived classes with a super() function that
calls the super method of the current function. Unless you supply an
explicit list of arguments, it will use the argument list of the current
function invocation.

 use Clarity -self;

 field 'foo';
 field 'bar';

 sub new {
     my ($foo, $bar) = @_;
     $self = super(foo => $foo, bar => $bar);
     ...
 }

If you really want to override the argument list with an empty list, you
can of course still use $self->SUPER::yourmethod();

=head2 mixins

Mixins are a way to mix the functionality of another class into your own
without resorting to multiple inheritance.
Mixin classes are created on the fly, populated with methods from the class
you are mixing in and inserted just below the current class in the @ISA
chain. Example:

 package MyExtraClass;

 sub stuff { ... }

 package MySubClass;

 use MyClass -self;

 # class tree now:
 #
 # MySubClass
 #     v
 #  MyClass
 #     v
 #  Clarity

 use MyExtraClass -mixin;

 # class tree now:
 #
 # MySubClass
 #     v
 # <generated class>
 #     v
 #  MyClass
 #     v
 #  Clarity

=head3 Caveats

Methods in the generated class will still follow their old
inheritance chain when they call super()/SUPER::somemethod().

Because of this, mixins work best when either the inheritance trees are
shallow or when there's a large overlap (Clarity deals with this overlap
properly).

If you try to mixin a class that is declared in the same file, be aware
of the order in which perl parses and processes things. In particular
accessor methods may not have been created yet. You may be able to get
around this by declaring accessors in a BEGIN {} block.

=head2 @EXPORT_BASE

Like @EXPORT and @EXPORT_OK but these methods/functions are exported to
derived classes (transitively).

=head2 carp/croak/confess/cluck

These functions from Carp are exported automatically using the @EXPORT_BASE
mechanism.

=head2 dispatching new()

The arguments to Clarity's new() are interpreted as a list of methods to
invoke on the new object. For example:

 my $o = new MyClass(foo => 3, -frob);

will result in Clarity invoking:

 $self->foo(3);
 $self->frob;

just after the object is created. If the method does not exist, the
corresponding field of the $self hash is set.

=head2 stubs

Using the stub() function you can create methods that die() when called
directly. You can supply an error message if you like.

 stub hook;
 stub callback => "No callback supplied";

     ...
     $self->hook;                       # will die() unless subclassed
     ...
     $self->callback;                   # will die() unless subclassed
                                        # (with a custom message)

=head1 DIFFERENCES WITH Spiffy.pm

=over

=item -

-Base is called -self

=item -

use <class> only accepts one flag at most (-base/-self/-mixin)

=item -

accessor creation methods do not accept flags

=item -

default value to an accessor can be a sub

=item -

const() does not share values between instances

=item -

accessor methods die() if used uninitialized and a default is not provided

=item -

accessor methods can use complex structures as default values

=item -

no XXX/YYY/ZZZ

=item -

no YAML

=item -

exports carp/croak/confess/cluck to base classes

=item -

arguments to new() are interpreted slightly differently

=item -

declaring multiple classes in the same file with -self works

=item -

automatic accessors

=item -

field/weakfield/const/stub do not need quotes around the name

=back

=head1 AUTHOR

Wessel Dankers <wsl(a)fruit.je>

=head1 COPYRIGHT

=over

=item Copyright (c) 2010-2012 Wessel Dankers <wsl@fruit.je>

=item Copyright (c) 2006 Ingy döt Net <ingy@cpan.org>

=item Copyright (c) 2004 Brian Ingerson

=back

=cut
