#include <Python.h>
#include <structmember.h>

#include <fcntl.h>
#include <boushi.h>

typedef struct {
	PyObject_HEAD
	uint64_t magic;
	boushi_t *b;
	bool initialized;
} Boushi;

#define BOUSHI_MAGIC UINT64_C(0xA464921EA699C0D9)

static bool Boushi_valid(const Boushi *b) {
	return b && b->magic == BOUSHI_MAGIC;
}


static void Boushi_dealloc(Boushi *self) {
	if(Boushi_valid(self)) {
		self->magic = 0;
		if(self->b) {
			boushi_exit(self->b);
			free(self->b);
		}
	}
	self->ob_type->tp_free((PyObject *)self);
}

static PyObject *Boushi_new(PyTypeObject *type, PyObject *args, PyObject *kwargs) {
	Boushi *self;

	self = (Boushi *)type->tp_alloc(type, 0);
	if(self) {
		self->magic = BOUSHI_MAGIC;
		self->b = NULL;
		self->initialized = false;
	}

	return (PyObject *)self;
}

static int Boushi_init(Boushi *self, PyObject *args, PyObject *kwargs) {
	char *s;

	if(!Boushi_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid Boushi object");
		return -1;
	}

	if(!args) {
		PyErr_SetString(PyExc_Exception, "Function args parameter is NULL");
		return -1;
	}

	if(!PyArg_ParseTuple(args, "s", &s))
		return -1;

	self->b = malloc(boushi_size());
	if(self->b) {
		if(boushi_init(self->b, s)) {
			self->initialized = true;
			return 0;
		}
		PyErr_SetString(PyExc_Exception, boushi_error(self->b));
		boushi_exit(self->b);
		free(self->b);
		self->b = NULL;
	}

	return -1;
}

static PyObject *Boushi_get(Boushi *self, PyObject *key) {
	void *buf;
	size_t len;
	PyObject *ret;

	if(!Boushi_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid Boushi object");
		return NULL;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized Boushi object");
		return NULL;
	}

	if(!key) {
		PyErr_SetString(PyExc_Exception, "Subscript parameter is NULL");
		return NULL;
	}

	if(!PyString_Check(key)) {
		PyErr_SetString(PyExc_TypeError, "Subscript is not a plain string");
		return NULL;
	}

	if(boushi_get(self->b, PyString_AsString(key), &buf, &len)) {
		if(buf) {
			ret = PyString_FromStringAndSize(buf, len);
			free(buf);
			return ret;
		}
		PyErr_SetObject(PyExc_KeyError, key);
		return NULL;
	}
	PyErr_SetString(PyExc_Exception, boushi_error(self->b));
	return NULL;
}

static int Boushi_put(Boushi *self, PyObject *key, PyObject *val) {
	bool found;

	if(!Boushi_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid Boushi object");
		return -1;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized Boushi object");
		return -1;
	}

	if(!key) {
		PyErr_SetString(PyExc_Exception, "Subscript parameter is NULL");
		return -1;
	}

	if(!PyString_Check(key)) {
		PyErr_SetString(PyExc_TypeError, "Subscript is not a plain string");
		return -1;
	}

	if(val) {
		if(!PyString_Check(val)) {
			PyErr_SetString(PyExc_TypeError, "Assigned value is not a plain string");
			return -1;
		}

		if(boushi_put(self->b, PyString_AsString(key), PyString_AsString(val), PyString_Size(val)))
			return 0;
	} else {
		if(boushi_del(self->b, PyString_AsString(key), &found)) {
			if(found)
				return 0;
			PyErr_SetObject(PyExc_KeyError, key);
			return -1;
		}
	}

	PyErr_SetString(PyExc_Exception, boushi_error(self->b));

	return -1;
}

static int Boushi_has(Boushi *self, PyObject *key) {
	bool found;

	if(!Boushi_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid Boushi object");
		return -1;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized Boushi object");
		return -1;
	}

	if(!key) {
		PyErr_SetString(PyExc_Exception, "Subscript parameter is NULL");
		return -1;
	}

	if(!PyString_Check(key)) {
		PyErr_SetString(PyExc_TypeError, "Substring is not a plain string");
		return -1;
	}

	if(boushi_has(self->b, PyString_AsString(key), &found))
		return found;

	PyErr_SetString(PyExc_Exception, boushi_error(self->b));

	return -1;
}

static PyTypeObject BoushiListingType;

static PyObject *Boushi_ls(Boushi *self, PyObject *args) {
	PyObject *s = NULL;
	PyObject *arguments;
	PyObject *result;

	if(!Boushi_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid Boushi object");
		return NULL;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized Boushi object");
		return NULL;
	}

	if(!args) {
		PyErr_SetString(PyExc_Exception, "Function args parameter is NULL");
		return NULL;
	}

	if(!PyArg_ParseTuple(args, "S", &s))
		return NULL;

	arguments = PyTuple_Pack(3, self, s, Py_False);
	if(!arguments)
		return NULL;

	result = PyObject_CallObject((PyObject *)&BoushiListingType, arguments);
	Py_DecRef(arguments);

	return result;
}

static PyObject *Boushi_find(Boushi *self, PyObject *args) {
	PyObject *s = NULL;
	PyObject *arguments;
	PyObject *result;

	if(!Boushi_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid Boushi object");
		return NULL;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized Boushi object");
		return NULL;
	}

	if(!args) {
		PyErr_SetString(PyExc_Exception, "Function args parameter is NULL");
		return NULL;
	}

	if(!PyArg_ParseTuple(args, "S", &s))
		return NULL;

	arguments = PyTuple_Pack(3, self, s, Py_True);
	if(!arguments)
		return NULL;

	result = PyObject_CallObject((PyObject *)&BoushiListingType, arguments);
	Py_DecRef(arguments);

	return result;
}

static PySequenceMethods BoushiType_as_sequence = {
	0,                         /*sq_length*/
	0,                         /*sq_concat*/
	0,                         /*sq_repeat*/
	0,                         /*sq_item*/
	0,                         /*sq_slice*/
	0,                         /*sq_ass_item*/
	0,                         /*sq_ass_slice*/
	(objobjproc)Boushi_has,    /*sq_contains*/
	0,                         /*sq_inplace_concat*/
	0                          /*sq_inplace_repeat*/
};

static PyMappingMethods BoushiType_as_mapping = {
	0,                         /*mp_length*/
	(binaryfunc)Boushi_get,    /*mp_subscript*/
	(objobjargproc)Boushi_put  /*mp_ass_subscript*/
};

static PyMethodDef Boushi_methods[] = {
	{"ls", (PyCFunction)Boushi_ls, METH_VARARGS, "Get a directory listing"},
	{"find", (PyCFunction)Boushi_find, METH_VARARGS, "Get a recursive listing"},
    {NULL}
};


static PyTypeObject BoushiType = {
	PyObject_HEAD_INIT(NULL)
	0,                         /*ob_size*/
	"boushi.Boushi",           /*tp_name*/
	sizeof(Boushi),            /*tp_basicsize*/
	0,                         /*tp_itemsize*/
	(destructor)Boushi_dealloc,  /*tp_dealloc*/
	0,                         /*tp_print*/
	0,                         /*tp_getattr*/
	0,                         /*tp_setattr*/
	0,                         /*tp_compare*/
	0,                         /*tp_repr*/
	0,                         /*tp_as_number*/
	&BoushiType_as_sequence,   /*tp_as_sequence*/
	&BoushiType_as_mapping,    /*tp_as_mapping*/
	0,                         /*tp_hash */
	0,                         /*tp_call*/
	0,                         /*tp_str*/
	0,                         /*tp_getattro*/
	0,                         /*tp_setattro*/
	0,                         /*tp_as_buffer*/
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
	"Boushi objects",          /* tp_doc */
	0,                         /* tp_traverse */
	0,                         /* tp_clear */
	0,                         /* tp_richcompare */
	0,                         /* tp_weaklistoffset */
	0,                         /* tp_iter */
	0,                         /* tp_iternext */
	Boushi_methods,            /* tp_methods */
	0,                         /* tp_members */
	0,                         /* tp_getset */
	0,                         /* tp_base */
	0,                         /* tp_dict */
	0,                         /* tp_descr_get */
	0,                         /* tp_descr_set */
	0,                         /* tp_dictoffset */
	(initproc)Boushi_init,     /* tp_init */
	0,                         /* tp_alloc */
	Boushi_new,                /* tp_new */
};

typedef struct {
	PyObject_HEAD
	uint64_t magic;
	Boushi *boushi;
	boushi_cursor_t *c;
	bool initialized;
} BoushiIterator;

#define BOUSHI_ITERATOR_MAGIC UINT64_C(0xBF82BE11022FFD27)

static bool BoushiIterator_valid(const BoushiIterator *b) {
	return b && b->magic == BOUSHI_ITERATOR_MAGIC;
}

static void BoushiIterator_dealloc(BoushiIterator *self) {
	if(BoushiIterator_valid(self)) {
		self->magic = 0;
		if(self->c) {
			boushi_cursor_exit(self->c);
			free(self->c);
		}
		if(self->boushi)
			Py_DecRef((PyObject *)self->boushi);
	}

	self->ob_type->tp_free((PyObject *)self);
}

static PyObject *BoushiIterator_new(PyTypeObject *type, PyObject *args, PyObject *kwargs) {
	BoushiIterator *self;

	self = (BoushiIterator *)type->tp_alloc(type, 0);
	if(self) {
		self->magic = BOUSHI_ITERATOR_MAGIC;
		self->boushi = NULL;
		self->c = NULL;
		self->initialized = false;
	}

	return (PyObject *)self;
}

static int BoushiIterator_init(BoushiIterator *self, PyObject *args, PyObject *kwargs) {
	Boushi *boushi;
	PyObject *p;
	PyObject *r = NULL;

	if(!BoushiIterator_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid BoushiIterator object");
		return -1;
	}

	if(!args) {
		PyErr_SetString(PyExc_Exception, "Function args parameter is NULL");
		return -1;
	}

	if(!PyArg_ParseTuple(args, "O!S|O", &BoushiType, &boushi, &p, &r))
		return -1;

	if(!Boushi_valid(boushi)) {
		PyErr_SetString(PyExc_TypeError, "BoushiIterator: first argument must be a valid Boushi object");
		return -1;
	}

	if(r && !PyBool_Check(r)) {
		PyErr_SetString(PyExc_TypeError, "BoushiIterator: third argument must be of boolean type");
		return -1;
	}

	Py_IncRef((PyObject *)boushi);
	self->boushi = boushi;

	self->c = malloc(boushi_cursor_size());
	if(self->c) {
		if(boushi_cursor_init(boushi->b, self->c, PyString_AsString(p), r != Py_False)) {
			self->initialized = true;
			return 0;
		}
		PyErr_SetString(PyExc_Exception, boushi_error(boushi->b));
	}

	return -1;
}

static PyObject *BoushiIterator_iter(BoushiIterator *self) {
	Py_IncRef((PyObject *)self);
	return (PyObject *)self;
}

static PyObject *BoushiIterator_iternext(BoushiIterator *self) {
	Boushi *boushi;
	char *s;
	PyObject *r;

	if(!BoushiIterator_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid BoushiIterator object");
		return NULL;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized BoushiIterator object");
		return NULL;
	}

	boushi = self->boushi;

	if(boushi_cursor_next(self->c, &s)) {
		if(!s)
			return NULL;
		r = PyString_FromString(s);
		return r;
	}
	PyErr_SetString(PyExc_Exception, boushi_error(boushi->b));
	return NULL;
}

static PyTypeObject BoushiIteratorType = {
	PyObject_HEAD_INIT(NULL)
	0,                         /*ob_size*/
	"boushi.BoushiIterator",   /*tp_name*/
	sizeof(BoushiIterator),    /*tp_basicsize*/
	0,                         /*tp_itemsize*/
	(destructor)BoushiIterator_dealloc,  /*tp_dealloc*/
	0,                         /*tp_print*/
	0,                         /*tp_getattr*/
	0,                         /*tp_setattr*/
	0,                         /*tp_compare*/
	0,                         /*tp_repr*/
	0,                         /*tp_as_number*/
	0,                         /*tp_as_sequence*/
	0,                         /*tp_as_mapping*/
	0,                         /*tp_hash */
	0,                         /*tp_call*/
	0,                         /*tp_str*/
	0,                         /*tp_getattro*/
	0,                         /*tp_setattro*/
	0,                         /*tp_as_buffer*/
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
	"Boushi iterator objects", /* tp_doc */
	0,		                   /* tp_traverse */
	0,		                   /* tp_clear */
	0,		                   /* tp_richcompare */
	0,		                   /* tp_weaklistoffset */
	(getiterfunc)BoushiIterator_iter, /* tp_iter */
	(iternextfunc)BoushiIterator_iternext, /* tp_iternext */
	0,                         /* tp_methods */
	0,                         /* tp_members */
	0,                         /* tp_getset */
	0,                         /* tp_base */
	0,                         /* tp_dict */
	0,                         /* tp_descr_get */
	0,                         /* tp_descr_set */
	0,                         /* tp_dictoffset */
	(initproc)BoushiIterator_init, /* tp_init */
	0,                         /* tp_alloc */
	BoushiIterator_new,        /* tp_new */
};

typedef struct {
	PyObject_HEAD
	PyObject *args;
	uint64_t magic;
	bool initialized;
} BoushiListing;

#define BOUSHI_LISTING_MAGIC UINT64_C(0x1B4B12DC9CD57695)

static bool BoushiListing_valid(const BoushiListing *b) {
	return b && b->magic == BOUSHI_LISTING_MAGIC;
}

static void BoushiListing_dealloc(BoushiListing *self) {
	if(BoushiListing_valid(self)) {
		if(self->args)
			Py_DecRef(self->args);
		self->magic = 0;
	}
	self->ob_type->tp_free((PyObject *)self);
}

static PyObject *BoushiListing_new(PyTypeObject *type, PyObject *args, PyObject *kwargs) {
	BoushiListing *self;

	self = (BoushiListing *)type->tp_alloc(type, 0);
	if(self) {
		self->magic = BOUSHI_LISTING_MAGIC;
		self->args = NULL;
		self->initialized = false;
	}

	return (PyObject *)self;
}

static int BoushiListing_init(BoushiListing *self, PyObject *args, PyObject *kwargs) {
	Boushi *boushi;
	PyObject *p;
	PyObject *r = NULL;

	if(!args) {
		PyErr_SetString(PyExc_Exception, "Function args parameter is NULL");
		return -1;
	}

	if(!PyArg_ParseTuple(args, "O!S|O", &BoushiType, &boushi, &p, &r))
		return -1;

	if(!Boushi_valid(boushi)) {
		PyErr_SetString(PyExc_TypeError, "BoushiListing: first argument must be a valid Boushi object");
		return -1;
	}

	if(r && !PyBool_Check(r)) {
		PyErr_SetString(PyExc_TypeError, "BoushiListing: third argument must be of boolean type");
		return -1;
	}

	Py_IncRef(args);
	self->args = args;
	self->initialized = true;

	return 0;
}

static PyObject *BoushiListing_iter(BoushiListing *self) {
	if(!BoushiListing_valid(self)) {
		PyErr_SetString(PyExc_Exception, "Not a valid BoushiListing object");
		return NULL;
	}

	if(!self->initialized) {
		PyErr_SetString(PyExc_Exception, "Uninitialized BoushiListing object");
		return NULL;
	}

	return PyObject_CallObject((PyObject *)&BoushiIteratorType, self->args);
}

static PyTypeObject BoushiListingType = {
	PyObject_HEAD_INIT(NULL)
	0,                         /*ob_size*/
	"boushi.BoushiListing",    /*tp_name*/
	sizeof(BoushiListing),     /*tp_basicsize*/
	0,                         /*tp_itemsize*/
	(destructor)BoushiListing_dealloc,  /*tp_dealloc*/
	0,                         /*tp_print*/
	0,                         /*tp_getattr*/
	0,                         /*tp_setattr*/
	0,                         /*tp_compare*/
	0,                         /*tp_repr*/
	0,                         /*tp_as_number*/
	0,                         /*tp_as_sequence*/
	0,                         /*tp_as_mapping*/
	0,                         /*tp_hash */
	0,                         /*tp_call*/
	0,                         /*tp_str*/
	0,                         /*tp_getattro*/
	0,                         /*tp_setattro*/
	0,                         /*tp_as_buffer*/
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
	"Boushi listing objects",  /* tp_doc */
	0,		                   /* tp_traverse */
	0,		                   /* tp_clear */
	0,		                   /* tp_richcompare */
	0,		                   /* tp_weaklistoffset */
	(getiterfunc)BoushiListing_iter, /* tp_iter */
	0,                         /* tp_iternext */
	0,                         /* tp_methods */
	0,                         /* tp_members */
	0,                         /* tp_getset */
	0,                         /* tp_base */
	0,                         /* tp_dict */
	0,                         /* tp_descr_get */
	0,                         /* tp_descr_set */
	0,                         /* tp_dictoffset */
	(initproc)BoushiListing_init, /* tp_init */
	0,                         /* tp_alloc */
	BoushiListing_new,         /* tp_new */
};

static PyMethodDef module_methods[] = {
	{NULL}  /* Sentinel */
};

/* declarations for DLL import/export */
#ifndef PyMODINIT_FUNC
#define PyMODINIT_FUNC void
#endif
PyMODINIT_FUNC initboushi(void) {
	PyObject* m;

	if(PyType_Ready(&BoushiType) < 0)
		return;
	if(PyType_Ready(&BoushiIteratorType) < 0)
		return;
	if(PyType_Ready(&BoushiListingType) < 0)
		return;

	m = Py_InitModule3("boushi", module_methods,
	   "Frontend to libboushi handler");

	if(!m)
	  return;

	Py_INCREF(&BoushiType);
	Py_INCREF(&BoushiIteratorType);
	Py_INCREF(&BoushiListingType);

	PyModule_AddObject(m, "Boushi", (PyObject *)&BoushiType);
	PyModule_AddObject(m, "BoushiIterator", (PyObject *)&BoushiIteratorType);
	PyModule_AddObject(m, "BoushiListing", (PyObject *)&BoushiListingType);
}
