/*****************************************************************************

	tar.c - source code for the boushi command line tool

	Copyright (c) 2010  Wessel Dankers <wsl@fruit.je>

	This file is part of boushi.

	boushi is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as
	published by the Free Software Foundation, either version 3 of
	the License, or (at your option) any later version.

	boushi is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU General Public License
	and a copy of the GNU Lesser General Public License along with
	boushi.  If not, see <http://www.gnu.org/licenses/>.

*****************************************************************************/

#include "config.h"
#include "io.h"
#include "tar.h"

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/mman.h>

struct tar {
	char name[100];
	char mode[8];
	char uid[8];
	char gid[8];
	char size[12];
	char mtime[12];
	char checksum[8];
	char typeflag;
	char linkname[100];
	char magic[6];
	char version[2];
	char uname[32];
	char gname[32];
	char devmajor[8];
	char devminor[8];
	char prefix[155];
	char blank[12];
};

static const struct tar tar0 = {{0}};

static unsigned int sum(const uint8_t *src) {
	unsigned int checksum = 8 * ' ';
	size_t len = 512;

	while(len--) {
		if(len < 356 || len > 363)
			checksum += *src;
		src++;
	}

	return checksum;
}

static bool tar_pack(void *buf, const char *name, size_t size, char type) {
	struct tar *tar = buf;

	*tar = tar0;

	memcpy(tar->magic, "ustar", 6);
	strncpy(tar->name, name, sizeof tar->name);
	strcpy(tar->uid, "0000000");
	strcpy(tar->gid, "0000000");
	strcpy(tar->uname, "root");
	strcpy(tar->gname, "root");
	sprintf(tar->mtime, "%011lo", (long int)time(NULL));
	tar->typeflag = type;
	sprintf(tar->size, "%011lo", (long int)size);
	strcpy(tar->mode, "644");

	sprintf(tar->checksum, "%07o", sum((uint8_t *)tar));

	return true;
}

static bool tar_unpack(const void *buf, char *name, char *linkname, size_t *len, char *type) {
	char str[13];
	const struct tar *tar = buf;

	memcpy(str, tar->checksum, sizeof tar->checksum);
	str[12] = '\0';
	if(sum(buf) != strtoul(str, NULL, 8))
		return false;
	memcpy(name, tar->name, sizeof tar->name);
	name[sizeof tar->name] = '\0';
	memcpy(linkname, tar->linkname, sizeof tar->linkname);
	linkname[sizeof tar->linkname] = '\0';
	memcpy(str, tar->size, sizeof tar->size);
	*len = strtoull(str, NULL, 8);
	*type = tar->typeflag;

	return true;
}

bool tar_write(int fd, const char *name, const void *data, size_t size) {
	char buf[512];
	size_t r;
	size_t len;

	len = strlen(name);

	if(len > 100) {
		if(!tar_pack(buf, "././@LongLink", len, 'L'))
			return false;

		if(!safewrite(fd, buf, sizeof buf))
			return false;

		r = len % sizeof buf;

		if(!safewrite(fd, name, len - r))
			return false;

		if(r) {
			memcpy(buf, name + len - r, r);
			memset(buf + r, 0, sizeof buf - r);
			if(!safewrite(fd, buf, sizeof buf))
				return false;
		}
	}

	if(!tar_pack(buf, name, size, '0'))
		return false;

	if(!safewrite(fd, buf, sizeof buf))
		return false;

	r = size % sizeof buf;

	if(!safewrite(fd, data, size - r))
		return false;

	if(r) {
		memcpy(buf, data + size - r, r);
		memset(buf + r, 0, sizeof buf - r);
		if(!safewrite(fd, buf, sizeof buf))
			return false;
	}
	return true;
}

bool tar_end(int fd) {
	char buf[1024];

	memset(buf, 0, sizeof buf);

	return safewrite(fd, buf, sizeof buf);
}

static bool allzeroes(const char *buf, size_t len) {
	while(len--)
		if(*buf++)
			return false;
	return true;
}

static bool tar_read_io(int fd, tar_func_t func, void *data) {
	return false;
}

static char *strmemdup(const void *buf, size_t len) {
	char *s;
	s = malloc(len + 1);
	if(!s)
		return NULL;
	memcpy(s, buf, len);
	s[len] = '\0';
	return s;
}

bool tar_read(int fd, tar_func_t func, void *data) {
	char filename[1024];
	char *file = filename;
	char *filebuf = NULL;
	char linkname[1024];
	char *link = linkname;
	char *linkbuf = NULL;
	struct stat st;
	char *mem, *cur, *end;
	size_t size, r;
	char type;
	bool zero = false;
	bool peachy = true;

	if(fstat(fd, &st) == -1)
		return tar_read_io(fd, func, data);

	if(!st.st_size || st.st_size % 512)
		return tar_read_io(fd, func, data);

	mem = mmap(NULL, st.st_size, PROT_READ, MAP_SHARED, fd, 0);
	if(mem == MAP_FAILED)
		return tar_read_io(fd, func, data);

	cur = mem;
	end = mem + st.st_size;
	while(peachy) {
		if(allzeroes(cur, 512)) {
			if(zero)
				break;
			zero = true;
			file = filename;
			filename[0] = '\0';
			type = 'Z';
			size = 0;
		} else {
			zero = false;
			if(!tar_unpack(cur, filename, linkname, &size, &type) || (ssize_t)size < 0) {
				fprintf(stderr, "Bad tar header\n");
				peachy = false;
				break;
			}
		}
		cur += 512;
		if(cur > end) {
			fprintf(stderr, "Premature end of input\n");
			peachy = false;
			break;
		}
		if(size > (size_t)st.st_size || cur + size > end) {
			fprintf(stderr, "Premature end of input\n");
			peachy = false;
			break;
		}
		r = size % 512;
		if(type == '\0' || type == '0') {
			if(!func(data, file, size ? cur : NULL, size))
				return false;
			link = linkname;
			file = filename;
		} else if(type == '1' || type == 1) {
			if(!func(data, file, link, 0))
				return false;
			link = linkname;
			file = filename;
		} else if(type == 'K') {
			if((cur + size < end && !cur[size]) || memrchr(cur, '\0', size)) {
				link = cur;
			} else if(size < sizeof linkname) {
				memcpy(linkname, cur, size);
				linkname[size] = '\0';
				link = linkname;
			} else {
				if(linkbuf)
					free(linkbuf);
				linkbuf = strmemdup(cur, size);
				if(!linkbuf) {
					perror("malloc()");
					peachy = false;
					break;
				}
			}
		} else if(type == 'L') {
			if((cur + size < end && !cur[size]) || memrchr(cur, '\0', size)) {
				file = cur;
			} else if(size < sizeof filename) {
				memcpy(filename, cur, size);
				filename[size] = '\0';
				file = filename;
			} else {
				if(filebuf)
					free(filebuf);
				filebuf = strmemdup(cur, size);
				if(!filebuf) {
					perror("malloc()");
					peachy = false;
					break;
				}
			}
		}
		cur += size;
		if(r)
			cur += 512 - r;
	}

	munmap(mem, st.st_size);

	if(linkbuf)
		free(linkbuf);

	if(filebuf)
		free(filebuf);

	return true;
}
