/*****************************************************************************

	libboushi.c - source code for libboushi

	Copyright (c) 2009-2010  Wessel Dankers <wsl@fruit.je>

	This file is part of boushi.

	boushi is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as
	published by the Free Software Foundation, either version 3 of
	the License, or (at your option) any later version.

	boushi is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU General Public License
	and a copy of the GNU Lesser General Public License along with
	boushi.  If not, see <http://www.gnu.org/licenses/>.

*****************************************************************************/

#include "config.h"
#include "boushi.h"
#include "sha1.h"
#include "io.h"

#include <stdbool.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <db.h>
#include <lzo/lzo1x.h>
#include <arpa/inet.h>

#define export __attribute__((visibility("default")))

static const char db_config_defaults[] =
	"set_flags DB_LOG_AUTOREMOVE\n"
	"#set_cachesize 1 0 0\n"
	"#set_lg_bsize 33554432\n"
	"#set_lg_max 134217728\n"
	"#set_lg_dir /some/reliable/disk1\n"
	"#set_data_dir /some/reliable/disk2\n";

struct boushi {
	uint64_t magic;
	DB_ENV *env;
	DB *content;
	DB *files;
	DB *directories;
	char *dbdir;
	char *error;
	char *lzowork;
	int err;
	bool initialized;
};

static const boushi_t boushi_0 = {UINT64_C(0x61D97141BCFDAE79), 0};

size_t boushi_size(void) export;
size_t boushi_size(void) {
	return sizeof boushi_0;
}

static bool boushi_valid(const boushi_t *b) {
	return b && b->magic == boushi_0.magic;
}

static const DBT DBT0 = {0};

static const char *boushi_enomem = "Out of memory";

/* These macros assume that b is a valid boushi_t */
#define ok(...) boushi_ok(b, __VA_ARGS__)
/* For logic/internal errors that don't have an errno: */
#define bok(...) boushi_ok(b, -1, __VA_ARGS__)

/* Store an error message (and code) for later retrieval.

	Does nothing and returns true if err==0
	Return false if err!=0
	Appends the error message if err != -1
*/
static bool boushi_ok(boushi_t *b, int err, const char *fmt, ...)
	__attribute__((format(printf, 3, 4)));
static bool boushi_ok(boushi_t *b, int err, const char *fmt, ...) {
	va_list ap;
	char *msg, *errstr = NULL;
	int len;

	if(!b || !fmt)
		return !err;

	b->err = err;
	if(b->error) {
		if(b->error != boushi_enomem)
			free(b->error);
		b->error = NULL;
	}

	if(!err)
		return true;

	va_start(ap, fmt);
	len = vsnprintf(NULL, 0, fmt, ap);
	va_end(ap);

	if(len < 0)
		return false;

	if(err != -1) {
		errstr = db_strerror(err);
		if(!errstr)
			return false;
		len += strlen(errstr);
	}
		
	msg = malloc(len + 10);
	if(!msg) {
		b->error = (char *)boushi_enomem;
		return false;
	}
	b->error = msg;

	va_start(ap, fmt);
	len = vsnprintf(msg, len, fmt, ap);
	va_end(ap);

	if(len < 0) {
		strcpy(msg, "FAIL");
		return false;
	}

	if(errstr) {
		if(len && !isspace(msg[len-1])) {
			memcpy(msg + len, ": ", 2);
			len += 2;
		}
		strcpy(msg + len, errstr);
	} else {
		msg[len] = '\0';
	}

	return false;
}

/* retrieve the error code, if any */
int boushi_errno(struct boushi *b) export;
int boushi_errno(struct boushi *b) {
	return b ? b->err : EFAULT;
}

/* return an error string, if any (never returns NULL) */
const char *boushi_error(struct boushi *b) export;
const char *boushi_error(struct boushi *b) {
	return b ? b->error ? b->error : "" : "boushi object is NULL";
}

/* normalize a path:

	- remove repeated slashes
	- remove leading/trailing slashes
	- remove occurrences of .
	- resolve occurrences of ..

	should be able to deal with src==dst (i.e., modifying the
	string in-place) but I haven't tested this
*/
static size_t file2path(char *dst, const char *src) {
	char *cur, *end;
	const char *sep;
	size_t len;

#if 0
	/* enable this to preserve absolute paths */
	if(*src == '/')
		*dst++ = *src++;
#endif

	cur = dst;
	do {
		/* strchrnul() */
		sep = strchr(src, '/');
		if(!sep)
			sep = src + strlen(src);
		len = sep - src;
		if(!len) {
			// do nothing
		} else if(len == 1 && *src == '.') {
			// do nothing
		} else if(len == 2 && src[0] == '.' && src[1] == '.') {
			end = memrchr(dst, '/', cur - dst);
			if(end)
				cur = end;
		} else {
			if(cur > dst)
				*cur++ = '/';
			memmove(cur, src, len);
			cur += len;
		}
		src = sep + 1;
	} while(*sep);

	*cur = '\0';

	return cur - dst;
}

/* compare two paths:

	- paths are sorted by the number of slashes
	- if that number is equal, do a standard string comparison
*/
static int pathcmp(DB *db, const DBT *a, const DBT *b) {
	const char *ap, *bp, *ae, *be;
	size_t al, bl;
	int c;

	al = a->size;
	bl = b->size;

	if(!al)
		return bl ? -1 : 0;
	if(!bl)
		return 1;

	ap = a->data;
	bp = b->data;
	ae = ap + al;
	be = bp + bl;

	for(;;) {
		ap = memchr(ap, '/', ae - ap);
		bp = memchr(bp, '/', be - bp);
		if(ap) {
			if(!bp)
				return 1;
		} else {
			if(bp)
				return -1;
			break;
		}
		ap++;
		bp++;
	}

	c = memcmp(a->data, b->data, al < bl ? al : bl);
	if(c)
		return c;
	return al < bl ? -1 : al > bl ? 1 : 0;
}

/* the number of bytes used to store the original length in the database */
#define SIZEBYTES 4
/* the resultant maximum original length */
#define MAXSIZE ((size_t)1 << (SIZEBYTES << 3))

/* read a SIZEBYTES integer without any specific alignment requirement */
static uint32_t readsize(const uint8_t *buf) {
	return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}

/* write a SIZEBYTES integer without any specific alignment requirement */
static void writesize(uint8_t *buf, size_t size) {
	buf[0] = size >> 24;
	buf[1] = (size >> 16) & 255;
	buf[2] = (size >> 8) & 255;
	buf[3] = size & 255;
}

/* prepare a (possibly compressed) database record

	memory is allocated using malloc()
	caller must free this memory when done with it
	pointer to this malloc()ed buffer is stored in outp
	size of malloc()ed buffer is stored in outlenp
*/
static bool lzo_pack(boushi_t *b, const uint8_t *in, size_t inlen, uint8_t **outp, size_t *outlenp) {
	lzo_uint lzolen = 0;
	uint8_t *out = NULL;
	static char *work = NULL;
	bool peachy = true;

	if(!inlen) {
		*outp = NULL;
		*outlenp = 0;
		return peachy;
	}

	if(inlen <= MAXSIZE) {
		work = b->lzowork;
		if(!work) {
			work = malloc(LZO1X_999_MEM_COMPRESS);
			if(work) {
				b->lzowork = work;
				memset(work, 0, LZO1X_999_MEM_COMPRESS);
			} else {
				peachy = ok(errno, "malloc(%"PRId64")", (int64_t)LZO1X_999_MEM_COMPRESS);
			}
		}
		if(peachy) {
			out = malloc(inlen + inlen / 16 + 64 + 3 + SIZEBYTES);
			if(!out)
				peachy = ok(errno, "malloc(%"PRId64")", (int64_t)(inlen + inlen / 16 + 64 + 3 + SIZEBYTES));
		}
		if(peachy) {
			if(lzo1x_999_compress(in, inlen, out, &lzolen, work) != LZO_E_OK)
				peachy = bok("lzo1x_999_compress()");
		}
		if(peachy && lzolen < inlen) {
			writesize(out + lzolen, inlen);
			*outp = out;
			*outlenp = lzolen + SIZEBYTES;
			return peachy;
		}
	}

	if(peachy && !out) {
		out = malloc(inlen + SIZEBYTES);
		if(!out)
			peachy = ok(errno, "malloc(%"PRId64")", (int64_t)(inlen + SIZEBYTES));
	}

	if(peachy) {
		writesize(out + inlen, 0);
		memcpy(out, in, inlen);

		*outp = out;
		*outlenp = inlen + SIZEBYTES;
		return peachy;
	}

	if(out)
		free(out);

	return peachy;
}

/* extract data from a database record

	returns a pointer to the record in out
	returns the size of the record in outsize
	if the returned pointer is not equal to buf it is malloc()ed
		and must be free()d by the caller
	input buffer may be modified
	record is guaranteed to be \0-terminated (\0 not included in size)
*/
static bool lzo_unpack(boushi_t *b, uint8_t *buf, size_t size, uint8_t **out, size_t *outsize) {
	bool peachy = true;
	uint8_t *lzo = NULL;
	uint32_t lzosize = 0;
	lzo_uint reslen = 0;

	if(!out || !outsize)
		return ok(EINVAL, "internal error at %s:%d", __FILE__, __LINE__);

	if(!size) {
		lzo = malloc(1);
		if(!lzo)
			return ok(errno, "malloc(1)");
		*lzo = '\0';
		*out = lzo;
		*outsize = 0;
		return peachy;
	}

	if(!buf)
		return ok(EINVAL, "internal error at %s:%d", __FILE__, __LINE__);
	if(size < SIZEBYTES)
		return ok(EINVAL, "internal error at %s:%d", __FILE__, __LINE__);

	size -= SIZEBYTES;

	lzosize = readsize(buf + size);
	if(!lzosize) {
		buf[size] = '\0';
		*out = buf;
		*outsize = size;
		return true;
	}
	lzo = malloc(lzosize + 1);
	if(!lzo)
		return ok(errno, "malloc(%"PRId64")", (int64_t)(lzosize + 1));

	if(lzo1x_decompress(buf, size, lzo, &reslen, NULL) != LZO_E_OK)
		peachy = false;
	if(reslen != lzosize)
		peachy = false;

	if(peachy) {
		lzo[lzosize] = '\0';
		*out = lzo;
		*outsize = lzosize;
	} else if(lzo) {
		free(lzo);
	}

	return peachy;
}

/* free memory associated with a boushi object, but not the struct itself */
bool boushi_exit(boushi_t *b) export;
bool boushi_exit(boushi_t *b) {
	if(!boushi_valid(b))
		return false;

	if(b->directories)
		b->directories->close(b->directories, 0);

	if(b->files)
		b->files->close(b->files, 0);

	if(b->content)
		b->content->close(b->content, 0);

	if(b->env)
		b->env->close(b->env, 0);

	if(b->error && b->error != boushi_enomem)
		free(b->error);

	if(b->dbdir)
		free(b->dbdir);

	if(b->lzowork)
		free(b->lzowork);

	*b = boushi_0;

	return true;
}

/* write a default DB_CONFIG file */
static bool boushi_write_db_config(boushi_t *b) {
	int fd = -1, dirfd = -1;
	ssize_t size = sizeof db_config_defaults - 1;
	bool peachy = true;

	if(peachy) {
		dirfd = open(b->dbdir, O_RDONLY|O_NOCTTY|O_DIRECTORY);
		if(dirfd == -1)
			peachy = ok(errno, "open(%s)", b->dbdir);
	}

	if(peachy) {
		fd = openat(dirfd, "DB_CONFIG", O_CREAT|O_EXCL|O_WRONLY|O_NOCTTY, 0666);
		if(fd == -1 && errno != EEXIST)
			peachy = ok(errno, "openat(%s, %s)", b->dbdir, "DB_CONFIG");
	}

	if(dirfd != -1) {
		if(close(dirfd) == -1)
			peachy = ok(errno, "close(%s)", b->dbdir);
	}

	/* may be -1 even if peachy (in case of EEXIST) */
	if(fd != -1) {
		if(peachy) {
			if(!safewrite(fd, db_config_defaults, size))
				peachy = ok(errno, "write(%s, ...)", "DB_CONFIG");
		}
		if(peachy) {
			if(fdatasync(fd) == -1)
				peachy = ok(errno, "fdatasync(%s, ...)", "DB_CONFIG");
		}

		if(close(fd) == -1)
			peachy = ok(errno, "close(%s)", "DB_CONFIG");
	}

	return peachy;
}

/* return the first 4 bytes (or fewer if length is too small) as an uint32_t (big-endian) */
static uint32_t boushi_hash(DB *db, const void *bytes, uint32_t length) {
	switch(length) {
		case 0: return 0;
		case 1: return ((uint8_t *)bytes)[0];
		case 2: return ((uint8_t *)bytes)[0] << 8 | ((uint8_t *)bytes)[1];
		case 3: return ((uint8_t *)bytes)[0] << 16 | ((uint8_t *)bytes)[1] << 8 | ((uint8_t *)bytes)[2];
		default: return ((uint8_t *)bytes)[0] << 24 | ((uint8_t *)bytes)[1] << 16 | ((uint8_t *)bytes)[2] << 8 | ((uint8_t *)bytes)[3];
	}
}

/* do a checkpoint 4 times per log file */
static bool boushi_checkpoint(boushi_t *b) {
	uint32_t x;
	return ok(b->env->get_lg_max(b->env, &x), "env->get_lg_max()")
		&& ok(b->env->txn_checkpoint(b->env, x/4096, 0, 0), "env->txn_checkpoint(%"PRIu32"k, 0, 0)", x/4096);
}

/* initialize a boushi struct

	you must call boushi_exit() even if this function fails
		or you will leak memory
*/
bool boushi_init(boushi_t *b, const char *dbdir) export;
bool boushi_init(boushi_t *b, const char *dbdir) {
	DB_TXN *txn = NULL;
	bool peachy = true;

	if(!b)
		return false;

	*b = boushi_0;

	if(!dbdir)
		return bok("boushi_init(b, dbdir): dbdir is NULL");

	if(peachy) {
		b->dbdir = strdup(dbdir);
		peachy = b->dbdir || ok(errno, "malloc(%"PRId64")", (int64_t)strlen(dbdir));
	}

	if(mkdir(dbdir, 0777) == -1)
		peachy = errno == EEXIST || ok(errno, "mkdir(%s)", dbdir);
	else
		peachy = boushi_write_db_config(b);

	if(peachy)
		peachy = ok(db_env_create(&b->env, 0), "db_env_create()");

	if(peachy)
		peachy = ok(b->env->open(b->env, dbdir, DB_CREATE|DB_RECOVER|DB_REGISTER|DB_THREAD|DB_INIT_LOCK|DB_INIT_LOG|DB_INIT_MPOOL|DB_INIT_TXN|DB_USE_ENVIRON, 0666), "env->open(%s)", dbdir);

	if(peachy)
		peachy = ok(db_create(&b->content, b->env, 0), "db_create()");

	if(peachy)
		peachy = ok(b->content->set_h_hash(b->content, boushi_hash), "content->set_h_hash()");

	if(peachy)
		peachy = ok(db_create(&b->files, b->env, 0), "db_create()");

	if(peachy)
		peachy = ok(db_create(&b->directories, b->env, 0), "db_create()");

	if(peachy)
		peachy = ok(b->directories->set_bt_compare(b->directories, pathcmp), "directories->set_bt_compare()");

	if(peachy)
		peachy = ok(b->env->txn_begin(b->env, NULL, &txn, 0), "env->txn_begin()");

	if(peachy)
		peachy = ok(b->content->open(b->content, txn, "content.hash", NULL, DB_HASH, DB_CREATE, 0666), "content->open(content.hash)");

	if(peachy)
		peachy = ok(b->files->open(b->files, txn, "directory.btree", "files", DB_BTREE, DB_CREATE, 0666), "files->open(directory.btree, files)");

	if(peachy)
		peachy = ok(b->directories->open(b->directories, txn, "directory.btree", "directories", DB_BTREE, DB_CREATE, 0666), "directories->open(directory.btree, directories)");

	if(peachy)
		peachy = ok(txn->commit(txn, 0), "txn->commit()");
	else if(txn)
		txn->abort(txn);

	if(peachy)
		peachy = boushi_checkpoint(b);

	if(peachy)
		b->initialized = true;

	return peachy;
}

/* calculate the sha1 hash of a memory buffer and store it in r */
static void sha1(const char *s, size_t len, uint8_t *r) {
	struct sha1_ctxt ctx;

	if(!r)
		return;

	if(s) {
		sha1_init(&ctx);
		sha1_loop(&ctx, (const uint8_t *)s, len);
		sha1_result(&ctx, r);
	} else {
		memset(r, '\0', SHA1_RESULTLEN);
	}
}

/* put a record in the database */
bool boushi_put(boushi_t *b, const char *name, const void *buf, size_t len) export;
bool boushi_put(boushi_t *b, const char *name, const void *buf, size_t len) {
	char *path = NULL;
	size_t pathlen = 0;
	uint8_t *out = NULL;
	size_t outlen = 0;
	DB_TXN *txn = NULL;
	DBT key = DBT0, val = DBT0;
	bool peachy = true;
	uint8_t sha1sum[SHA1_RESULTLEN];
	uint32_t refcount;
	bool found = false;
	bool isfile = true;
	int err;
	char *s;

	if(!boushi_valid(b))
		return false;

	if(!b->initialized)
		return bok("boushi_put(boushi, name, buf, len): boushi object not initialized");

	if(!name)
		return bok("boushi_put(boushi, name, buf, len): name parameter is NULL");

	if(!buf && len)
		return bok("boushi_put(boushi, %s, buf, %"PRIu64"): buf parameter is NULL", name, (uint64_t)len);

	path = alloca(strlen(name) + 1);
	pathlen = file2path(path, name);

	sha1(path, pathlen, sha1sum);

	peachy = lzo_pack(b, buf, len, &out, &outlen);

	if(peachy && outlen > UINT32_MAX)
		peachy = bok("boushi_put(boushi, %s, buf, %"PRIu64"): buffer too large (%"PRIu64" bytes even after compression, max is 2^32)", name, (uint64_t)len, (uint64_t)outlen);

	if(peachy)
		peachy = ok(b->env->txn_begin(b->env, NULL, &txn, 0), "env->txn_begin()");

	if(peachy) {
		key.data = sha1sum;
		key.ulen =
		key.size = sizeof sha1sum;
		val.data = out;
		val.ulen =
		val.size = outlen;
		peachy = ok(b->content->put(b->content, txn, &key, &val, 0), "content->put(%s)", name);
	}

	if(out)
		free(out);

	if(peachy) {
		key.data = path;
		key.ulen =
		key.size = pathlen + 1;
		val.data = NULL;
		val.ulen =
		val.size = 0;
		peachy = ok(b->files->put(b->files, txn, &key, &val, 0), "content->put(%s)", name);
	}

	/* traverse down the tree, inserting an element for each path component:
		foo/bar/baz
		foo/bar
		foo

		unless we encounter an existing node, in which case we just adjust the refcount
	*/
	while(peachy) {
		key.data = path;
		key.ulen =
		key.size = pathlen + 1;
		val.data = &refcount;
		val.ulen = sizeof refcount;
		val.flags = DB_DBT_USERMEM;
		err = b->directories->get(b->directories, txn, &key, &val, DB_RMW);
		if(err) {
			if(err == DB_NOTFOUND) {
				refcount = 0;
			} else {
				peachy = ok(err, "directories->get(%s)", path);
				break;
			}
		} else {
			if(val.size != sizeof refcount) {
				bok("boushi_get(%s): database corrupted (needs dump+load)", path);
				break;
			}
			/* we found an existing subdirectory, so we can stop iterating real soon now */
			found = true;
			refcount = ntohl(refcount);
		}

		if(isfile)
			refcount |= UINT32_C(0x80000000);
		else
			refcount++;

		refcount = htonl(refcount);

		val.data = &refcount;
		val.ulen =
		val.size = sizeof refcount;
		peachy = ok(b->directories->put(b->directories, txn, &key, &val, 0), "directories->put(%s)", path);
		if(!peachy)
			break;

		/* we found an existing subdirectory, so we can stop iterating now */
		if(found)
			break;

		s = memrchr(path, '/', pathlen);
		if(!s)
			break;
		*s = '\0';
		pathlen = s - path;

		/* only the first node we insert refers to a file */
		isfile = false;
	}

	if(peachy)
		peachy = ok(txn->commit(txn, 0), "txn->commit()");
	else if(txn)
		txn->abort(txn);

	if(peachy)
		peachy = boushi_checkpoint(b);

	return peachy;
}

/* try to delete a record and report if it was found */
bool boushi_del(boushi_t *b, const char *name, bool *isfound) export;
bool boushi_del(boushi_t *b, const char *name, bool *isfound) {
	char *path = NULL;
	size_t pathlen = 0;
	DB_TXN *txn = NULL;
	DBT key = DBT0, val = DBT0;
	bool peachy = true;
	uint8_t sha1sum[SHA1_RESULTLEN];
	uint32_t refcount;
	bool found = true, isfile = true;
	int err;
	char *s;

	if(!boushi_valid(b))
		return false;

	if(!b->initialized)
		return bok("boushi_del(boushi, name, isfound): boushi object not initialized");

	if(!name)
		return bok("boushi_del(boushi, name, isfound): name parameter is NULL");

	path = alloca(strlen(name) + 1);
	pathlen = file2path(path, name);

	sha1(path, pathlen, sha1sum);

	peachy = ok(b->env->txn_begin(b->env, NULL, &txn, 0), "env->txn_begin()");

	if(peachy) {
		key.data = sha1sum;
		key.ulen =
		key.size = sizeof sha1sum;
		err = b->content->del(b->content, txn, &key, 0);
		if(err) {
			if(err == DB_NOTFOUND)
				found = false;
			else
				peachy = ok(err, "content->del(%s)", name);
		}
	}

	if(found) {
		if(peachy) {
			key.data = path;
			key.ulen =
			key.size = pathlen + 1;
			peachy = ok(b->files->del(b->files, txn, &key, 0), "files->del(%s)", name);
		}

		/* traverse down the tree, inserting an element for each path component:
			foo/bar/baz
			foo/bar
			foo

			unless we encounter an existing node, in which case we just adjust the refcount
		*/
		for(;;) {
			key.data = path;
			key.ulen =
			key.size = pathlen + 1;
			val.data = &refcount;
			val.ulen = sizeof refcount;
			val.flags = DB_DBT_USERMEM;
			peachy = ok(b->directories->get(b->directories, txn, &key, &val, DB_RMW), "directories->get(%s)", path);
			if(!peachy)
				break;

			if(val.size != sizeof refcount) {
				bok("boushi_get(%s): database corrupted (needs dump+load)", path);
				break;
			}

			refcount = ntohl(refcount);
			if(isfile) {
				if(!(refcount & UINT32_C(0x80000000))) {
					bok("boushi_get(%s): database corrupted (needs dump+load)", path);
					break;
				}
				refcount &= ~UINT32_C(0x80000000);
			} else {
				if(!refcount) {
					bok("boushi_get(%s): database corrupted (needs dump+load)", path);
					break;
				}
				refcount--;
			}
			refcount = htonl(refcount);

			if(refcount) {
				val.data = &refcount;
				val.ulen =
				val.size = sizeof refcount;
				peachy = ok(b->directories->put(b->directories, txn, &key, &val, 0), "directories->put(%s)", path);
				/* we're not deleting this node, so we can stop going down the tree */
				break;
			} else {
				peachy = ok(b->directories->del(b->directories, txn, &key, 0), "directories->del(%s)", path);
				if(!peachy)
					break;
			}

			s = memrchr(path, '/', pathlen);
			if(!s)
				break;
			*s = '\0';
			pathlen = s - path;

			/* only the first node we delete refers to a file */
			isfile = false;
		}
	}

	if(peachy)
		peachy = ok(txn->commit(txn, 0), "txn->commit()");
	else if(txn)
		txn->abort(txn);

	if(peachy && isfound)
		*isfound = found;

	if(peachy)
		peachy = boushi_checkpoint(b);

	return peachy;
}

/* see if a record is in the database */
bool boushi_has(boushi_t *b, const char *name, bool *isfound) export;
bool boushi_has(boushi_t *b, const char *name, bool *isfound) {
	char *path = NULL;
	size_t pathlen = 0;
	DBT key = DBT0;
	uint8_t sha1sum[SHA1_RESULTLEN];
	int err;

	if(!boushi_valid(b))
		return false;

	if(!b->initialized)
		return bok("boushi_has(boushi, name, isfound): boushi object not initialized");

	if(!name)
		return bok("boushi_has(boushi, name, isfound): name parameter is NULL");

	path = alloca(strlen(name) + 1);
	pathlen = file2path(path, name);

	sha1(path, pathlen, sha1sum);

	key.data = sha1sum;
	key.ulen =
	key.size = sizeof sha1sum;
	err = b->content->exists(b->content, NULL, &key, 0);
	switch(err) {
		case 0:
			if(isfound)
				*isfound = true;
			return true;
		case DB_NOTFOUND:
			if(isfound)
				*isfound = false;
			return true;
		default:
			return ok(err, "content->exists(%s)", name);
	}
}

/* retrieve a record from the database

	returns NULL in buf and 0 in len if not found
*/
bool boushi_get(boushi_t *b, const char *name, void **buf, size_t *len) export;
bool boushi_get(boushi_t *b, const char *name, void **buf, size_t *len) {
	char *path = NULL;
	size_t pathlen = 0;
	uint8_t *out = NULL;
	size_t outlen = 0;
	DBT key = DBT0, val = DBT0;
	bool peachy = true;
	uint8_t sha1sum[SHA1_RESULTLEN];
	int err;

	if(!boushi_valid(b))
		return false;

	if(!b->initialized)
		return bok("boushi_get(boushi, name, buf, len): boushi object not initialized");

	if(!name)
		return bok("boushi_get(boushi, name, buf, len): name parameter is NULL");

	if(!buf && len)
		return bok("boushi_get(boushi, %s, buf, len): buf parameter is NULL", name);

	path = alloca(strlen(name) + 1);
	pathlen = file2path(path, name);

	sha1(path, pathlen, sha1sum);

	key.data = sha1sum;
	key.ulen =
	key.size = sizeof sha1sum;
	val.flags = DB_DBT_MALLOC;
	err = b->content->get(b->content, NULL, &key, &val, 0);
	if(err == DB_NOTFOUND) {
		*buf = NULL;
		*len = 0;
		return true;
	}

	peachy = ok(err, "content->get(%s)", name);

	if(peachy)
		peachy = lzo_unpack(b, val.data, val.size, &out, &outlen);

	if(peachy && val.data == out)
		val.data = NULL;

	if(val.data)
		free(val.data);

	if(peachy) {
		*buf = out;
		*len = outlen;
	}

	return peachy;
}

struct boushi_cursor {
	uint64_t magic;
	boushi_t *b;
	/* pointer to the current item/next item returned */
	char *next;
	/* prefix that we're comparing against */
	char *prefix;
	/* length of the prefix */
	size_t len;
	bool recursive;
	bool initialized;
};

static const boushi_cursor_t boushi_cursor_0 = {UINT64_C(0x106A09A91875811B), 0};

size_t boushi_cursor_size(void) export;
size_t boushi_cursor_size(void) {
	return sizeof boushi_cursor_0;
}

static bool boushi_cursor_valid(const boushi_cursor_t *c) {
	return c && c->magic == boushi_cursor_0.magic;
}

/* free memory associated with a boushi_cursor object, but not the struct itself */
bool boushi_cursor_exit(boushi_cursor_t *c) export;
bool boushi_cursor_exit(boushi_cursor_t *c) {
	if(!boushi_cursor_valid(c))
		return false;
	if(c->prefix)
		free(c->prefix);
	if(c->next)
		free(c->next);
	*c = boushi_cursor_0;
	return true;
}

/* given an entry in the database, return the next entry */
static bool boushi_cursor_succ(boushi_t *b, DB *db, const char *s, char **o, size_t *olen) {
	DBC *cursor;
	DBT key = DBT0, val = DBT0;
	size_t len;
	int err;
	bool peachy = true;

	len = strlen(s);

	if(!ok(db->cursor(db, NULL, &cursor, 0), "%s->cursor()",
			db == b->files ? "files" : db == b->directories ? "directories" : "db"))
		return false;

	key.data = (void *)s;
	key.ulen =
	key.size = len;
	key.flags = DB_DBT_MALLOC;

	/* value will be empty anyway */
	val.data = NULL;
	val.ulen =
	val.dlen =
	val.doff = 0;
	val.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

	err = cursor->get(cursor, &key, &val, DB_SET_RANGE);

	if(!err && key.size == len + 1 && !memcmp(key.data, s, key.size)) {
		free(key.data);
		err = cursor->get(cursor, &key, &val, DB_NEXT);
	}

	switch(err) {
		case 0:
			*o = key.data;
			*olen = key.size;
			break;
		case DB_NOTFOUND:
			*o = NULL;
			*olen = 0;
			break;
		default:
			peachy = ok(err, "cursor->get(%s)", s);
	}

	cursor->close(cursor);

	return peachy;
}

/* initialize a boushi_cursor struct

	you must call boushi_cursor_exit() even if this function fails
		or you will leak memory
*/
bool boushi_cursor_init(boushi_t *b, boushi_cursor_t *c, const char *s, bool recursive) export;
bool boushi_cursor_init(boushi_t *b, boushi_cursor_t *c, const char *s, bool recursive) {
	size_t len;

	if(c)
		*c = boushi_cursor_0;
	if(!boushi_valid(b))
		return false;
	if(!b->initialized)
		return bok("boushi_cursor_init(boushi, cursor, prefix, recursive): boushi object not initialized");
	if(!c)
		return bok("boushi_cursor_init(boushi, cursor, prefix, recursive): cursor parameter is NULL");
	if(!s)
		return bok("boushi_cursor_init(boushi, cursor, prefix, recursive): prefix parameter is NULL");

	c->b = b;
	c->recursive = recursive;

	c->prefix = malloc(strlen(s) + 2);
	if(!c->prefix)
		return ok(errno, "malloc(%"PRId64")", (int64_t)(strlen(s) + 2));

	len = file2path(c->prefix, s);
	if(len) {
		c->prefix[len++] = '/';
		c->prefix[len] = '\0';
	}
	c->len = len;

	c->initialized = true;

	return true;
}

/* retrieve the next item for this cursor

	memory for the returned pointer is managed by the cursor,
	it is only valid until the next call to boushi_cursor_next()
	or boushi_cursor_exit()
*/
bool boushi_cursor_next(boushi_cursor_t *c, char **r) export;
bool boushi_cursor_next(boushi_cursor_t *c, char **r) {
	boushi_t *b;
	bool peachy;
	char *prev;
	size_t len;
	DB *db;

	if(!boushi_cursor_valid(c))
		return false;

	b = c->b;

	if(!b)
		return false;

	if(!c->initialized)
		return bok("boushi_cursor_next(cursor, ret): cursor object not initialized");

	db = c->recursive ? b->files : b->directories;

	if(c->next) {
		prev = c->next;
		c->next = NULL;
		peachy = boushi_cursor_succ(b, db, prev, &c->next, &len);
		free(prev);
	} else {
		peachy = boushi_cursor_succ(b, db, c->prefix, &c->next, &len);
	}

	if(c->next && (len < c->len /* too small for the prefix */
		|| memcmp(c->next, c->prefix, c->len) /* doesn't match the prefix */
		|| (!c->recursive && strchr(c->next + c->len, '/')))) {
			/* this entry does not qualify for this listing */
			free(c->next);
			c->next = NULL;
	}
	if(peachy)
		*r = c->next;

	return peachy;
}
