/*****************************************************************************

	boushi.c - source code for the boushi command line tool

	Copyright (c) 2009-2010  Wessel Dankers <wsl@fruit.je>

	This file is part of boushi.

	boushi is free software: you can redistribute it and/or modify
	it under the terms of the GNU Lesser General Public License as
	published by the Free Software Foundation, either version 3 of
	the License, or (at your option) any later version.

	boushi is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Lesser General Public License for more details.

	You should have received a copy of the GNU General Public License
	and a copy of the GNU Lesser General Public License along with
	boushi.  If not, see <http://www.gnu.org/licenses/>.

*****************************************************************************/

#include "config.h"
#include "error.h"
#include "tar.h"
#include "io.h"
#include "boushi.h"

#include <stdbool.h>
#include <stdarg.h>
#include <stdint.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <ftw.h>
#include <getopt.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <arpa/inet.h>

static boushi_t *b = NULL;

static bool quiet = false;

typedef uint64_t stamp_t;
typedef long double real;

#define second UINT64_C(1000000000)
#define seconds * second
static stamp_t gettime(void) {
	struct timespec ts;
	if(clock_gettime(CLOCK_REALTIME, &ts) == -1)
		return 0;
	return (UINT64_C(1000000000) * (stamp_t)ts.tv_sec) + (stamp_t)ts.tv_nsec;
}

static void progress(uint64_t num) {
	static uint64_t count = 0, prev = 0;
	static stamp_t start = 0, last = 0;
	static stamp_t a, b;
	stamp_t now, d;
	static bool printed = false;

	if(!start)
		start = gettime();

	count += num;

	if(quiet)
		return;

	if(!last)
		last = start;

	now = gettime();
	if(now <= start)
		now = start + 1;

	if(num) {
		num = count - prev;
		d = last < now ? now - last : 1;
		if(d > 1 seconds) {
			a = (real)second * (real)second * (real)count / (real)(now - start);
			b = (real)second * (real)second * (real)num / (real)d;
			fprintf(stderr, "\rProcessed %"PRIu64" entries (%"PRIu64".%09"PRIu64"/s). Last %"PRIu64"s: %"PRIu64" (%"PRIu64".%09"PRIu64"/s).\033[K\r",
				count, a / (1 seconds), a % (1 seconds),
				d / (1 seconds), num, b / (1 seconds), b % (1 seconds));
			fflush(stderr);
			printed = true;
			prev = count;
			last = now;
		}
	} else {
		if(printed) {
			d = now - start;
			a = (real)second * (real)second * (real)count / (real)d;
			fprintf(stderr, "\rProcessed %"PRIu64" entries in %"PRIu64".%09"PRIu64"s (%"PRIu64".%09"PRIu64"/s).\033[K\n",
				count, d / (1 seconds), d % (1 seconds), a / (1 seconds), a % (1 seconds));
			fflush(stderr);
			printed = false;
			prev = count = 0;
			start = last = 0;
		}
	}
}

static int putfile_ftw(const char *file, const struct stat *st, int type, struct FTW *f) {
	int fd = -1;
	void *mem = MAP_FAILED;
	size_t size;
	bool peachy = true;

	if(type != FTW_F)
		return 0;

	size = st->st_size;
	if(peachy && size) {
		fd = open(file, O_RDONLY|O_LARGEFILE);
		if(fd == -1)
			peachy = ok(errno, "open", file);
	}

	if(peachy && size) {
		mem = mmap(NULL, size, PROT_READ, MAP_SHARED, fd, 0);
		if(mem == MAP_FAILED)
			peachy = ok(errno, "mmap", file);
	}

	if(fd != -1) {
		if(close(fd) == -1)
			peachy = ok(errno, "close", file);
	}

	if(peachy)
		peachy = boushi_put(b, file, mem, size);

	if(mem != MAP_FAILED) {
		if(munmap(mem, size) == -1)
			peachy = ok(errno, "munmap", file);
	}

	if(peachy)
		progress(1);

	return !peachy;
}

static bool delfile(const char *file) {
	bool found;
	if(boushi_del(b, file, &found)) {
		if(!found && !quiet)
			fprintf(stderr, "%s: not found\n", file);
		progress(1);
		return true;
	} else {
		fprintf(stderr, "%s\n", boushi_error(b));
		return false;
	}
}

static bool recursive_mkdir(const char *file) {
	char *p;
	char *s;
	s = p = alloca(strlen(file) + 1);
	strcpy(p, file);
	while((s = strchr(s, '/'))) {
		*s = '\0';
		if(mkdir(p, 0777) == -1 && errno != EEXIST) {
			ok(errno, "mkdir", p);
			return false;
		}
		*s++ = '/';
	}
	return true;
}

static bool writefile(const char *file, const void *buf, size_t len) {
	int fd;
	bool peachy = true;

	if(!recursive_mkdir(file))
		return false;

	if(!len) {
		fd = open(file, O_WRONLY|O_CREAT|O_TRUNC|O_NOCTTY, 0666);
		if(fd == -1) {
			ok(errno, "open", file);
			return false;
		} else {
			if(close(fd) == -1) {
				ok(errno, "close", file);
				return false;
			}
		}
		return true;
	}

	fd = open(file, O_WRONLY|O_CREAT|O_NOCTTY, 0666);
	if(fd == -1) {
		ok(errno, "open", file);
		return false;
	}
	if(!safewrite(fd, buf, len)) {
		peachy = false;
		ok(errno, "write", file);
	}
	if(peachy && ftruncate(fd, len) == -1) {
		peachy = false;
		ok(errno, "ftruncate", file);
	}
	if(close(fd) == -1 && peachy) {
		peachy = false;
		ok(errno, "close", file);
	}
	return peachy;
}

static bool getfile(const char *file) {
	void *buf = NULL;
	size_t size;
	bool peachy;

	if(!boushi_get(b, file, &buf, &size)) {
		fprintf(stderr, "boushi_get(%s): %s\n", file, boushi_error(b));
		return false;
	}

	if(!buf)
		return ok(ENOENT, "get", file);

	peachy = writefile(file, buf, size);
	free(buf);
	if(peachy)
		progress(1);
	return peachy;
}

static bool dumpfile(const char *file) {
	void *buf = NULL;
	size_t size;
	bool peachy;

	if(!boushi_get(b, file, &buf, &size)) {
		fprintf(stderr, "%s\n", boushi_error(b));
		return false;
	}

	if(!buf)
		return ok(ENOENT, "get", file);

	peachy = tar_write(STDOUT_FILENO, file, buf, size);
	free(buf);
	if(peachy)
		progress(1);
	return peachy;
}

static bool loadfile(void *_, const char *name, const void *buf, size_t len) {
	bool peachy;
	const char *link;
	void *linkbuf;
	if(buf && !len) {
		link = buf;
		if(boushi_get(b, link, &linkbuf, &len)) {
			if(buf) {
				peachy = boushi_put(b, name, linkbuf, len);
				free(linkbuf);
				if(peachy)
					progress(1);
				if(peachy)
					return true;
			} else {
				if(!quiet)
					fprintf(stderr, "Can't link '%s' to '%s': entry not found\n", link, name);
				return quiet;
			}
		}
	} else {
		if(boushi_put(b, name, buf, len)) {
			progress(1);
			return true;
		}
	}

	fprintf(stderr, "%s\n", boushi_error(b));
	return false;
}

static bool print(const char *s) {
	if(puts(s) == EOF)
		return errno == EPIPE ? false : ok(errno, "puts()");
	return true;
}

static bool foreach(int argc, char **argv, bool (*func)(const char *f), bool recursive) {
	boushi_cursor_t *c;
	bool peachy = true;
	char *s;
	int i;

	c = malloc(boushi_cursor_size());

	for(i = 0; i < argc; i++) {
		peachy = boushi_cursor_init(b, c, argv[i], recursive);
		if(!peachy)
			fprintf(stderr, "%s\n", boushi_error(b));

		while(peachy) {
			peachy = boushi_cursor_next(c, &s);
			if(!peachy) {
				fprintf(stderr, "%s\n", boushi_error(b));
				break;
			}
			if(s)
				peachy = func(s);
			else
				break;
		}
		boushi_cursor_exit(c);

		if(!peachy)
			break;
	}

	free(c);

	return peachy;
}

static bool cmd_ls(int argc, char **argv) {
	char *def = "";
	if(!argc) {
		argc = 1;
		argv = &def;
	}

	return foreach(argc, argv, print, false);
}

static bool cmd_find(int argc, char **argv) {
	char *def = "";
	if(!argc) {
		argc = 1;
		argv = &def;
	}

	return foreach(argc, argv, print, true);
}

static bool cmd_cat(int argc, char **argv) {
	int i;
	void *buf = NULL;
	size_t len;
	bool peachy = true;

	if(!argc && !quiet) {
		fprintf(stderr, "Usage: boushi cat <file ...>\n");
		return false;
	}

	for(i = 0; i < argc; i++) {
		if(!boushi_get(b, argv[i], &buf, &len)) {
			fprintf(stderr, "%s\n", boushi_error(b));
			break;
		}
		if(buf) {
			if(!safewrite(STDOUT_FILENO, buf, len)) {
				ok(errno, "write");
				break;
			}
			free(buf);
			buf = NULL;
		} else {
			peachy = ok(ENOENT, argv[i]);
		}
	}
	if(buf)
		free(buf);

	return peachy;
}

static bool cmd_put(int argc, char **argv) {
	struct stat st;
	int i;

	if(!argc && !quiet) {
		fprintf(stderr, "Usage: boushi put <files or directories ...>\n");
		return false;
	}

	for(i = 0; i < argc; i++) {
		if(stat(argv[i], &st) == -1)
			return ok(errno, "stat", argv[i]);
		if(S_ISREG(st.st_mode)) {
			if(putfile_ftw(argv[i], &st, FTW_F, NULL))
				return false;
		} else {
			if(nftw(argv[i], putfile_ftw, 500, FTW_DEPTH|FTW_PHYS))
				return false;
		}
	}

	return true;
}

static bool cmd_del(int argc, char **argv) {
	if(!argc && !quiet) {
		fprintf(stderr, "Usage: boushi del <files or directories ...>\n");
		return false;
	}

	return foreach(argc, argv, delfile, true);
}

static bool cmd_load(int argc, char **argv) {
	int i, fd;
	bool peachy = true;

	if(!argc && !quiet) {
		fprintf(stderr, "Usage: boushi load <file.tar ...>\n");
		return false;
	}

	for(i = 0; i < argc; i++) {
		fd = open(argv[i], O_RDONLY|O_NOCTTY);
		if(fd == -1) {
			ok(errno, argv[i]);
		} else {
			if(!tar_read(fd, loadfile, b))
				peachy = false;
			close(fd);
		}
	}

	return peachy;
}

static bool cmd_dump(int argc, char **argv) {
	char *def = "";
	if(!argc) {
		argc = 1;
		argv = &def;
	}

	if(!quiet && isatty(STDOUT_FILENO)) {
		fprintf(stderr, "Refusing to write to a terminal.\n");
		return false;
	}
	
	if(!foreach(argc, argv, dumpfile, true))
		return false;

	return tar_end(STDOUT_FILENO);
}

static bool cmd_get(int argc, char **argv) {
	char *def = "";
	if(!argc) {
		argc = 1;
		argv = &def;
	}

	return foreach(argc, argv, getfile, true);
}

static const struct option long_options[] = {
	{"help\0             Print this message to stdout", 0, 0, 'h'},
	{"version\0          Print the program version", 0, 0, 'v'},
	{"quiet\0            Error messages only", 0, 0, 'q'},
	{"database\0 <dir>   Directory where the database resides", 1, 0, 'd'},
	{0, 0, 0, 0}
};

static const int num_options = sizeof long_options / sizeof *long_options;

static const struct {
	const char *name;
	bool (*func)(int argc, char **argv);
	const char *desc;
} commands[] = {
	{ "ls", cmd_ls, "list files and directories" },
	{ "cat", cmd_cat, "dump file contents to stdout" },
	{ "get", cmd_get, "get files from the database (recursively)" },
	{ "put", cmd_put, "add files/directories to the database" },
	{ "add", cmd_put, "(alias for put)" },
	{ "del", cmd_del, "delete files from the database (recursively)" },
	{ "find", cmd_find, "list files (recursively)" },
	{ "dump", cmd_dump, "write a tar archive (recursively)" },
	{ "load", cmd_load, "load a tar archive" }
};

static const int commands_count = sizeof commands / sizeof *commands;

static void usage(FILE *fh, const char *progname) {
	int i;
	char col[100];

	fprintf(fh, "Usage: %s [-", progname);
	for(i = 0; long_options[i].name; i++)
		if(long_options[i].val && !long_options[i].has_arg)
			fputc(long_options[i].val, fh);
	fprintf(fh, "] -d <directory> <command> [file...]\n");
	for(i = 0; long_options[i].name; i++)
		fprintf(fh, "\t-%c, --%s%s\n",
			long_options[i].val,
			long_options[i].name,
			long_options[i].name + strlen(long_options[i].name) + 1);
	fprintf(fh, "Subcommands:\n");
	for(i = 0; i < commands_count; i++) {
		snprintf(col, sizeof col - 1, "%s [...]", commands[i].name);
		fprintf(fh, "\t%s %-15s %s\n", progname, col, commands[i].desc);
	}
}

static void getshortoptions(char *dst) {
	int i, j;
	*dst++ = ':';
	for(i = 0; long_options[i].name; i++) {
		if(!isprint(long_options[i].val))
			continue;
		*dst++ = long_options[i].val;
		for(j = 0; j < long_options[i].has_arg; j++)
			*dst++ = ':';
	}
	*dst++ = '\0';
}

int main(int argc, char **argv) {
	int i, option_index;
	char *short_options;
	char *dbdir = NULL;
	bool (*func)(int argc, char **argv) = NULL;
	bool peachy = false;

	quiet = !isatty(STDERR_FILENO);

	signal(SIGPIPE, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);

	short_options = alloca(num_options * 3);
	if(!short_options) {
		ok(errno, "alloca");
		exit(2);
	}
	getshortoptions(short_options);

	dbdir = getenv("BOUSHI_DB");

	opterr = 0;
	while((i = getopt_long(argc, argv, short_options, long_options, &option_index)) != EOF) {
		switch(i) {
			case 'h':
				puts("boushi - use a database for file storage");
				usage(stdout, *argv);
				exit(0);
			case 'v':
				printf("boushi %s\ncopyright (c) 2008-2010 Wessel Dankers <wsl@fruit.je>\n", VERSION);
				exit(0);
			case 'd':
				if(optarg)
					dbdir = strdup(optarg);
				break;
			case 'q':
				quiet = true;
				break;
			case ':':
				usage(stderr, *argv);
				if(optopt)
					fprintf(stderr, "Option -%c requires an argument\n", optopt);
				else
					fprintf(stderr, "Option requires an argument\n");
				exit(1);
			case '?':
				usage(stderr, *argv);
				if(optopt)
					fprintf(stderr, "Unknown option: -%c\n", optopt);
				else
					fprintf(stderr, "Unknown option\n");
				exit(1);
			default:
				usage(stderr, *argv);
				exit(1);
		}
	}

	if(optind == argc || !dbdir) {
		usage(stderr, *argv);
		exit(1);
	}

	for(i = 0; i < commands_count; i++)
		if(!strcasecmp(argv[optind], commands[i].name))
			func = commands[i].func;

	if(!func) {
		fprintf(stderr, "Unknown command '%s'\n", argv[optind]);
		exit(1);
	}

	optind++;

	b = malloc(boushi_size());

	if(b) {
		if(boushi_init(b, dbdir))
			peachy = func(argc - optind, argv + optind);
		else
			fprintf(stderr, "%s\n", boushi_error(b));
		boushi_exit(b);
		free(b);
	}

	progress(0);

	return !peachy;
}
