/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.request.handler.entree.sso.service;

import java.util.Hashtable;
import java.util.logging.Level;

import javax.servlet.ServletConfig;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.aselect.server.config.ASelectConfigManager;
import org.aselect.server.elo.ELO;
import org.aselect.server.elo.ELOFactory;
import org.aselect.server.elo.IELOStorage;
import org.aselect.server.log.ASelectAuthenticationLogger;
import org.aselect.server.processor.IProcessor;
import org.aselect.server.request.RequestState;
import org.aselect.server.request.handler.AbstractRequestHandler;
import org.aselect.server.tgt.TGTIssuer;
import org.aselect.server.tgt.TGTManager;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.utils.Utils;

/**
 * The SSO service handler is where the user is redirected to after authentication
 * with the ELO.
 * <br><br>
 * <b>Description:</b><br>
 * The handler receives the request sent by the user. The request's rid parameter is
 * extracted and the associated session variables are loaded to create a TGT for the
 * user.
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public class SSOServiceHandler extends AbstractRequestHandler
{

    private static final String MODULE = "SSOServiceHandler";
    private ASelectAuthenticationLogger _authenticationLogger;
    private IProcessor _processorSSOService;
    private IELOStorage _oStore = null;
    private String _sMyServerID = null;
    private TGTManager _tgtManager = null;
    
    /**
     * Default constructor
     * <br><br>
     * <b>Description:</b>
     * <br>
     * Initializes the ELO store, if needed.
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * @throws ASelectException If the ELO store cannot be initialized.
     */
    public SSOServiceHandler() throws ASelectException
    {
        _oStore = ELOFactory.getHandle().getEloStore();
        _authenticationLogger = ASelectAuthenticationLogger.getHandle();
        _tgtManager = TGTManager.getHandle();
    }
    
    /**
     * Initialization of the handler. Reads the configuration.
     * <br><br>
     * @see org.aselect.server.request.handler.AbstractRequestHandler#init(javax.servlet.ServletConfig, java.lang.Object)
     */
    public void init(ServletConfig oServletConfig, Object oConfig)
        throws ASelectException
    {
        String sMethod = "init()";
        try
        {
            super.init(oServletConfig, oConfig);
            
            Object oASelect = null;
            try
            {
                oASelect = _configManager.getSection(null, 
                    "aselect");
            }
            catch(ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Could not find 'aselect' config section in config file");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
    
            try
            {
                _sMyServerID  = _configManager.getParam(oASelect, "server_id");
            }
            catch(ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Could not retrieve 'server_id' config parameter in 'aselect' config section");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            _processorSSOService = createProcessor(_configManager, oConfig, "sso_service");
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, "Could not initialize", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR, e);
        }
    }
    
    
    /**
     * Doesn't really do anything.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#destroy()
     */
    public void destroy()
    {
        //does not have to do anything
    }

    /**
     * Processes the request. Requires the 'rid' parameter and loads the associated session
     * variables.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#process(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    public RequestState process(HttpServletRequest request,
            HttpServletResponse response) throws ASelectException 
    {
        String sMethod = "process()";
        
        String sRid = request.getParameter("rid");
        if (sRid == null || "".equals(sRid))
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "ELO SSO service request lacks 'rid' parameter, and response could not be sent");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        Hashtable htSession = _oSessionManager.getSessionContext(sRid);
        if (htSession == null)
        {
            _systemLogger.log(Level.FINE, MODULE, sMethod, "Session could not be loaded");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
        }
        
        String sEloID = (String)htSession.get("elo_id");
        if (sEloID == null)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "ELO ID could not be loaded from session");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        ELO oElo = _oStore.getEloByID(sEloID);
        if (oElo == null)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "ELO could not be loaded from ELO Storage");
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        
        Hashtable htRemoteAttributes = new Hashtable();
        
        Object sUid = htSession.get("uid");
        if (sUid != null) htRemoteAttributes.put("uid", sUid);
        htRemoteAttributes.put("organization", oElo.getID());
        htRemoteAttributes.put("authsp_level", ""+oElo.getLevel());
        htRemoteAttributes.put("authsp", oElo.getID());
        
        String sAtts = (String)htSession.get("elo_attributes");
        if (sAtts != null)
        {
            htRemoteAttributes.put("attributes", sAtts);
        }
        
        boolean bResume = true;
        
        if (_processorSSOService != null)
            bResume = _processorSSOService.process(response, sRid, 
                null, htRemoteAttributes);
        
        if (bResume)
        {
            _authenticationLogger.log(new Object[] {
                "SSOService", sUid, request.getRemoteAddr(), 
                oElo.getID(), htSession.get("app_id"), "granted"});

            TGTIssuer oTGTIssuer = new TGTIssuer(_sMyServerID);
            oTGTIssuer.issueCrossTGT(sRid, null, htRemoteAttributes,response, 
                getTGTFromCredentials(request), request);
        }
        
        return new RequestState(null);
    }
    
    private IProcessor createProcessor(ASelectConfigManager configManager, Object oConfig, String ProcessorID) 
        throws ASelectException
    {
        String sMethod = "createProcessor()";
        IProcessor processor = null;
        Object oProcessor = null;
        try
        {
            oProcessor = configManager.getSection(oConfig, "processor", "id=" + ProcessorID);
        }
        catch (ASelectConfigException e)
        {
            _systemLogger.log(Level.CONFIG, MODULE, sMethod, 
                "No optional 'processor' section found in configuration with id: " + ProcessorID);
        }
        
        if (oProcessor != null)
        {
            String sClass = null;
            try
            {
                sClass = configManager.getParam(oProcessor, "class");
            }
            catch (ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "No 'class' item in 'processor' section found in configuration", e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            Class cProcessor = null;
            try
            {
                cProcessor = Class.forName(sClass);
            }
            catch (ClassNotFoundException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "Configured 'class' item in 'processor' section not found: " + sClass, e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            Object objProcessor = null;
            try
            {
                objProcessor = cProcessor.newInstance();
            }
            catch (InstantiationException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "Could not instantiate: " + sClass, e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            catch (IllegalAccessException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "Could not access: " + sClass, e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            try
            {
                processor = (IProcessor)objProcessor;
            }
            catch (ClassCastException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "Configured class is not of type IProcessor: " + sClass, e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            try
            {
                processor.init(configManager, oProcessor);
            }
            catch (ASelectException e) 
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "Could not initialize: " + sClass, e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            _systemLogger.log(Level.INFO, MODULE, sMethod, "Processor initialized: " + sClass);
        }
        
        return processor;
    }
    
    /*
     * Retrieves the  A-Select credentials from the cookie.
     */ 
    private String getTGTFromCredentials(HttpServletRequest request)
    {
        String sMethod = "getTGTFromCredentials()";
        
        // check for credentials that might be present
        Cookie[] caCookies = request.getCookies();
        if (caCookies == null)
            return null;
        
        String sCredentialsCookie = null;
        for (int i = 0; i < caCookies.length; i++)
        {
            if (caCookies[i].getName().equals("aselect_credentials"))
            {
                sCredentialsCookie = caCookies[i].getValue();
                
                _systemLogger.log(Level.FINER, MODULE, sMethod, 
                    "Cookie 'aselect_credentials' has value: " + sCredentialsCookie);
                
                //remove '"' surrounding cookie if applicable
                int iLength = sCredentialsCookie.length();
                if(sCredentialsCookie.charAt(0) == '"' &&
                    sCredentialsCookie.charAt(iLength-1) == '"')
                {
                    sCredentialsCookie = sCredentialsCookie.substring(
                        1, iLength-1);
                }
            }
        }
        if (sCredentialsCookie == null)
            return null;
        
        Hashtable sCredentialsParams = Utils.convertCGIMessage(sCredentialsCookie);
        if (sCredentialsParams == null)
            return null;
        
        String sTgt = (String)sCredentialsParams.get("tgt");
        if (sTgt == null)
            return null;
        
        String sUserId = (String)sCredentialsParams.get("uid");
        if (sUserId == null)
            return null;
        
        String sServerId = (String)sCredentialsParams.get("a-select-server");
        if (sServerId == null)
            return null;
        
        if (!sServerId.equals(_sMyServerID))
            return null;
        
        Hashtable htTGTContext = _tgtManager.getTGT(sTgt);
        if (htTGTContext == null)
            return null;
        
        if (!sUserId.equals(htTGTContext.get("uid")))
            return null;
        
        return sTgt;
    }
}
