/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.request.handler.entree.sso.notification;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.logging.Level;

import javax.servlet.ServletConfig;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.aselect.server.elo.ELO;
import org.aselect.server.elo.ELOFactory;
import org.aselect.server.elo.IELOStorage;
import org.aselect.server.request.RequestState;
import org.aselect.server.request.handler.AbstractRequestHandler;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;

/**
 * Sets a cookie with the ELO ID, if the ELO is registered in the ELO store.
 * <br><br>
 * <b>Description:</b><br>
 * This handler extracts the 'elo' parameter from the request and verifies the result as the
 * ELO ID in the ELO store. If the ELO ID is found, the ELO is registered and a cookie is set in the
 * response. The cookie is in the form [elo='elo_id'] and can be used to determine at what ELO the user
 * has authenticated. If the ELO ID cannot be found in the store, no cookie is set.
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public class SSONotifcationServiceHandler extends AbstractRequestHandler
{
    private final static String MODULE = "SSONotifcationServiceHandler";
    private final static String COOKIE_NOTIFICATION = "elo";
    private String _sCookieDomain;
    private String _sCookieDomainPath;
    private IELOStorage _oStore = null;
    private String _sTemplate;
    
    /**
     * Initializes the handler, retrieves the ELO store.
     * <br><br>
     * @see org.aselect.server.request.handler.AbstractRequestHandler#init(javax.servlet.ServletConfig, java.lang.Object)
     */
    public void init(ServletConfig oServletConfig, Object oConfig) throws ASelectException
    {
        String sMethod = "init()";
        try
        {
            super.init(oServletConfig, oConfig);
            _oStore = ELOFactory.getHandle().getEloStore();
            readCookieSettings(oConfig);
            
            String sTemplateName = null;
            try
            {
                sTemplateName = _configManager.getParam(oConfig, "template");
            }
            catch (ASelectConfigException e)
            {
                _sTemplate = null;
                _systemLogger.log(Level.CONFIG, MODULE, sMethod, "No optional 'template' item found in configuration", e);
            }
            
            if (sTemplateName != null)
            {
                String sWorkingDir = _configManager.getWorkingdir();
                StringBuffer sbTemplateFilename = new StringBuffer();
                sbTemplateFilename.append(sWorkingDir);
                if (!sWorkingDir.endsWith(File.separator))
                    sbTemplateFilename.append(File.separator);
                sbTemplateFilename.append("conf");
                sbTemplateFilename.append(File.separator);
                sbTemplateFilename.append("html");
                sbTemplateFilename.append(File.separator);
                sbTemplateFilename.append(sTemplateName);
                
                File fTemplate = new File(sbTemplateFilename.toString());
                if (!fTemplate.exists())
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod
                        , "Configured template does not exists: " + sbTemplateFilename.toString());
                    throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
                }
                _sTemplate = readTemplate(fTemplate);
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod
                , "Could not initialize", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
    }
    
    /**
     * Processes the request, thereby setting a cookie in the response if the ELO ID that was
     * send in the request is found in the ELO store. If the 'elo' parameter is missing, a '400 Bad
     * Request' reply is returned.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#process(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    public RequestState process(HttpServletRequest request,
            HttpServletResponse response) throws ASelectException
    {
        String sMethod = "process()";
        
        try
        {
            String sEloID = request.getParameter("elo");
            if (sEloID == null || "".equals(sEloID))
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "ELO notification request lacks 'elo' parameter, and response could not be sent");
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }

            ELO oElo = _oStore.getEloByID(sEloID);
            if (oElo == null)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Unknown ELO id in request: " + sEloID);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
            
            verifyReferer(request, oElo);
            
            //ELO url found, so set cookie
            Cookie oCookie = new Cookie(COOKIE_NOTIFICATION, oElo.getURL());
            
            if (_sCookieDomain != null)
                oCookie.setDomain(_sCookieDomain);
            
            if (_sCookieDomainPath != null)
                oCookie.setPath(_sCookieDomainPath);
            
            if (_configManager.getCookiesVersion() != -1)
                oCookie.setVersion(_configManager.getCookiesVersion());
            
            response.addCookie(oCookie);
            
            if (_sTemplate != null && oElo.showTemplate())
                showSuccess(response);
        }
        catch(ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, "Could not process", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        return new RequestState(null);
    }
    
	/**
	 * Does not do anything.
	 * <br><br>
	 * @see org.aselect.server.request.handler.IRequestHandler#destroy()
	 */
	public void destroy()
	{
	    //does not have to do anything.
	}
	
    private void readCookieSettings(Object config) throws ASelectException
    {
        String sMethod = "readCookieSettings()";
        
        Object oCookieDomain = null;
        try
        {
            oCookieDomain = _configManager.getSection(config, "notification_cookie");
        }
        catch (ASelectException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod,  
                "No 'notification_cookie' section found in configuration");
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
        }
        
        try
        {
            _sCookieDomain = _configManager.getParam(oCookieDomain, "domain");
            
            //Needs to use a dot as prefix, to make it an official domain name
            if (!_sCookieDomain.startsWith("."))
                _sCookieDomain = "." + _sCookieDomain;
            
            _systemLogger.log(Level.INFO, MODULE, sMethod,  
                "Notification cookie will be set on domain: " + _sCookieDomain);
        }
        catch (ASelectException e)
        {
            _systemLogger.log(Level.CONFIG, MODULE, sMethod,  
                "No optional 'domain' item in 'notification_cookie' section found in configuration");
        }
        
        try
        {
            _sCookieDomainPath = _configManager.getParam(oCookieDomain, "path");
            
            _systemLogger.log(Level.INFO, MODULE, sMethod,  
                "Notification cookie will be set on path: " + _sCookieDomainPath);
        }
        catch (ASelectException e)
        {
            _systemLogger.log(Level.CONFIG, MODULE, sMethod,  
                "No optional 'path' item in 'notification_cookie' section found in configuration");
        }
    }
    
    private void verifyReferer(HttpServletRequest request, ELO oElo) 
        throws ASelectException
    {
        String sMethod = "verifyReferer()";
                
        //verify target URL with ELO url
        String sReferer = request.getHeader("referer");
        if (sReferer == null)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                "No referer header in request");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        URL urlReferer = null;
        try
        {
            urlReferer = new URL(sReferer);
        }
        catch (MalformedURLException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                "Invalid referer header in request: " + sReferer, e);
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        URL urlElo = null;
        try
        {
            urlElo = new URL(oElo.getURL());
        }
        catch (MalformedURLException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                "Invalid ELO url in ELO store: " + oElo.getURL(), e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        
        if (!urlElo.getHost().equalsIgnoreCase(urlReferer.getHost()))
        {
            StringBuffer sbWarning = new StringBuffer("Header referer contains invalid hostname for ELO with id '");
            sbWarning.append(oElo.getID());
            sbWarning.append("': ");
            sbWarning.append(sReferer);
            _systemLogger.log(Level.WARNING, MODULE, sMethod, sbWarning.toString());
                
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
    }
    
    private void showSuccess(HttpServletResponse response) 
        throws ASelectException
    {
        String sMethod = "showSuccess()";
        
        PrintWriter pwOut = null;
        try
        {
            pwOut = response.getWriter();
            pwOut.print(_sTemplate);
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod
                , "Can't show success page", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        finally
        {
            if (pwOut != null)
                pwOut.close();
        }
    }
    
    /**
     * Reads the given file and returns it as String.
     * <br><br>
     * @param fTemplate the full file name to the template
     * @return the template as String
     * @throws ASelectException if the file couldn't be read
     */
    private String readTemplate(File fTemplate) throws ASelectException
    {
        String sMethod = "readTemplate()";
        BufferedReader brIn = null;
        String sLine = null;
        StringBuffer sbReturn = new StringBuffer();
        try
        {
            brIn = new BufferedReader(
                new InputStreamReader(new FileInputStream(fTemplate)));

            while ((sLine = brIn.readLine()) != null)
            {
                sbReturn.append(sLine);
                sbReturn.append("\n");
            }
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, "Could not read template", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR, e);
        }
        finally
        {
            try
            {
                if (brIn != null)
                    brIn.close();
            }
            catch (IOException e)
            {
                _systemLogger.log(Level.FINE, MODULE, sMethod, "Could not close BufferedReader", e);
            }
        }
        return sbReturn.toString();
    }
}
