/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */

package org.aselect.server.processor;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;
import java.util.logging.Level;

import org.aselect.server.log.ASelectSystemLogger;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectException;

/**
 * A collection of NMVPairs. Duplicate names are not allowed. <br>
 * <br>
 * <b>Description:</b><br>
 * A collection of NMVPairs. Duplicate names are not allowed. They will be
 * stored as a list under the key. Attributes can be encoded as a URLEncoded
 * string. <br>
 * <br>
 * <b>Concurrency issues:</b> <br> - <br>
 * 
 * @author Alfa & Ariss
 */
public class AttributesMulti
{
    private final static String MODULE = "AttributesMulti";
    private ASelectSystemLogger _systemLogger;
    private Hashtable _attributes;
    
    /**
     * Create an empty {@link AttributesMulti} <br>
     * <b>Description:</b> <br>
     * Initializes the attributes.<br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     */
    public AttributesMulti ()
    {
        _systemLogger = ASelectSystemLogger.getHandle();
        _attributes = new Hashtable();
    }

    /**
     * Create a new {@link AttributesMulti}, based on a URL encoded name/value
     * string. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param urlEncoded URL encoded string with attributes
     * @throws ASelectException if creation fails
     */
    public AttributesMulti (String urlEncoded) throws ASelectException
    {
        _systemLogger = ASelectSystemLogger.getHandle();
        _attributes = new Hashtable();
        if (urlEncoded != null)
            addString(urlEncoded);
    }

    /**
     * Create a {@link AttributesMulti} object from a Hashtable (containing
     * Strings and Vectors). <br>
     * <br>
     * <b>Description:</b> <br>
     * Instantiates a new object with the same values as the Hashtable. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param htAttributes A Hashtable to create the object from.
     */
    public AttributesMulti (Hashtable htAttributes)
    {
        _systemLogger = ASelectSystemLogger.getHandle();
        _attributes = new Hashtable();
        
        Iterator iter = htAttributes.keySet().iterator();
    
        // Iterate through all the keys.
        while (iter.hasNext())
        {
            // Retrieve the key
            String key = (String)iter.next();
    
            // Retrieve the value
            Object value = htAttributes.get(key);
    
            if (value instanceof Vector)
                addAttributes(key, (Vector)value);
            else
                addAttribute(key, (String)value);
        }
    }

    /**
     * Add a new name, value pair to the list of attributes. If the name already
     * exists, it will be added to the name. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name The name of the pair
     * @param value The value of the pair.
     */
    public void addAttribute(String name, String value)
    {
        NMVPair nmvpair = null;

        // Determine if key already exists
        if (_attributes.containsKey(name))
        {
            // Yes, get it
            nmvpair = (NMVPair)_attributes.get(name);
        }
        else
        {
            // No, add it to the table
            nmvpair = new NMVPair(name);
            _attributes.put(name, nmvpair);
        }

        nmvpair.addValue(value);
    }

    /**
     * Add a new NVPair to the list of attributes. <br>
     * <br>
     * <b>Description:</b> <br>
     * Add a new NVPair to the list of attributes. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name
     *            The name of the attribute
     * @param values
     *            The values to add to the attribute.
     */
    public void addAttributes(String name, Vector values)
    {
        NMVPair nmvpair = null;
    
        if (_attributes.containsKey(name))
        {
            nmvpair = (NMVPair)_attributes.get(name);
        }
        else
        {
            nmvpair = new NMVPair(name);
            _attributes.put(name, nmvpair);
        }
    
        nmvpair.addValues(values);
    }

    /**
     * Get the attribute by name (will only return the first one if it is a
     * multivalued attribute.) <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name The name of the attribute to retrieve.
     * @return The attribute value (will only return the first one if it is a 
     * multivalued attribute).
     */
    public String getAttribute(String name)
    {
        if (_attributes.containsKey(name))
        {
            return (String)getAttributes(name).get(0);
        }
        return null;
    }

    /**
     * Get a list of attributes corresponding to the name. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name The name of the attribute to retrieve.
     * @return The list of values;
     */
    public List getAttributes(String name)
    {
        NMVPair pair = (NMVPair)_attributes.get(name);
        return pair.getValues();
    }

    /**
     * Remove attribute from the list. <br>
     * <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param key The attribute to remove.
     */
    public void remove(String key)
    {
        if (_attributes.containsKey(key))
        {
            _attributes.remove(key);
        }
    }

    /**
     * Create a hashtable (mixed with Vectors and String) based on this object.
     * <br>
     * <br>
     * <b>Description:</b> <br>
     * Create a hashtable (mixed with Vectors and String) based on this object.
     * <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @return The hashtable containing the attributes.
     */
    public Hashtable toHashtable()
    {
        Hashtable result = new Hashtable();
    
        Iterator iter = _attributes.values().iterator();
        while (iter.hasNext())
        {
            NMVPair tmpPair = (NMVPair)iter.next();
            tmpPair.updateMixedHashtable(result);
        }
    
        return result;
    }

    /**
     * URLEncode the attributes. <br>
     * <b>Description:</b> <br>
     * URLEncode the attributes which are present in this object.<br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @return The encoded string.
     */
    public String toEncodedString()
    {
        StringBuffer sbEncoded = new StringBuffer();
    
        Iterator i = _attributes.keySet().iterator();
        while (i.hasNext())
        {
            String key = (String)i.next();
            NMVPair tmpPair = (NMVPair)_attributes.get(key);
            sbEncoded.append(tmpPair.encodeMV());
            if (i.hasNext())
            {
                sbEncoded.append('&');
            }
        }
        return sbEncoded.toString();
    }

    /**
     * Add a name/value string to the attributes list. <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param s The string to add to the attributes.
     * @throws ASelectException 
     */
    private void addString(String s) throws ASelectException
    {
        String sMethod = "addString()";
        
        try
        {
            String[] splitted = s.split("&");
            for (int i = 0; i < splitted.length; i++)
            {
                String[] result = splitted[i].split("=");
                
                if (result.length < 1 || result.length > 2)
                {
                    _systemLogger.log(Level.FINE, MODULE, sMethod, 
                        "Malformed attribute name/value pair: " + splitted[i]);
                    throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
                }
                
                String decodedName = "";
                if (result.length > 0)
                {
                    decodedName = URLDecoder.decode(result[0], "UTF-8");
                
                    String decodedValue = "";
                    if (result.length > 1)
                        decodedValue = URLDecoder.decode(result[1], "UTF-8");
                    
                    if (!decodedName.trim().equals(""))
                        addAttribute(decodedName, decodedValue);
                }
            }
        }
        catch (UnsupportedEncodingException e)
        {
            _systemLogger.log(Level.FINE, MODULE, sMethod, 
                "Could not use decode with UTF-8", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
    }
}
