/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.processor;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import java.util.logging.Level;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.GetMethod;
import org.aselect.server.config.ASelectConfigManager;
import org.aselect.server.log.ASelectAuthenticationLogger;
import org.aselect.server.log.ASelectSystemLogger;
import org.aselect.server.processor.IProcessor;
import org.aselect.server.sam.ASelectSAMAgent;
import org.aselect.server.session.SessionManager;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.sam.agent.SAMResource;
import org.aselect.system.utils.BASE64Encoder;
import org.aselect.system.utils.Base64;


/**
 * The attribute processor. This processor can modify the user attributes which
 * are passed to this module. <br>
 * <br>
 * <br>
 * <b>Concurrency issues:</b> <br> - <br>
 * 
 * @author Alfa & Ariss
 */
public class AttributeProcessor implements IProcessor
{
	private final static String MODULE = "AttributeProcessor";

	private ASelectAuthenticationLogger _authenticationLogger;
	private SAMResource _samActiveResource;
	private String _sTargetUrl;
	private String _sTargetResourceGroup;
	private String _sResumeUrl;
	private HttpClient _httpClient;
	private ASelectSystemLogger _systemLogger;
	private SessionManager _sessionManager;
	private ASelectSAMAgent _samAgent;
	private ASelectConfigManager _configManager;

	/**
	 * Create a new {@link AttributeProcessor} <br>
	 * <br>
	 * <b>Concurrency issues:</b> <br> - <br>
	 * <br>
	 * <b>Preconditions:</b> <br> - <br>
	 * <br>
	 * <b>Postconditions:</b> <br> - <br>
	 */
	public AttributeProcessor()
	{
		_systemLogger = ASelectSystemLogger.getHandle();
		_authenticationLogger = ASelectAuthenticationLogger.getHandle();
		_samAgent = ASelectSAMAgent.getHandle();
        _sessionManager = SessionManager.getHandle();
	}

	/**
	 * Clear up remaining vars. <br>
	 * <br>
	 * 
	 * @see org.aselect.server.processor.IProcessor#destroy()
	 */
	public void destroy()
	{
	}

	/**
	 * Init the module. Load the configuration. <br>
	 * <br>
	 * <br>
	 * 
	 * @see org.aselect.server.processor.IProcessor#init(org.aselect.server.config.ASelectConfigManager,
	 *      java.lang.Object)
	 */
	public void init(ASelectConfigManager configManager, Object oConfig) 
	    throws ASelectException
	{
	    String sMethod = "init()";
	    try
	    {
	        _configManager = configManager;
	        
    		// Init the httpclient
    		_httpClient = new HttpClient(new MultiThreadedHttpConnectionManager());
    
    		// Get the right element from the config
    		Object secTarget = getSection(configManager, "target", oConfig);
    		if (secTarget == null)
    		{
    		    _systemLogger.log(Level.WARNING, MODULE, sMethod, "No 'target' section found in configuration");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
    		}
    		
    		// Retrieve URL from config.
    		_sTargetResourceGroup = getParam(configManager, secTarget, "resourcegroup");
    		if (_sTargetResourceGroup == null)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "No 'resourcegroup' item found in 'target' section in configuration");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
    		
    		getTargetUrl();
    		
    		Object secResume = getSection(configManager, "resume", oConfig);
            if (secResume == null)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "No 'resume' section found in configuration");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            //Retrieve redirect url from config.
            _sResumeUrl = getParam(configManager, secResume, "url");
            if (_sResumeUrl == null)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "No 'url' item found in 'resume' section in configuration");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            try
            {
                new URL(_sResumeUrl);
            }
            catch(MalformedURLException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Configured 'url' item in 'resume' section is not a URL: " + _sResumeUrl);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
	    }
	    catch (ASelectException e)
	    {
	        throw e;
	    }
	    catch (Exception e)
	    {
	        _systemLogger.log(Level.SEVERE, MODULE, sMethod, "Could not initialize", e);
	        throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
	    }
	}

	/**
	 * Processes the actual request. <br>
	 * <b>Description:</b> <br>
	 * Deserializes the string. Converts these into a list. <br>
	 * <br><br>
	 * @see org.aselect.server.processor.IProcessor#process(javax.servlet.http.HttpServletResponse, java.lang.String, java.util.Hashtable, java.util.Hashtable)
	 */
	public boolean process(HttpServletResponse servletResponse, String sRid, 
	    Hashtable serviceRequest, Hashtable additional) throws ASelectException
	{
		boolean bContinueProcess = true;
		final String sMethod = "process()";

		try
		{
		    AttributesMulti attributes = null;
		    String sSerializedAttributes = (String)additional.get("attributes");
		    if (sSerializedAttributes != null)
		    {
		        Hashtable htAttributes = deserializeAttributes(sSerializedAttributes);
		        if (htAttributes != null)
		            attributes = new AttributesMulti(htAttributes);
		    }

			// Perform call, can modify am
			bContinueProcess = performCall(servletResponse, sRid, attributes, additional);
		}
		catch (ASelectException e)
		{
			throw e;
		}
		catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, "Could not process request", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
		
		return bContinueProcess;
	}


	/**
	 * Make a call to a remote application to retrieve attributes and user id. 
	 * <br>
	 * <b>Description:</b>
	 * <br>
	 * Performs a call to a remote application. The remote application can 
	 * optionally return attributes and/or a user id. The A-Select 
	 * 'remote_attributes' and 'uid' parameter will be overwritten with the 
	 * returned information from the application.   
	 * <br>
	 * <b>Concurrency issues:</b> <br> - <br>
	 * <br>
	 * <b>Preconditions:</b> <br> - <br>
	 * <br>
	 * <b>Postconditions:</b> <br> - <br>
	 * 
	 * @param response The HTTP response object. This will be used to send a 
	 * redirect when necessary.
	 * @param rid The request id.
	 * @param attributes A collection of attributes.
	 * @param additional A Hashtable contains a range of variables that will be 
	 * written to the TGT.
	 * @throws ASelectException if call cannot be performed
	 */
	private boolean performCall(HttpServletResponse response, String rid, 
	    AttributesMulti attributes, Hashtable additional)
		throws ASelectException
	{
	    final String sMethod = "performCall()";
	    boolean bContinueProcess = true;
	    GetMethod method = null;
		
		try
        {
    		Hashtable htSession = _sessionManager.getSessionContext(rid);
    		if (htSession == null)
    		{
    		    _systemLogger.log(Level.FINE, MODULE, sMethod, "Session expired or no session found with rid: " + rid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_SESSION_EXPIRED);
    		}
            
    		String uid = (String) additional.get("uid");
    		if (uid == null)
    		{
    			_systemLogger.log(Level.WARNING, MODULE, sMethod, "No uid found in request with rid: " + rid);
    			throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
    		}
    
            String remoteIDP = (String) htSession.get("remote_organization");
    		if (remoteIDP == null)
    		{
    		    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
    		        "No 'remote_organization' found in session with rid: " + rid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
    		}
    
            String requestorURL = (String) htSession.get("app_url");
    		if (requestorURL == null)
    		{
    		    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "No 'app_url' found in session with rid: " + rid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
    		}
    
            String requestor = (String) htSession.get("app_id");
    		if (requestor == null)
    		{
    			_systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "No 'app_id' found in session with rid: " + rid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
    		}
    
    		AttributesMulti methodParams = new AttributesMulti();
    		methodParams.addAttribute("rid", rid);
    		methodParams.addAttribute("uid", uid);
    		if (attributes != null)
    			methodParams.addAttribute("attributes", attributes.toEncodedString());

    		methodParams.addAttribute("requestor", requestor);
    		methodParams.addAttribute("remote_idp", remoteIDP);
    		methodParams.addAttribute("requestor_url", requestorURL);
    
    		//retrieve active url from SAM Agent / Resourcegroup
    		String sTargetUrl = getTargetUrl();
    		
    		method = new GetMethod(sTargetUrl);
    		method.setQueryString(methodParams.toEncodedString());

            _systemLogger.log(Level.FINER, MODULE, sMethod, "Sending " + method.getURI());
			// Actual call
    		try
    		{
    		    _httpClient.executeMethod(method);
    		}
			catch (HttpException e)
	        {
	            _systemLogger.log(Level.WARNING, MODULE, sMethod, 
	                "Could not execute method call: " + method.toString(), e);
	            throw new ASelectException(Errors.ERROR_ASELECT_IO);

	        }

			// Retrieve result
			String resultBody = method.getResponseBodyAsString();
			
			StringBuffer sbInfo = new StringBuffer("Response from ");
			sbInfo.append(sTargetUrl);
			sbInfo.append(" contains message body: ");
			sbInfo.append(resultBody);
			_systemLogger.log(Level.FINER, MODULE, sMethod, sbInfo.toString());

			// Convert received attributes
			AttributesMulti responseParams = new AttributesMulti(resultBody);
			String sResponseAttributes = responseParams.getAttribute("attributes");
			if (sResponseAttributes != null)
			{
			    AttributesMulti responseAttributes = new AttributesMulti(sResponseAttributes);
			    Hashtable htResponseAttributes = responseAttributes.toHashtable();
			    String sSerializedAttributes = serializeAttributes(htResponseAttributes);
			    //overwrite A-Select attributes with supplied attributes
			    if (sSerializedAttributes != null)
			        additional.put("attributes", sSerializedAttributes);
			    else if (additional.containsKey("attributes"))
			        additional.remove("attributes");
			}
			
			String sResponseUID = responseParams.getAttribute("uid");
			if (sResponseUID != null)
			{
			    //overwrite A-Select uid with supplied uid
				additional.put("uid", sResponseUID);
			}
			
			String sAction = responseParams.getAttribute("action");
			if (sAction == null)
			{
				_systemLogger.log(Level.FINE, MODULE, sMethod, "No required 'action' parameter in response");
				throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
			}
			
			boolean bActionProceed = true;
			if (sAction.equals("t"))
			{
				//TERMINATE
			    bActionProceed = false;
				
				//clean up the sessions
			    _sessionManager.remove(rid);
			}
			else if (!sAction.equals("p"))
			{
				_systemLogger.log(Level.FINE, MODULE, sMethod, 
				    "Invalid 'action' parameter in response: " + sAction);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
			}
			
			if (bActionProceed)
            {
                //update session with parameters needed by the request handler during authentication resume
    			htSession.put("uid", uid);
    			htSession.put("cross_attributes", additional);
                _sessionManager.updateSession(rid, htSession);
            }

			String redirectURL = responseParams.getAttribute("redirect_url");
			if (redirectURL != null)
			{
			    bContinueProcess = false;
			    
			    try
			    {
			        new URL(redirectURL);
			    }
			    catch(MalformedURLException e)
			    {
			        _systemLogger.log(Level.FINE, MODULE, sMethod, 
			            "Invalid 'redirect_url' parameter in response: " + redirectURL, e);
	                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
			    }

				performRedirect(response, redirectURL, rid, bActionProceed);
			}
			else if (!bActionProceed)
			{
			    //Log successful authentication
	            _authenticationLogger.log(new Object[] {
	                    "Processor", 
	                    uid,
	                    null, 
	                    remoteIDP,
	                    requestor,
	                    "denied"});
                throw new ASelectException(Errors.ERROR_ASELECT_AUTHSP_ACCESS_DENIED);
			}
		}
		catch (ASelectException e)
		{
		    throw e;
		}
		catch (Exception e)
		{
			_systemLogger.log(Level.SEVERE, MODULE, sMethod,
					"Could not perform call", e);
			throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
		}
		finally
		{
		    if (method != null)
		        method.releaseConnection();
		}

		return bContinueProcess;
	}
	
	
	private void performRedirect(HttpServletResponse response, 
	    String redirectTarget, String rid, boolean bResumeAfterRedirect)
	    throws ASelectException
	{
	    String sMethod = "performRedirect()";
	    StringBuffer sbRedirect = null;
	    try
	    {
	        //create query string parameters
            AttributesMulti redirectParams = new AttributesMulti();
            redirectParams.addAttribute("rid", rid);
            if (bResumeAfterRedirect)
            {
                StringBuffer sbResumeUrl = new StringBuffer(_sResumeUrl);
                sbResumeUrl.append("?rid=");
                sbResumeUrl.append(rid);
                redirectParams.addAttribute("redirect_url", sbResumeUrl.toString());
            }
            
	        //create redirect Url
	        sbRedirect = new StringBuffer(redirectTarget);
	        
	        if (redirectTarget.indexOf('?') == -1)
	            sbRedirect.append("?");
	        else
	            sbRedirect.append("&");
	        
	        sbRedirect.append(redirectParams.toEncodedString());
	        	        
            _systemLogger.log(Level.FINER, MODULE, sMethod,
                    "Performing redirect: " + sbRedirect.toString());
            
            //perform redirect
            response.sendRedirect(sbRedirect.toString());
	    }
	    catch (IOException e)
	    {
	        _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                "Could not send redirect: " + sbRedirect.toString(), e);
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
	    }
	}

	/**
     * Searches in the configuration tree for a corresponding element.
     * 
     * @param configManager
     *            The configuration manager to use.
     * @param name
     *            The destination string. Takes the form of
     *            "&lt;element&gt;[@id]/&lt;element&gt;[@id]/..."
     * @param root
     *            The root element to start searching
     * @return The element if found, otherwise "null"
     */
    private Object getSection(ASelectConfigManager configManager, String name, Object root)
    {
    	Object tmp = null;
    	String restString = null;
    	String firstElement = null;
    	String id = null;
    
    	int pos = name.indexOf("/");
    	if (pos == -1)
    	{
    		// Last element in the list.
    		firstElement = name;
    	}
    	else
    	{
    		firstElement = name.substring(0, pos);
    	}
    
    	// Determine if an ID was given
    	int posID = firstElement.indexOf("@");
    	if (posID > -1)
    	{
    		id = firstElement.substring(posID + 1);
    		firstElement = firstElement.substring(0, posID);
    	}
    
    	if (id == null)
    	{
    		try
    		{
    			tmp = configManager.getSection(root, firstElement);
    		}
    		catch (ASelectConfigException e)
    		{
    			tmp = null;
    		}
    	}
    	else
    	{
    		try
    		{
    			tmp = configManager.getSection(root, firstElement, "id=" + id);
    		}
    		catch (ASelectConfigException e)
    		{
    			tmp = null;
    		}
    	}
    
    	if ((pos == -1) || (tmp == null))
    	{
    		// End of the line
    		return tmp;
    	}
    
    	restString = name.substring(pos + 1);
    	return getSection(configManager, restString, tmp);
    }

    /**
     * Retrieve a parameter from the configuration. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param configManager
     *            The configuration manager.
     * @param section
     *            The section to retrieve from.
     * @param name
     *            The parameter to retrieve.
     * @return The corresponding value.
     */
    private String getParam(ASelectConfigManager configManager, Object section, String name)
    {
    	String result = null;
    	try
    	{
    		result = configManager.getParam(section, name);
    	}
    	catch (ASelectConfigException e)
    	{
    		return null;
    	}
    
    	return result;
    }

    /**
     * Deserialize attributes and conversion to a <code>Hashtable</code>.
     * 
     * @param sSerializedAttributes The serialized attributes.
     * @return The deserialized attributes (key,value in <code>Hashtable</code>)
     * @throws ASelectException If URLDecode fails
     */
    private Hashtable deserializeAttributes(String sSerializedAttributes)
    {
    	String sMethod = "deSerializeAttributes()";
    	Hashtable htAttributes = new Hashtable();
    	if (sSerializedAttributes != null) // Attributes available
    	{
    		try
    		{
    			// base64 decode
    			String sDecodedUserAttrs = new String(Base64
    					.decode(sSerializedAttributes));
    
    			// decode & and = chars
    			String[] saAttrs = sDecodedUserAttrs.split("&");
    			for (int i = 0; i < saAttrs.length; i++)
    			{
    				int iEqualChar = saAttrs[i].indexOf("=");
    				String sKey = "";
    				String sValue = "";
    				Vector vVector = null;
    
    				if (iEqualChar > 0)
    				{
    					sKey = URLDecoder.decode(saAttrs[i].substring(0,
    							iEqualChar), "UTF-8");
    
    					sValue = URLDecoder.decode(saAttrs[i]
    							.substring(iEqualChar + 1), "UTF-8");
    
    					if (sKey.endsWith("[]"))
    					{ // it's a multi-valued attribute
    						// Strip [] from sKey
    						sKey = sKey.substring(0, sKey.length() - 2);
    
    						if ((vVector = (Vector) htAttributes.get(sKey)) == null)
    						{
    							vVector = new Vector();
    						}
    
    						vVector.add(sValue);
    					}
    				}
    				else
    				{
    					sKey = URLDecoder.decode(saAttrs[i], "UTF-8");
    				}
    
    				if (vVector != null)
    				{
    					// store multivalue attribute
    					htAttributes.put(sKey, vVector);
    				}
    				else
    				{
    					// store singlevalue attribute
    					htAttributes.put(sKey, sValue);
    				}
    			}
    		}
    		catch (Exception e)
    		{
    			_systemLogger.log(Level.WARNING, MODULE, sMethod,
    					"Error during deserialization of attributes", e);
    		}
    	}
    	return htAttributes;
    }

    /**
     * Serialize attributes contained in a hashtable. <br>
     * <br>
     * <b>Description:</b> <br>
     * This method serializes attributes contained in a hashtable:
     * <ul>
     * <li>They are formatted as attr1=value1&attr2=value2;...
     * <li>If a "&amp;" or a "=" appears in either the attribute name or value,
     * they are transformed to %26 or %3d respectively.
     * <li>The end result is base64 encoded.
     * </ul>
     * <br>
     * 
     * @param htAttributes
     *            Hashtable containing all attributes
     * @return Serialized representation of the attributes
     * @throws ASelectException
     *             If serialization fails.
     */
    private String serializeAttributes(Hashtable htAttributes)
    		throws ASelectException
    {
    	final String sMethod = "serializeAttributes()";
    	try
    	{
    		if ((htAttributes == null) || htAttributes.isEmpty())
    		{
    			return null;
    		}
    		StringBuffer sb = new StringBuffer();
    		for (Enumeration e = htAttributes.keys(); e.hasMoreElements();)
    		{
    			String sKey = (String) e.nextElement();
    			Object oValue = htAttributes.get(sKey);
    
    			if (oValue instanceof Vector)
    			{// it's a multivalue attribute
    				Vector vValue = (Vector) oValue;
    
    				sKey = URLEncoder.encode(sKey + "[]", "UTF-8");
    				Enumeration eEnum = vValue.elements();
    				while (eEnum.hasMoreElements())
    				{
    					String sValue = (String) eEnum.nextElement();
    
    					// add: key[]=value
    					sb.append(sKey);
    					sb.append("=");
    					sb.append(URLEncoder.encode(sValue, "UTF-8"));
    
    					if (eEnum.hasMoreElements())
    					{
    						sb.append("&");
    					}
    				}
    			}
    			else if (oValue instanceof String)
    			{// it's a single value attribute
    				String sValue = (String) oValue;
    
    				sb.append(URLEncoder.encode(sKey, "UTF-8"));
    				sb.append("=");
    				sb.append(URLEncoder.encode(sValue, "UTF-8"));
    			}
    
    			if (e.hasMoreElements())
    			{
    				sb.append("&");
    			}
    		}
    		BASE64Encoder b64enc = new BASE64Encoder();
    		return b64enc.encode(sb.toString().getBytes("UTF-8"));
    	}
    	catch (Exception e)
    	{
    		_systemLogger.log(Level.WARNING, MODULE, sMethod,
    				"Could not serialize attributes", e);
    		throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
    	}
    }
    
    private String getTargetUrl() throws ASelectException
    {
        String sMethod = "getTargetUrl()";
        try
        {
            if (_sTargetUrl == null || _samActiveResource == null || !_samActiveResource.live())
            {
                _samActiveResource = _samAgent.getActiveResource(_sTargetResourceGroup);
                
                _sTargetUrl = getParam(_configManager, _samActiveResource.getAttributes(), "url");
                if (_sTargetUrl == null)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                        "No 'url' configuration item found in resourcegroup with id: " + _sTargetResourceGroup);
                    throw new ASelectException(Errors.ERROR_ASELECT_CONFIG_ERROR);
                }
                
                try
                {
                    new URL(_sTargetUrl);
                }
                catch(MalformedURLException e)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                        "Configured 'url' item in resourcegroup is not a valid URL: " + _sTargetUrl, e);
                    throw new ASelectException(Errors.ERROR_ASELECT_CONFIG_ERROR);
                }
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not retrieve remote url from resourcegroup: " + 
                _sTargetResourceGroup, e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        return _sTargetUrl;
    }
}
