/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.elo;

import org.aselect.system.configmanager.ConfigManager;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.exception.ASelectStorageException;
import org.aselect.system.logging.SystemLogger;

/**
 * Interface describing a method for loading Elo property objects.
 * <br><br>
 * <b>Description:</b><br>
 * Possible implementations include:
 * <ul>
 * <li>Elo information gathered from the A-Select configuration file;</li>
 * <li>Elo information gathered from a relation database.</li>
 * </ul>
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public interface IELOStorage
{
    /**
     * Initializes the Elo store.
     * <br><br>
     * <b>Description:</b>
     * <br>
     * The config section should fit the elo store's type.
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b>
     * <br> - oConfigSection != null<br>
     * <br> - oConfigManager != null<br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * @param oConfigSection Section from the configuration applicable for the Elo Store.
     * @param oConfigManager The configuration manager.
     * @param systemLogger The system logger.
     * @throws ASelectException If an error occurs while processing the configuration.
     */
    public void init(Object oConfigSection, ConfigManager oConfigManager,
        SystemLogger systemLogger) throws ASelectException;
	
	/**
	 * Gives the Elo store a chance to clean up.
	 * <br><br>
	 * <b>Concurrency issues:</b> <br> - <br>
	 * <br>
	 * <b>Preconditions:</b> <br> - <br>
	 * <br>
	 * <b>Postconditions:</b> <br> - <br>
	 * @throws ASelectException If clean-up fails.
	 */
	public void destroy() throws ASelectException;
	
	/**
	 * Returns an <code>Elo</code> object from the store, using its ID.
	 * <br><br>
	 * <b>Concurrency issues:</b> <br> - <br>
	 * <br>
	 * <b>Preconditions:</b> <br> - <br>
	 * <br>
	 * <b>Postconditions:</b> <br> - <br>
	 * @param sId The id of the Elo.
	 * @return The <code>Elo</code> object attached to the given ID or <code>null</code> if the
	 * Elo with the given ID could not be found.
	 * @throws ASelectStorageException
	 */
	public ELO getEloByID(String sId) throws ASelectStorageException;
	
	/**
     * Returns an <code>Elo</code> object from the store, using the URL.
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * @param sURL The URL of the Elo.
     * @return The <code>Elo</code> object attached to the given URL or <code>null</code> if the
     * Elo with the given URL could not be found.
     * @throws ASelectStorageException
     */
    public ELO getEloByURL(String sURL) throws ASelectStorageException;
}
