/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.elo;

import java.util.logging.Level;

import org.aselect.server.config.ASelectConfigManager;
import org.aselect.server.log.ASelectSystemLogger;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.logging.SystemLogger;

/**
 * The ELO factory is responsible for the initialization of the ELO property objects and their
 * storage facility.
 * <br><br>
 * <b>Description:</b><br>
 * At initialization, the Elo factory retrieves the configuration manager and retrieves its
 * own elo configuration. The configuration data is used to initialize the storage construction.
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public class ELOFactory
{
    private static final String MODULE = "EloFactory";
    private static ELOFactory _oInstance = null;
    
    private SystemLogger _oLogger = null;
    private IELOStorage _oStorage = null;
    
    /**
     * Returns the singleton instance.
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * @return the instance.
     * @throws ASelectException if the instance cannot be initialized.
     */
    public static ELOFactory getHandle() throws ASelectException
    {
        if (_oInstance == null) _oInstance = new ELOFactory();
        return _oInstance;
    }
    
    /**
     * returns the initialized ELO store.
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * @return the initialized ELO store.
     */
    public IELOStorage getEloStore()
    {
        return _oStorage;
    }

    private ELOFactory() throws ASelectException
    {
        
        String sMethod = "ELOFactory()";
        ASelectConfigManager oConfigManager = ASelectConfigManager.getHandle();
        _oLogger = ASelectSystemLogger.getHandle();

        Object oEloStore = null;
        try
        {
            oEloStore = oConfigManager.getSection(null, "elostore");
        }
        catch (ASelectConfigException e)
        {
            _oLogger.log(Level.WARNING, MODULE, sMethod, "No config section 'elostore' found");
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
        }
        
        String sClass = null;
        try
        {
            sClass = oConfigManager.getParam(oEloStore, "class");
        }
        catch (ASelectConfigException e)
        {
            _oLogger.log(Level.WARNING, MODULE, sMethod, "No config item 'class' in section 'elostore' found");
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
        }
        
        Class cEloStore = null;
        try
        {
            cEloStore = Class.forName(sClass);
            _oStorage = (IELOStorage)cEloStore.newInstance();
            _oStorage.init(oEloStore, oConfigManager, _oLogger);
        }
        catch (ClassNotFoundException cnfe)
        {
            _oLogger.log(Level.WARNING, MODULE, sMethod, "'IEloStore' class not found: " + sClass, cnfe);
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR, cnfe);
        }
        catch (ASelectException ase)
        {
            _oLogger.log(Level.WARNING, MODULE, sMethod, "Could not initialize 'IEloStore' class: " + sClass, ase);
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR, ase);
        }
        catch (ClassCastException cce)
        {
            _oLogger.log(Level.WARNING, MODULE, sMethod, "Not a correct 'IEloStore' class (wrong type): " + sClass, cce);
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR, cce);
        }
        catch (Exception e)
        {
            _oLogger.log(Level.WARNING, MODULE, sMethod, "'IEloStore' class could not be loaded: " + sClass, e);
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR, e);
        }
    }
}
