#! /usr/bin/perl

# $Id: xyzzy.t 39147 2013-03-27 14:00:11Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/t/xyzzy.t $

use strict;
use warnings FATAL => 'all';

package Dummy::LDAP::Result;

use Clarity -base;

sub dn { return 'cn=bob' }
sub get_value { return 'bob' }

package Net::LDAP;

use Clarity -base;

sub bind {
	my ($self, $uid, undef, $pwd) = @_;
	die "bad password. go sit in corner.\n" if defined $uid && $pwd ne 'hunter2';
	return;
}
sub search { return new Clarity(entries => new Dummy::LDAP::Result, count => 1) }

sub AUTOLOAD { warn our $AUTOLOAD }

package main;

use Test::More;
use Test::Xyzzy qw(xyzzy_from_hash xyzzy_request xyzzy_request_as_cgi);
use XML::LibXML;
use Aselect::Util qw(aselect_split);
use Data::Dumper;

my $app = xyzzy_from_hash(
	Application => 'Aselect',
	AselectSecret => 'secret',
	AselectServerID => 'sso.example.com',
	StylesheetDir => 'etc/stylesheets',
	LDAPBase => 'foo',
	AuthFailTimeout => 0,
	AselectRequestor => 'frobber',
);

my $handler = $app->handler;

isa_ok($handler, 'Xyzzy::Handler');

sub parse_xml {
	my $parser = new XML::LibXML;
	$parser->validation(0);
	$parser->load_ext_dtd(0);
	$parser->expand_entities(0);
	$parser->pedantic_parser(1);
	return $parser->parse_string(shift);
}

sub bake_cookies {
	my %cookies = @_;
	my @cookies;
	while(my ($name, $value) = each %cookies) {
		push @cookies, "$name=$value";
	}
	return join('; ', @cookies);
}

sub parse_cookies {
	return map { /^Set-Cookie:\s+([^;=]+)=([^;]+)/ ? ($1, $2) : () } @_;
}

sub parse_cgi {
	my ($headers, $body) = split(/\015\012\015\012/, shift, 2);
	my @headers = split(/\015\012/, $headers);
	return $body, @headers;
}

do {
	my $res = xyzzy_request($handler, '/status');
	like($res, qr{Status overview}, "output contains stuff from the templates");
};

do {
	my $res = xyzzy_request_as_cgi($handler, '/login', 'raw');
	my ($body, @headers) = parse_cgi($res);
	my %cookies = parse_cookies(@headers);
	my $xml = parse_xml($body);
	my $lt = $xml->findvalue('/password/lt');

	foreach my $remember ('true', 'false') {
		foreach my $setting (undef, 'true', 'false') {
			foreach my $service (undef, 'http://example.com') {
				local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /bad password/ };
				my %c = %cookies;
				$c{remember} = $setting if defined $setting;
				my $param = $remember eq 'true' ? '&remember=$remember' : '';
				$param .= "&service=$service" if defined $service;
				my $effective_remember = defined $service ? $remember : 'true';
				my $effective_setting = $setting // 'true';
				my $effective_service = $service // 'http://localhost/status';
				$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=*******&lt=$lt$param", HTTP_COOKIE => bake_cookies(%c));
				($body, @headers) = parse_cgi($res);
				$xml = parse_xml($body);
				is($xml->findvalue('/*/login/remember'), $effective_remember, "remember is $effective_remember");
				is($xml->findvalue('/*/settings/remember'), $effective_setting, "remember setting is $effective_setting");
				ok(!(grep { m{^Location: \Q$effective_service} } @headers), "headers do not contain location to $effective_service");
			}
		}
	}

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt", HTTP_COOKIE => bake_cookies(%cookies, warn => 'true'));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { $_ eq 'Location: http://localhost/status' } @headers), "headers contain location to status url");
	is($xml->findvalue('/login-success/login/url'), 'http://localhost/status', "result contains status url");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&gateway=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { $_ eq 'Location: http://example.com/foo' } @headers), "headers contain location to service url");
	is($xml->findvalue('/giveup/login/url'), 'http://example.com/foo', "result contains service url");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt&service=http://example.com", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com} } @headers), "headers contain location to service url");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com', "result contains service url");

	my %noremember_cookies = (%cookies, parse_cookies(@headers));

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%noremember_cookies));
	($body, @headers) = parse_cgi($res);
	ok(!(grep { m{^Location: http://example.com/foo} } @headers), "headers do not contain location to status url after no-remember login");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { $_ eq 'Location: http://localhost/status' } @headers), "headers contain location to status url");
	is($xml->findvalue('/login-success/login/url'), 'http://localhost/status', "result contains status url");

	%cookies = (%cookies, parse_cookies(@headers));

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	my $ticket = '';
	foreach(@headers) {
		if(m{^Location: http://example.com/foo\?ticket=(ST-[^&]+)}) {
			$ticket = $1;
			last;
		}
	}

	$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket");
	($body, @headers) = parse_cgi($res);
	is($body, "yes\nbob\n", "result contains uid");

	do {
		local $SIG{__WARN__} = sub { local $_ = shift; die $_ unless /CAS validation request requested checking/ };
		$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket&renew=true");
		($body, @headers) = parse_cgi($res);
		is($body, "no\n\n", "result does not contain uid");
	};

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&gateway=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&renew=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	is($xml->findvalue('/password/session/uid'), 'bob', "result is a password page");
	$lt = $xml->findvalue('/password/lt');

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&username=bob&password=hunter2&lt=$lt&service=http://example.com/foo&renew=true", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	foreach(@headers) {
		if(m{^Location: http://example.com/foo\?ticket=(ST-[^&]+)}) {
			$ticket = $1;
			last;
		}
	}

	$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket");
	($body, @headers) = parse_cgi($res);
	is($body, "yes\nbob\n", "result contains uid");

	$res = xyzzy_request_as_cgi($handler, '/validate', "service=http://example.com/foo&ticket=$ticket&renew=true");
	($body, @headers) = parse_cgi($res);
	is($body, "yes\nbob\n", "result contains uid");

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies, warn => 'true'));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok(!(grep { m{^Location:} } @headers), "headers do not contain redirect");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	my $nonce = $xml->findvalue('/login-success/nonce');

	$res = xyzzy_request_as_cgi($handler, '/login', "raw&service=http://example.com/foo&nonce=$nonce", HTTP_COOKIE => bake_cookies(%cookies, warn => 'true'));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?ticket=ST-} } @headers), "headers contain location to service url with ticket");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");

	$res = xyzzy_request_as_cgi($handler, '/server', "request=authenticate&app_id=frobber&app_url=http://example.com/foo", HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	my %res = aselect_split($body);

	is($res{result_code}, '0000', "request=authenticate returned code 0000");
	my $as_url = $res{as_url};
	like($as_url, qr{^http://localhost/login}, "request=authenticate returned an url");
	$as_url =~ s{^https?://[^/]+}{};

	my $args = "raw&rid=$res{rid}&a-select-server=$res{'a-select-server'}";
	$res = xyzzy_request_as_cgi($handler, $as_url, $args, HTTP_COOKIE => bake_cookies(%cookies));
	($body, @headers) = parse_cgi($res);
	$xml = parse_xml($body);
	ok((grep { m{^Location: http://example.com/foo\?aselect_credentials=} } @headers), "headers contain location to service url");
	is($xml->findvalue('/login-success/login/url'), 'http://example.com/foo', "result contains service url");
};

done_testing();
