#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;

BEGIN {
	use_ok('Aselect::URL', qw(valid_http_url normalize_url acl_to_re_url))
		or BAIL_OUT('need Aselect::URL to run');
}

foreach my $t (
	['http://EXAMPLE.COM', 'http://example.com'],
	['http://EXAMPLE.COM?', 'http://example.com?'],
	['http://EXAMPLE.COM#', 'http://example.com#'],
	['http://EXAMPLE.COM:1234', 'http://example.com:1234'],
	['http://EXAMPLE.COM:1234?', 'http://example.com:1234?'],
	['http://EXAMPLE.COM:1234#', 'http://example.com:1234#'],
	['http://EXAMPLE#', undef],
	['http://foo@EXAMPLE.COM/', undef],
	['http://foo:bar@EXAMPLE.COM/', undef],
	['http://EXAMPLE.COM/', 'http://example.com/'],
	['http://EXAMPLE.COM/MOO', 'http://example.com/MOO'],
	['http://EXAMPLE.COM/M O', 'http://example.com/M%20O'],
) {
	my ($orig, $norm) = @$t;
	if(defined $norm) {
		ok(valid_http_url($orig), "accept url");
		is(normalize_url($orig), $norm, "normalize url");
	} else {
		ok(!valid_http_url($orig), "reject url");
	}
}

foreach my $t (
	['http://example.com', 1, 0, 1, 0],
	['http://example.com:1234', 0, 1, 0, 1],
	['http://an.example.com', 0, 0, 1, 0],
	['http://an.example.com:1234', 0, 0, 0, 1],

	['http://example.com?foo=bar', 1, 0, 1, 0],
	['http://example.com:1234?foo=bar', 0, 1, 0, 1],
	['http://an.example.com?foo=bar', 0, 0, 1, 0],
	['http://an.example.com:1234?foo=bar', 0, 0, 0, 1],

	['http://example.com/path?foo=bar', 1, 0, 1, 0],
	['http://example.com:1234/path?foo=bar', 0, 1, 0, 1],
	['http://an.example.com/path?foo=bar', 0, 0, 1, 0],
	['http://an.example.com:1234/path?foo=bar', 0, 0, 0, 1],

	['http://example.com/path?foo=bar#frag?/', 1, 0, 1, 0],
	['http://example.com:1234/path?foo=bar#frag?/', 0, 1, 0, 1],
	['http://an.example.com/path?foo=bar#frag?/', 0, 0, 1, 0],
	['http://an.example.com:1234/path?foo=bar#frag?/', 0, 0, 0, 1],

	['http://example.com#frag?/', 1, 0, 1, 0],
	['http://example.com:1234#frag?/', 0, 1, 0, 1],
	['http://an.example.com#frag?/', 0, 0, 1, 0],
	['http://an.example.com:1234#frag?/', 0, 0, 0, 1],

	['http://.:example.com#example.com', 0, 0, 0, 0],
	['http://example.com@example.org/example.com#example.com', 0, 0, 0, 0],
	['http://.example.com@example.org/example.com#example.com', 0, 0, 0, 0],
	['http://example.com@foo.example.org/example.com#example.com', 0, 0, 0, 0],
	['http://.example.com@foo.example.org/example.com#example.com', 0, 0, 0, 0],
	['http://example.org', 0, 0, 0, 0],
	['example.org', 0, 0, 0, 0],
	['.example.com', 0, 0, 0, 0],
	['http://example.org', 0, 0, 0, 0],
	['http://.example.com', 0, 0, 0, 0],
	['http://_.example.com', 0, 0, 0, 0],

	['http://eXaMpLe.cOm', 1, 0, 1, 0],
	['http://eXaMpLe.cOm:1234', 0, 1, 0, 1],
	['http://aN.eXaMpLe.cOm', 0, 0, 1, 0],
	['http://aN.eXaMpLe.cOm:1234', 0, 0, 0, 1],
) {
	my ($url, $n, $p, $nd, $pd) = @$t;

	my $tn = $url =~ acl_to_re_url('example.com');
	ok($n ? $tn : !$tn, "match $url against example.com, expected: $n");

	my $tp = $url =~ acl_to_re_url('example.com:1234');
	ok($p ? $tp : !$tp, "match $url against example.com:1234, expected: $p");

	my $tnd = $url =~ acl_to_re_url('.example.com');
	ok($nd ? $tnd : !$tnd, "match $url against .example.com, expected: $nd");

	my $tpd = $url =~ acl_to_re_url('.example.com:1234');
	ok($pd ? $tpd : !$tpd, "match $url against .example.com:1234, expected: $pd");
}

done_testing();
