# $Id: RequestAuthentication.pm 34594 2011-04-14 07:49:33Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/RequestAuthentication.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::RequestAuthentication::Request;

use Aselect::Responder;

use Aselect::Responder::Request -self;

param signature => sub { die "0030 Required request parameter 'signature' not supplied\n" unless defined };

param app_id;
param app_url => sub {
	die "0030 Required parameter 'app_url' not supplied\n" unless defined;
	die "0030 Required parameter 'app_url' invalid\n" unless m{^https?://[!-~]+$};
};

param local_organization;
param local_as_url => sub {
	die "0030 Required parameter 'local_as_url' not supplied\n" unless defined;
	die "0030 Required parameter 'local_as_url' invalid\n" unless m{^https?://[!-~]+$};
};

const requestor_origin => sub { shift->rid; return };
const requestor_id => sub { shift->rid; return };
const requestor_url => sub { shift->rid; return };

const rid => sub {
	my $self = shift;

	my ($origin, $id, $url);

	if(my $app_id = $self->app_id) {
		$origin = 'l';
		$id = $app_id;
		$url = $self->app_url;
	} elsif(my $local_organization = $self->local_organization) {
		$origin = 'r';
		$id = $local_organization;
		$url = $self->local_as_url;
	} else {
		die "0030 No 'app_id' or 'local_organization' parameter supplied\n";
	}

	$self->{requestor_origin} = $origin;
	$self->{requestor_id} = $id;
	$self->{requestor_url} = $url;

	return scalar $self->crypto->create_token('r', $origin, $id, $url);
};

sub response {
	$self->check_signature(@{self->signature_params});

	my $as_url = $self->cfg->login_url // $self->script_url . '/login';

	return {result_code => '0000', rid => $self->rid, as_url => $as_url};
}

sub signature_params {
	my $origin = $self->requestor_origin or confess 'internal error';
	return $origin eq 'l' ? [qw(a-select-server app_id app_url country forced_logon language remote_organization uid)]
		: $origin eq 'r' ? [qw(a-select-server arp_target country forced_logon language
			local_as_url local_organization remote_organization required_level uid)]
		: confess 'internal error';
}

package Aselect::RequestAuthentication;

use Aselect::Handler -self;

sub handle {
	my $req = new Aselect::RequestAuthentication::Request(cfg => $self, ctx => shift);
	my $doc = new Aselect::Responder::Document(req => $req);
	return $doc->response;
}
