# $Id: Login.pm 34881 2011-05-23 18:31:11Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/Login.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::Login::Request;

use Aselect::Util;
use Aselect::Request -self;

field request_timeout => sub { shift->cfg->request_timeout };
field org => sub { shift->cfg->organization };
sub uid { undef }

param rid => sub {
	my $self = shift;
	if(defined) {
		my $crypto = $self->crypto;
		my (undef, undef, $origin, $id, $url) =
			eval { $crypto->check_token('r', $_, $self->cfg->request_timeout) };
		die $self->error('rid', $@) if $@;
		$self->requestor_origin($origin);
		$self->requestor_id($id);
		$self->requestor_url($url);
	} else {
		$self->requestor_origin(undef);
		$self->requestor_id(undef);
		$self->requestor_url(undef);
	}
};

param service => sub {
	die shift->error('service')
		if defined && !m{^https?://[!-~]+$};
};

param nonce => sub {
	my $self = shift;
	my @nonce = eval { $self->crypto->check_token('n'.$self->session, $_, $self->cfg->request_timeout) };
	$_ = @nonce > 0;
};

field nag => sub { defined shift->param('warn') };

field requestor_origin => sub { shift->rid; return };
field requestor_id => sub { shift->rid; return };
field requestor_url => sub { shift->rid; return };

field session => sub { shift->sessioncookie; return };

field sessioncookie => sub {
	my $self = shift;
	my $uid = $self->uid or return;
	my @nag = 1 if $self->nag;
	my (undef, $salt, $token) = $self->crypto->create_token('s', $uid, @nag);
	$self->session($salt);
	return $token;
};

field credentials => sub {
	my $self = shift;
	my $requestor_id = $self->requestor_id or return;
	my $uid = $self->uid or return;
	return scalar $self->crypto->create_token('c', $requestor_id, $uid);
};

field ticket => sub {
	my $self = shift;
	my $service = $self->service or return;
	my $uid = $self->uid or return;
	return scalar $self->crypto->create_cas_token('ST', $service, $uid);
};

field url => sub {
	my $self = shift;
	if(my $rid = $self->rid) {
		return aselect_url($self->requestor_url, {
			rid => $rid,
			'a-select-server' => $self->server_id,
			aselect_credentials => $self->credentials,
		});
	} elsif(my $service = $self->service) {
		return aselect_url($service, {ticket => $self->ticket});
	} else {
		return $self->script_url . '/';
	}
};

field bare_url => sub {
	my $self = shift;
	if(my $requestor_url = $self->requestor_url) {
		return $requestor_url;
	} elsif(my $service = $self->service) {
		return $service;
	} else {
		return $self->script_url . '/';
	}
};

package Aselect::Login::Success;

use Aselect::Document -self;

sub build {
	my $req = $self->req;

	my $root = $self->construct('login-success' =>
		[url => $req->bare_url],
		[nonce => scalar $req->crypto->create_token('n'.$req->session)]
	);
	$self->setDocumentElement($root);

	if(my $rid = $req->rid) {
		$root->appendTextChild(rid => $rid);
	}

	if(my $service = $req->service) {
		$root->appendTextChild(service => $service);
	}

	if(my $nag = $req->nag) {
		$root->appendTextChild(warn => 'true');
	}
}

sub response {
	my $res = super;

	my $req = $self->req;

	if(!$req->nag || $req->nonce) {
		$res->status(302);
		$res->setheader(Location => $req->url);
	}

	if(my $session = $req->sessioncookie) {
		my $cfg = $self->cfg;
		my $cookie = $self->bakecookie(session => $session);
		if(my $domain = $cfg->cookiedomain) {
			$cookie->setparameter(Domain => $domain);
		}
		$res->setcookie($cookie);
	}
	return $res;
}

package Aselect::Login;

use Aselect::Handler -self;
use Xyzzy::First -mixin;
