# $Id: SPNEGO.pm 34425 2011-03-30 10:05:36Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/SPNEGO.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::SPNEGO::Config;

use Xyzzy -self;

field spnego_useragent => undef;

sub set_spnegouseragent {
	my $re = shift;
	$self->spnego_useragent(qr/$re/o);
}

package Aselect::SPNEGO::Authenticate;

use Aselect::Document -self;

sub build {
	$self->setDocumentElement($self->construct('spnego'));
}

sub response {
	my $res = super;
	$res->status(401);
	$res->addheader('WWW-Authenticate' => 'Negotiate');
	my $cookie = $self->bakecookie(SPNEGO => 'fail');
	$cookie->expire('1y');
	$res->setcookie($cookie);
	return $res;
}

package Aselect::SPNEGO::Account;

use Aselect::Document -self;

sub build {
	$self->setDocumentElement($self->construct('spnego-account',
		[uid => $self->req->spnego_uid]
	));
}

sub response {
	my $res = super;
	my $cookie = $self->bakecookie(SPNEGO => 'func');
	$cookie->expire('1y');
	$res->setcookie($cookie);
	return $res;
}

package Aselect::SPNEGO::Success;

use Aselect::Login;
use Aselect::Login::Success -self;

sub response {
	my $res = super;
	my $cookie = $self->bakecookie(SPNEGO => 'ok');
	$cookie->expire('1y');
	$res->setcookie($cookie);
	return $res;
}

package Aselect::SPNEGO::Request;

use MIME::Base64;

use Aselect::Login::Request -self;

const authorization => sub {
	return shift->http('HTTP_AUTHORIZATION');
};

const negotiate => sub {
	my $auth = shift->authorization;
	return undef unless defined $auth;
	unless($auth =~ s/^Negotiate //) {
		warn "wrong Authorization: header found\n";
		return undef;
	}
	return $auth;
};

const feasible => sub {
	my $self = shift;

	return 1 if $self->authorization;
	return 1 if $self->param('domain');

	if(my $cookie = $self->cookie('SPNEGO')) {
		return $cookie eq 'ok';
	}

	if(my $re = $self->cfg->spnego_useragent and my $ua = $self->user_agent) {
		return $ua =~ $re;
	}

	return undef;
};

field spnego_uid => sub { shift->uid; return };

const uid => sub {
	my $self = shift;
	my $cfg = $self->cfg;

	my $negotiate = $self->negotiate or return undef;

	my $uid = eval {
		my $binary = decode_base64($negotiate);

		my $asn1 = $cfg->asn1;
		my $decoded = $asn1->decode($binary)
			or die $asn1->error;

		my $token = $decoded->{negToken}{negTokenInit}{mechToken};

		die "no mechToken found\n"
			unless defined $token;

		$cfg->gssapi->authenticate($token);
	};
	warn $@ if $@;

	$self->spnego_uid($uid);

	die Aselect::SPNEGO::Account->new(req => $self)->response
		unless $cfg->dir->search($uid)->count == 1;

	return $uid;
};

package Aselect::SPNEGO;

use Aselect::ASN1;

use Aselect::Handler -self;

sub handle {
	my $req = new Aselect::SPNEGO::Request(cfg => $self, ctx => shift);
	return undef unless $req->feasible;
	my $doc = $req->uid
		? new Aselect::SPNEGO::Success(req => $req)
		: new Aselect::SPNEGO::Authenticate(req => $req);
	return $doc->response;
}

const asn1 => sub {
	my $asn1 = new Aselect::ASN1;
	$asn1->configure(encoding => 'DER');
	$asn1->prepare(<<'EOT') or die $asn1->error;
		SPNEGO ::= [APPLICATION 0] SEQUENCE {
			spnego       MechType,
			negToken     NegotiationToken
		}

		NegotiationToken ::= CHOICE {
			negTokenInit    [0] EXPLICIT NegTokenInit,
			negTokenResp    [1] EXPLICIT NegTokenResp
		}

		MechType ::= OBJECT IDENTIFIER

		MechTypeList ::= SEQUENCE OF MechType

		NegTokenResp ::= SEQUENCE {
			negState       [0] EXPLICIT ENUMERATED,
			supportedMech  [1] EXPLICIT MechType      OPTIONAL,
			responseToken  [2] EXPLICIT OCTET STRING  OPTIONAL,
			mechListMIC    [3] EXPLICIT OCTET STRING  OPTIONAL
		}

		ContextFlags ::= BIT_STRING

		NegTokenInit ::= SEQUENCE {
			mechTypes      [0] EXPLICIT MechTypeList  OPTIONAL,
			reqFlags       [1] EXPLICIT ContextFlags  OPTIONAL,
			mechToken      [2] EXPLICIT OCTET STRING  OPTIONAL,
			mechListMIC    [3] EXPLICIT OCTET STRING  OPTIONAL
		}
EOT

	return $asn1->find('SPNEGO');
};
