# $Id: SPNEGO.pm 34104 2011-03-02 16:44:09Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/SPNEGO.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::SPNEGO::Authenticate;

use Aselect::Document -self;

sub build {
	$self->setDocumentElement($self->construct('spnego'));
}

sub response {
	my $res = super;
	$res->status(401);
	my $req = $self->req;
	$res->addheader('WWW-Authenticate' => 'Negotiate');
	my $cookie = $self->bakecookie(SPNEGO => 'fail');
	$cookie->expire('1y');
	$res->setcookie($cookie);
	return $res;
}

package Aselect::SPNEGO::Success;

use Aselect::Login::Success -self;

sub response {
	my $res = super;
	my $req = $self->req;
	my $cookie = $self->bakecookie(SPNEGO => 'ok');
	$cookie->expire('1y');
	$res->setcookie($cookie);
	return $res;
}

package Aselect::SPNEGO::Request;

use MIME::Base64;

use Aselect::Login::Request -self;

const authorization => sub { shift->http('HTTP_AUTHORIZATION') };

const feasible => sub {
	my $self = shift;

	return 1 if $self->authorization;
	return 1 if $self->param('domain');

	if(my $cookie = $self->cookie('SPNEGO')) {
		return $cookie eq 'ok';
	}

	if(my $ua = $self->user_agent) {
		return index($ua, 'UvT-Campus') != -1;
	}

	return undef;
};

const uid => sub {
	my $self = shift;

	my $auth = $self->authorization or return undef;
	unless($auth =~ s/^Negotiate //) {
		warn "wrong Authorization: header found\n";
		return undef;
	}

	my $binary = decode_base64($auth);

	my $cfg = $self->cfg;
	my $asn1 = $cfg->asn1;
	my $decoded = $asn1->decode($binary)
		or die $asn1->error;

	my $cred = $decoded->{negToken}{0}{negTokenInit}{mechToken};

	return scalar $cfg->gssapi->authenticate($cred);
};


package Aselect::SPNEGO;

use Aselect::ASN1;

use Aselect::Handler -self;

sub handle {
	my $req = new Aselect::SPNEGO::Request(cfg => $self, ctx => shift);
	return undef unless $req->feasible;
	my $doc = $req->uid
		? new Aselect::SPNEGO::Success(req => $req)
		: new Aselect::SPNEGO::Authenticate(req => $req);
	return $doc->response;
}

const asn1 => sub {
	my $asn1 = new Aselect::ASN1;
	$asn1->configure(encoding => 'DER');
	$asn1->prepare(<<'EOT') or die $asn1->error;
		SPNEGO ::= [APPLICATION 0] SEQUENCE {
			spnego       MechType,
			negToken     NegotiationToken
		}

		NegotiationToken ::= CHOICE {
			negTokenInit    [0] EXPLICIT NegTokenInit,
			negTokenResp    [1] EXPLICIT NegTokenResp
		}

		MechType ::= OBJECT IDENTIFIER

		MechTypeList ::= SEQUENCE OF MechType

		NegTokenResp ::= SEQUENCE {
		   negState       [0] EXPLICIT ENUMERATED,
		   supportedMech  [1] EXPLICIT MechType      OPTIONAL,
		   responseToken  [2] EXPLICIT OCTET STRING  OPTIONAL,
		   mechListMIC    [3] EXPLICIT OCTET STRING  OPTIONAL
		}

		ContextFlags ::= BIT_STRING

		NegTokenInit ::= SEQUENCE {
			mechTypes      [0] EXPLICIT MechTypeList  OPTIONAL,
			reqFlags       [1] EXPLICIT ContextFlags  OPTIONAL,
			mechToken      [2] EXPLICIT OCTET STRING  OPTIONAL,
			mechListMIC    [3] EXPLICIT OCTET STRING  OPTIONAL
		}
EOT

	return $asn1->find('SPNEGO');
};
